package ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.direct.libs.keywordutils.parser.KeywordParser;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator.keywordSyntaxValidator;
import static ru.yandex.direct.testing.matchers.validation.Matchers.hasNoDefectsDefinitions;

/**
 * Валидатор синтаксиса проверяет синтаксис ключевой фразы с учетом того,
 * что в ключевой фразе должны быть только единичные минус-слова
 * с учетом разделения слов при парсинге.
 */
@RunWith(Parameterized.class)
public class PhraseSyntaxValidatorPositiveTest {

    @Parameterized.Parameters(name = "{0}")
    public static Collection<Object[]> parameters() {
        return Arrays.asList(new Object[][]{

                // 1 плюс-слово с различными операторами
                {"слон"},
                {" слон "},
                {"!слон"},
                {" !слон "},
                {"+слон"},
                {" +слон "},

                {"[слон]"},
                {" [ слон ] "},
                {"[!слон]"},
                {" [ !слон ] "},

                {"\"слон\""},
                {"\" слон \""},
                {"\"!слон\""},
                {"\" !слон \""},
                {"\"+слон\""},
                {"\" +слон \""},

                {"\"[слон]\""},
                {"\" [ слон ] \""},
                {"\"[!слон]\""},
                {"\" [ !слон ] \""},

                // 1 плюс-слово с дефисом с различными операторами
                {"санкт-петебург"},
                {"!санкт-петебург"},
                {"+санкт-петебург"},

                {"[санкт-петебург]"},
                {"[!санкт-петебург]"},

                {"\"санкт-петебург\""},
                {"\"!санкт-петебург\""},
                {"\"+санкт-петебург\""},

                {"\"[санкт-петебург]\""},
                {"\"[!санкт-петебург]\""},

                // 2 плюс-слова с различными операторами
                {"розовый слон"},
                {"розовый !слон"},
                {"!розовый слон"},
                {"!розовый !слон"},
                {"розовый +слон"},
                {"+розовый слон"},
                {"+розовый +слон"},

                {"розовый [слон]"},
                {"[розовый] слон"},
                {"[розовый] [слон]"},
                {"[розовый слон]"},

                {"розовый [!слон]"},
                {"[!розовый] слон"},

                {"+розовый [слон]"},

                {"\"розовый слон\""},
                {"\"розовый !слон\""},
                {"\"!розовый слон\""},
                {"\"!розовый !слон\""},
                {"\"розовый +слон\""},
                {"\"+розовый слон\""},
                {"\"+розовый +слон\""},

                {"\"розовый [слон]\""},
                {"\"[розовый] слон\""},
                {"\"[розовый] [слон]\""},
                {"\"[розовый слон]\""},

                {"\"розовый [!слон]\""},
                {"\"!розовый [!слон]\""},
                {"\"[!розовый] слон\""},

                // плюс-слово с минус-словом с различными операторами
                {"слон -купить"},
                {" слон -купить "},
                {"слон -!купить"},
                {"слон -+купить"},

                // плюс-слово и минус-словом с различными операторами
                {"!слон -купить"},
                {"!слон -!купить"},
                {"!слон -+купить"},
                {"+слон -купить"},
                {"+слон -!купить"},
                {"+слон -+купить"},

                {"[слон] -купить"},
                {"[слон] -!купить"},
                {"[слон] -+купить"},
                {"[!слон] -купить"},
                {"[!слон] -!купить"},
                {"[!слон] -+купить"},

                // плюс-слово с 2 минус-словами с различными операторами
                {"слон -купить -продать"},
                {"слон -!купить -продать"},
                {"слон -купить -!продать"},
                {"слон -!купить -!продать"},
                {"слон -+купить -продать"},
                {"слон -купить -+продать"},
                {"слон -+купить -+продать"},

                // разрешено более одного пробела между словами
                {"  слон [  купить  коня  ]  -+купить  -+продать"},
                {"\"  слон [  купить  коня  ]  \""},

                // использование точки в плюс-фразе и минус-словах
                {"слон."},
                {" слон. "},
                {"конь слон. купить"},
                {"[слон.]"},
                {"[ слон. ]"},
                {"\"слон.\""},
                {"\" слон. \""},
                {"\"[слон.]\""},
                {"\" [ слон. ] \""},
                {"слон.ру"},
                {" слон.ру "},
                {"123."},
                {"123.45"},
                {"123.45 123.45"},
                {"фраза -123.45"},
                {"фраза -слово -123.45"},

                // сложные фразы
                {"слон летающий -купить -продать"},
                {"слон [летающий как белка] -купить -продать"},
                {"слон [!летающий как белка] -купить -продать"},
                {"!слон [!летающий как белка] но [не крокодил] -купить -продать"},
                {"!слон [летающий как !белка] +но [не крокодил] -купить -продать"},
                {"!слон [летающий !как белка] [не крокодил] +но -купить -продать"},
                {"[!летающий как белка] !слон [не крокодил] +но -купить -продать"},
                {"[!летающий как белка] !слон [не крокодил] +но -!купить -+продать"},
                {"[!летающий как белка] !слон [санкт-петербург] +но -!купить -+продать"},
                {"[!летающий как белка] !слон [санкт-петербург] +но -!купить -+продать"},
                {"\"слон летающий\""},
                {"\"слон [летающий как белка]\""},
                {"\"слон [!летающий как белка]\""},
                {"\"!слон [!летающий как белка] но [не крокодил]\""},
                {"\"!слон [летающий как !белка] +но [не крокодил]\""},
                {"\"!слон [летающий !как белка] [не крокодил] +но\""},
                {"\"[!летающий как белка] !слон [не крокодил] +но\""},
        });
    }

    @Parameterized.Parameter
    public String keyword;

    @Test
    public void testValidator() {
        ValidationResult<String, Defect> validationResult =
                keywordSyntaxValidator().apply(keyword);
        assertThat(validationResult, hasNoDefectsDefinitions());
    }

    /**
     * Все положительные кейсы должны успешно парситься
     */
    @Test
    public void testParsing() {
        try {
            KeywordParser.parseWithMinuses(keyword);
        } catch (Exception e) {
            fail(e.toString());
        }
    }
}
