package ru.yandex.direct.core.entity.uac.service.trackingurl

import junitparams.JUnitParamsRunner
import junitparams.Parameters
import junitparams.naming.TestCaseName
import org.assertj.core.api.Assertions.assertThat
import org.junit.Test
import org.junit.runner.RunWith
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.common.spring.AbstractSpringTest
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.testing.configuration.CoreTest

@CoreTest
@RunWith(JUnitParamsRunner::class)
@Suppress("HttpUrlsUsage")
class TrackingUrlParseServiceTest : AbstractSpringTest() {

    @Autowired
    private lateinit var trackingUrlParseService: TrackingUrlParseService

    fun provideTrackingUrls() = arrayOf(
        arrayOf(
            "https://google.com",
            null,
            null
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id",
            null,
            "https://control.kochava.com/v1/cpi/click?click_id={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&network_id=221&adid=&click_id={logid}&device_id=&device_id_type=&site_id=p277-s322-cf1a3fda0&append_app_conv_trk_params=1&creative_id=Call%20Home%20Span%20480x320\n ",
            null,
            null
        ),
        arrayOf(
            "https://redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            null,
            "https://redirect.appmetrica.yandex.com/serve/123/?click_id={logid}"
        ),
        arrayOf(
            "https://1.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            null,
            "https://1.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}"
        ),
        arrayOf(
            "https://12468.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            null,
            "https://12468.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}"
        ),
        arrayOf(
            "http://app.appsflyer.com/com.greatapp?pid=chartboost_int&c=christmas_sale&af_adset_id=54822",
            null,
            "http://app.appsflyer.com/com.greatapp?c=christmas_sale&af_adset_id=54822&pid=yandexdirect_int&clickid={logid}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://app.appsflyer.com/com.im30.ROE.gp?pid=yandexdirect_int&af_c_id={campaign_id}_{campaign_name_lat}&af_ad={gbid}&af_sub1={ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}&af_click_lookback=7d&clickid={logid}&google_aid={googleaid}&android_id={androidid}&ya_click_id={logid}",
            null,
            "https://app.appsflyer.com/com.im30.ROE.gp?af_ad={gbid}&af_sub1={ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}&af_click_lookback=7d&google_aid={googleaid}&android_id={androidid}&ya_click_id={logid}&pid=yandexdirect_int&clickid={logid}&c={campaign_name}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://app.appsflyer.com/com.im30.ROE.gp?pid=yandexdirect_int&clickid={logid}&idfa={ios_ifa}",
            null,
            "https://app.appsflyer.com/com.im30.ROE.gp?idfa={ios_ifa}&pid=yandexdirect_int&clickid={logid}&c={campaign_name}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?reqid={logid}&event-link-type=install&tracker=flurry",
            null,
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?event-link-type=install&tracker=flurry&reqid={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/ri1r8u8?campaign={campaign_name}.{campaignid}&adgroup={gbid}&creative={banner_id}_{keyword}_{phrase_id}&redirect=https://itunes.apple.com/ru/app/id1438528886?mt=8&ya_click_id={LOGID}&idfa={ios_ifa}",
            null,
            "https://app.adjust.com/ri1r8u8?campaign={campaign_name}.{campaignid}&adgroup={gbid}&creative={banner_id}_{keyword}_{phrase_id}&redirect=https://itunes.apple.com/ru/app/id1438528886?mt=8&idfa={ios_ifa}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://8jxm.adj.st/?adjust_t=t6f6c0&adjust_campaign={campaign_name}.{campaignid}&tracker_limit=1000000&adjust_adgroup={gbid}&adj_ya_click_id={LOGID}&adjust_creative={banner_id}_{keyword}&adjust_idfa={ios_ifa}&adj_deeplink=yandextaxi://",
            null,
            "https://8jxm.adj.st/?adjust_t=t6f6c0&adjust_campaign={campaign_name}.{campaignid}&tracker_limit=1000000&adjust_adgroup={gbid}&adjust_creative={banner_id}_{keyword}&adjust_idfa={ios_ifa}&adj_deeplink=yandextaxi://&adj_ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp",
            null,
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp&ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?ya_click_id={log_id}&idfa={google_aid}",
            null,
            "https://app.adjust.com/xyz4567?idfa={google_aid}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://trk.mail.ru/c/smt7g7?_1lpb_id=4702&_1lpb_clickId={logid}&mt_gaid={google_aid}&mt_sub1={banner_id}&mt_sub2={phrase_id}&mt_sub3={keyword}&mt_sub4={position_type}&mt_sub5={region_id}&mt_network={source_type}&mt_campaign={campaign_id}&mt_adset={gbid}&mt_creative={ad_id}",
            null,
            "https://trk.mail.ru/c/smt7g7?_1lpb_id=4702&_1lpb_clickId={logid}&mt_gaid={google_aid}&mt_sub1={banner_id}&mt_sub2={phrase_id}&mt_sub3={keyword}&mt_sub4={position_type}&mt_sub5={region_id}&mt_network={source_type}&mt_campaign={campaign_id}&mt_adset={gbid}&mt_creative={ad_id}&clickId={logid}&regid={logid}"
        ),
        arrayOf(
            "https://trk.mail.ru/c/a12345?clickId={logid}",
            null,
            "https://trk.mail.ru/c/a12345?clickId={logid}&regid={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?clickid={loasdagid}&click_id={logid}",
            null,
            "https://control.kochava.com/v1/cpi/click?clickid={loasdagid}&click_id={logid}"
        ),
        arrayOf(
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?reqid={logid}&event-link-type=install&tracker=flurry",
            null,
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?event-link-type=install&tracker=flurry&reqid={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&network_id=221&click_id={logid}&site_id=p277-s322-",
            null,
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&network_id=221&site_id=p277-s322-&click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/nnaes87?campaign={campaign_id}_{campaign_name_lat}&idfa={ios_ifa}&gps_adid={google_aid}&campaign_id={campaign_id}&creative_id={ad_id}&publisher_id={gbid}&ya_click_id={logid}",
            null,
            "https://app.adjust.com/nnaes87?campaign={campaign_id}_{campaign_name_lat}&idfa={ios_ifa}&gps_adid={google_aid}&campaign_id={campaign_id}&creative_id={ad_id}&publisher_id={gbid}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.link",
            null,
            "https://app.link?~click_id={logid}&%243p=a_yandex_direct"
        ),
        arrayOf(
            "http://336140.measurementapi.com/serve?action=click&publisher_id=336140&site_id=129211&destination_id=438087&my_site={source}&my_ad={ad_id}&my_keyword={keyword}",
            null,
            "http://336140.measurementapi.com/serve?action=click&publisher_id=336140&site_id=129211&destination_id=438087&my_site={source}&my_ad={ad_id}&my_keyword={keyword}&publisher_ref_id={logid}"
        ),

        arrayOf(
            "https://google.com",
            Platform.ANDROID,
            null
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id",
            Platform.ANDROID,
            "https://control.kochava.com/v1/cpi/click?network_id=1517&device_id_type=adid&adid={google_aid}&device_id={google_aid}&android_id={android_id}&click_id={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&network_id=221&adid=&click_id={logid}&device_id=&device_id_type=&site_id=p277-s322-cf1a3fda0&append_app_conv_trk_params=1&creative_id=Call%20Home%20Span%20480x320\n ",
            Platform.ANDROID,
            null
        ),
        arrayOf(
            "https://redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            Platform.ANDROID,
            "https://redirect.appmetrica.yandex.com/serve/123/?google_aid={google_aid}&click_id={logid}"
        ),
        arrayOf(
            "https://1.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            Platform.ANDROID,
            "https://1.redirect.appmetrica.yandex.com/serve/123/?google_aid={google_aid}&click_id={logid}"
        ),
        arrayOf(
            "https://12468.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            Platform.ANDROID,
            "https://12468.redirect.appmetrica.yandex.com/serve/123/?google_aid={google_aid}&click_id={logid}"
        ),
        arrayOf(
            "http://app.appsflyer.com/com.greatapp?pid=chartboost_int&c=christmas_sale&af_adset_id=54822",
            Platform.ANDROID,
            "http://app.appsflyer.com/com.greatapp?c=christmas_sale&af_adset_id=54822&advertising_id={google_aid}&oaid={oaid}&pid=yandexdirect_int&clickid={logid}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://app.appsflyer.com/com.im30.ROE.gp?pid=yandexdirect_int&af_c_id={campaign_id}_{campaign_name_lat}&af_ad={gbid}&af_sub1={ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}&af_click_lookback=7d&clickid={logid}&google_aid={googleaid}&android_id={androidid}&ya_click_id={logid}",
            Platform.ANDROID,
            "https://app.appsflyer.com/com.im30.ROE.gp?af_ad={gbid}&af_sub1={ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}&af_click_lookback=7d&google_aid={googleaid}&android_id={androidid}&ya_click_id={logid}&advertising_id={google_aid}&oaid={oaid}&pid=yandexdirect_int&clickid={logid}&c={campaign_name}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://app.appsflyer.com/com.im30.ROE.gp?pid=yandexdirect_int&clickid={logid}&idfa={ios_ifa}",
            Platform.ANDROID,
            "https://app.appsflyer.com/com.im30.ROE.gp?idfa={ios_ifa}&advertising_id={google_aid}&oaid={oaid}&pid=yandexdirect_int&clickid={logid}&c={campaign_name}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?reqid={logid}&event-link-type=install&tracker=flurry",
            Platform.ANDROID,
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?event-link-type=install&tracker=flurry&google_aid={google_aid}&adid={google_aid}&reqid={logid}"        ),
        arrayOf(
            "https://app.adjust.com/ri1r8u8?campaign={campaign_name}.{campaignid}&adgroup={gbid}&creative={banner_id}_{keyword}_{phrase_id}&redirect=https://itunes.apple.com/ru/app/id1438528886?mt=8&ya_click_id={LOGID}&idfa={ios_ifa}",
            Platform.ANDROID,
            "https://app.adjust.com/ri1r8u8?campaign={campaign_name}.{campaignid}&adgroup={gbid}&creative={banner_id}_{keyword}_{phrase_id}&redirect=https://itunes.apple.com/ru/app/id1438528886?mt=8&idfa={ios_ifa}&gps_adid={google_aid}&oaid={oaid}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://8jxm.adj.st/?adjust_t=t6f6c0&adjust_campaign={campaign_name}.{campaignid}&tracker_limit=1000000&adjust_adgroup={gbid}&adj_ya_click_id={LOGID}&adjust_creative={banner_id}_{keyword}&adjust_idfa={ios_ifa}&adj_deeplink=yandextaxi://",
            Platform.ANDROID,
            "https://8jxm.adj.st/?adjust_t=t6f6c0&adjust_campaign={campaign_name}.{campaignid}&tracker_limit=1000000&adjust_adgroup={gbid}&adjust_creative={banner_id}_{keyword}&adjust_idfa={ios_ifa}&adj_deeplink=yandextaxi://&adj_gps_adid={google_aid}&adj_oaid={oaid}&adj_ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp",
            Platform.ANDROID,
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp&gps_adid={google_aid}&oaid={oaid}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?ya_click_id={log_id}&idfa={google_aid}",
            Platform.ANDROID,
            "https://app.adjust.com/xyz4567?idfa={google_aid}&gps_adid={google_aid}&oaid={oaid}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://trk.mail.ru/c/smt7g7?_1lpb_id=4702&_1lpb_clickId={logid}&mt_gaid={google_aid}&mt_sub1={banner_id}&mt_sub2={phrase_id}&mt_sub3={keyword}&mt_sub4={position_type}&mt_sub5={region_id}&mt_network={source_type}&mt_campaign={campaign_id}&mt_adset={gbid}&mt_creative={ad_id}",
            Platform.ANDROID,
            "https://trk.mail.ru/c/smt7g7?_1lpb_id=4702&_1lpb_clickId={logid}&mt_sub1={banner_id}&mt_sub2={phrase_id}&mt_sub3={keyword}&mt_sub4={position_type}&mt_sub5={region_id}&mt_network={source_type}&mt_campaign={campaign_id}&mt_adset={gbid}&mt_creative={ad_id}&mt_gaid={google_aid}&clickId={logid}&regid={logid}"        ),
        arrayOf(
            "https://trk.mail.ru/c/a12345?clickId={logid}",
            Platform.ANDROID,
            "https://trk.mail.ru/c/a12345?mt_gaid={google_aid}&clickId={logid}&regid={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?clickid={loasdagid}&click_id={logid}",
            Platform.ANDROID,
            "https://control.kochava.com/v1/cpi/click?clickid={loasdagid}&network_id=1517&device_id_type=adid&adid={google_aid}&device_id={google_aid}&android_id={android_id}&click_id={logid}"
        ),
        arrayOf(
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?reqid={logid}&event-link-type=install&tracker=flurry",
            Platform.ANDROID,
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?event-link-type=install&tracker=flurry&google_aid={google_aid}&adid={google_aid}&reqid={logid}"        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&network_id=221&click_id={logid}&site_id=p277-s322-",
            Platform.ANDROID,
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&site_id=p277-s322-&network_id=1517&device_id_type=adid&adid={google_aid}&device_id={google_aid}&android_id={android_id}&click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/nnaes87?campaign={campaign_id}_{campaign_name_lat}&idfa={ios_ifa}&gps_adid={google_aid}&campaign_id={campaign_id}&creative_id={ad_id}&publisher_id={gbid}&ya_click_id={logid}",
            Platform.ANDROID,
            "https://app.adjust.com/nnaes87?campaign={campaign_id}_{campaign_name_lat}&idfa={ios_ifa}&campaign_id={campaign_id}&creative_id={ad_id}&publisher_id={gbid}&gps_adid={google_aid}&oaid={oaid}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.link",
            Platform.ANDROID,
            "https://app.link?%24aaid={google_aid}&~click_id={logid}&%243p=a_yandex_direct"
        ),
        arrayOf(
            "http://336140.measurementapi.com/serve?action=click&publisher_id=336140&site_id=129211&destination_id=438087&my_site={source}&my_ad={ad_id}&my_keyword={keyword}",
            Platform.ANDROID,
            "http://336140.measurementapi.com/serve?action=click&publisher_id=336140&site_id=129211&destination_id=438087&my_site={source}&my_ad={ad_id}&my_keyword={keyword}&google_aid={google_aid}&publisher_ref_id={logid}"
        ),

        arrayOf(
            "https://google.com",
            Platform.IOS,
            null
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id",
            Platform.IOS,
            "https://control.kochava.com/v1/cpi/click?network_id=1516&device_id_type=idfa&idfa={ios_ifa}&device_id={ios_ifa}&ios_idfa={ios_ifa}&click_id={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&network_id=221&adid=&click_id={logid}&device_id=&device_id_type=&site_id=p277-s322-cf1a3fda0&append_app_conv_trk_params=1&creative_id=Call%20Home%20Span%20480x320\n ",
            Platform.IOS,
            null
        ),
        arrayOf(
            "https://redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            Platform.IOS,
            "https://redirect.appmetrica.yandex.com/serve/123/?ios_ifa={ios_ifa}&click_id={logid}"
        ),
        arrayOf(
            "https://1.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            Platform.IOS,
            "https://1.redirect.appmetrica.yandex.com/serve/123/?ios_ifa={ios_ifa}&click_id={logid}"
        ),
        arrayOf(
            "https://12468.redirect.appmetrica.yandex.com/serve/123/?click_id={logid}",
            Platform.IOS,
            "https://12468.redirect.appmetrica.yandex.com/serve/123/?ios_ifa={ios_ifa}&click_id={logid}"
        ),
        arrayOf(
            "http://app.appsflyer.com/com.greatapp?pid=chartboost_int&c=christmas_sale&af_adset_id=54822",
            Platform.IOS,
            "http://app.appsflyer.com/com.greatapp?c=christmas_sale&af_adset_id=54822&idfa={ios_ifa}&pid=yandexdirect_int&clickid={logid}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://app.appsflyer.com/com.im30.ROE.gp?pid=yandexdirect_int&af_c_id={campaign_id}_{campaign_name_lat}&af_ad={gbid}&af_sub1={ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}&af_click_lookback=7d&clickid={logid}&google_aid={googleaid}&android_id={androidid}&ya_click_id={logid}",
            Platform.IOS,
            "https://app.appsflyer.com/com.im30.ROE.gp?af_ad={gbid}&af_sub1={ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}&af_click_lookback=7d&google_aid={googleaid}&android_id={androidid}&ya_click_id={logid}&idfa={ios_ifa}&pid=yandexdirect_int&clickid={logid}&c={campaign_name}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://app.appsflyer.com/com.im30.ROE.gp?pid=yandexdirect_int&clickid={logid}&idfa={ios_ifa}",
            Platform.IOS,
            "https://app.appsflyer.com/com.im30.ROE.gp?idfa={ios_ifa}&pid=yandexdirect_int&clickid={logid}&c={campaign_name}&af_c_id={campaign_id}"
        ),
        arrayOf(
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?reqid={logid}&event-link-type=install&tracker=flurry",
            Platform.IOS,
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?event-link-type=install&tracker=flurry&ios_ifa={ios_ifa}&ios_idfa={ios_ifa}&reqid={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/ri1r8u8?campaign={campaign_name}.{campaignid}&adgroup={gbid}&creative={banner_id}_{keyword}_{phrase_id}&redirect=https://itunes.apple.com/ru/app/id1438528886?mt=8&ya_click_id={LOGID}&idfa={ios_ifa}",
            Platform.IOS,
            "https://app.adjust.com/ri1r8u8?campaign={campaign_name}.{campaignid}&adgroup={gbid}&creative={banner_id}_{keyword}_{phrase_id}&redirect=https://itunes.apple.com/ru/app/id1438528886?mt=8&idfa={ios_ifa}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://8jxm.adj.st/?adjust_t=t6f6c0&adjust_campaign={campaign_name}.{campaignid}&tracker_limit=1000000&adjust_adgroup={gbid}&adj_ya_click_id={LOGID}&adjust_creative={banner_id}_{keyword}&adjust_idfa={ios_ifa}&adj_deeplink=yandextaxi://",
            Platform.IOS,
            "https://8jxm.adj.st/?adjust_t=t6f6c0&adjust_campaign={campaign_name}.{campaignid}&tracker_limit=1000000&adjust_adgroup={gbid}&adjust_creative={banner_id}_{keyword}&adjust_idfa={ios_ifa}&adj_deeplink=yandextaxi://&adj_idfa={ios_ifa}&adj_ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp",
            Platform.IOS,
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp&idfa={ios_ifa}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?ya_click_id={log_id}&idfa={google_aid}",
            Platform.IOS,
            "https://app.adjust.com/xyz4567?idfa={ios_ifa}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://trk.mail.ru/c/smt7g7?_1lpb_id=4702&_1lpb_clickId={logid}&mt_gaid={google_aid}&mt_sub1={banner_id}&mt_sub2={phrase_id}&mt_sub3={keyword}&mt_sub4={position_type}&mt_sub5={region_id}&mt_network={source_type}&mt_campaign={campaign_id}&mt_adset={gbid}&mt_creative={ad_id}",
            Platform.IOS,
            "https://trk.mail.ru/c/smt7g7?_1lpb_id=4702&_1lpb_clickId={logid}&mt_gaid={google_aid}&mt_sub1={banner_id}&mt_sub2={phrase_id}&mt_sub3={keyword}&mt_sub4={position_type}&mt_sub5={region_id}&mt_network={source_type}&mt_campaign={campaign_id}&mt_adset={gbid}&mt_creative={ad_id}&mt_idfa={ios_ifa}&clickId={logid}&regid={logid}"
        ),
        arrayOf(
            "https://trk.mail.ru/c/a12345?clickId={logid}",
            Platform.IOS,
            "https://trk.mail.ru/c/a12345?mt_idfa={ios_ifa}&clickId={logid}&regid={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?clickid={loasdagid}&click_id={logid}",
            Platform.IOS,
            "https://control.kochava.com/v1/cpi/click?clickid={loasdagid}&network_id=1516&device_id_type=idfa&idfa={ios_ifa}&device_id={ios_ifa}&ios_idfa={ios_ifa}&click_id={logid}"
        ),
        arrayOf(
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?reqid={logid}&event-link-type=install&tracker=flurry",
            Platform.IOS,
            "https://ad.apps.fm/t0MOILM-olslPmrs8T?event-link-type=install&tracker=flurry&ios_ifa={ios_ifa}&ios_idfa={ios_ifa}&reqid={logid}"
        ),
        arrayOf(
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&network_id=221&click_id={logid}&site_id=p277-s322-",
            Platform.IOS,
            "https://control.kochava.com/v1/cpi/click?campaign_id=koidtandroidus205652e7fe894452f471b27363e9ea&site_id=p277-s322-&network_id=1516&device_id_type=idfa&idfa={ios_ifa}&device_id={ios_ifa}&ios_idfa={ios_ifa}&click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/nnaes87?campaign={campaign_id}_{campaign_name_lat}&idfa={ios_ifa}&gps_adid={google_aid}&campaign_id={campaign_id}&creative_id={ad_id}&publisher_id={gbid}&ya_click_id={logid}",
            Platform.IOS,
            "https://app.adjust.com/nnaes87?campaign={campaign_id}_{campaign_name_lat}&gps_adid={google_aid}&campaign_id={campaign_id}&creative_id={ad_id}&publisher_id={gbid}&idfa={ios_ifa}&ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.link",
            Platform.IOS,
            "https://app.link?%24idfa={ios_ifa}&~click_id={logid}&%243p=a_yandex_direct"
        ),
        arrayOf(
            "http://336140.measurementapi.com/serve?action=click&publisher_id=336140&site_id=129211&destination_id=438087&my_site={source}&my_ad={ad_id}&my_keyword={keyword}",
            Platform.IOS,
            "http://336140.measurementapi.com/serve?action=click&publisher_id=336140&site_id=129211&destination_id=438087&my_site={source}&my_ad={ad_id}&my_keyword={keyword}&ios_ifa={ios_ifa}&publisher_ref_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp&install_callback=http%3A%2F%2Fpostback.yandexadexchange.net%2Fpostback%3Fwhatever",
            null,
            "https://app.adjust.com/xyz4567?label=user_id&deeplink=myapp&ya_click_id={logid}"
        ),
        arrayOf(
            "https://app.adjust.com/xyz4567?ya_click_id={logid}&idfa={google_aid}&install_callback=http%3A%2F%2Fexample.com%2Fpostback%3Fwhatever",
            null,
            "https://app.adjust.com/xyz4567?idfa={google_aid}&install_callback=http%3A%2F%2Fexample.com%2Fpostback%3Fwhatever&ya_click_id={logid}"
        ),
    )

    @Test
    @TestCaseName("testParseUrl({index})")
    @Parameters(method = "provideTrackingUrls")
    fun testParseUrl(url: String, platform: Platform?, expected: String?) {
        val actual = trackingUrlParseService.parse(url, platform, ParserType.TRACKING_URL)?.getUrl()
        assertThat(actual).isEqualTo(expected)
    }

}
