#!/usr/bin/env perl

=head1 DESCRIPTION

    set-disk-scheduler.pl - выставляем scheduler для всех дисков в зависимости от типа (hdd/ssd/zfs)

    set-disk-scheduler.pl [ --<hdd|ssd|zfs> <deadline|noop|cfq> ]

    Отдает приоритет zfs дискам, т.е. если --zfs noop, то не важно, ssd они, или hdd

=cut

use strict;
use warnings;
use Getopt::Long;

my ($hdd_sched, $ssd_sched, $zfs_sched) = ("cfq", "noop", "noop");

GetOptions(
  "hdd=s" => \$hdd_sched,
  "ssd=s" => \$ssd_sched,
  "zfs=s" => \$zfs_sched,
);

my %disks = map { m!/block/([a-z]+)$!; $1 => 1 } grep { m!/block/[a-z]+$! } glob "/sys/class/block/sd*";
my (%ssd_disks, %hdd_disks);

for my $disk (keys %disks) {
    open(my $fh, "<", "/sys/class/block/$disk/queue/rotational") or next;
    my $rot = do { local $/; <$fh> };
    chomp $rot;
    if ($rot eq "0") {
        $ssd_disks{$disk} = 1;
    } elsif ($rot eq "1") {
        $hdd_disks{$disk} = 1;
    }
}

my %zfs_disks = map { /(sd[a-z]+)/; $1 => 1 } grep { /^\s*sd[a-z]+/ } split /\n/, (qx(zpool status 2>&1) || "");

for my $disk (keys %zfs_disks) {
    print "zfs disk: $disk\n";
    open(my $fh, ">", "/sys/class/block/$disk/queue/scheduler") or next;

    print "set $disk sched to $zfs_sched\n";
    print $fh $zfs_sched;
}

for my $disk (keys %ssd_disks) {
    next if $zfs_disks{$disk};
    print "ssd disk: $disk\n";

    open(my $fh, ">", "/sys/class/block/$disk/queue/scheduler") or next;
    print "set $disk sched to $ssd_sched\n";
    print $fh $ssd_sched;
}

for my $disk (keys %hdd_disks) {
    next if $zfs_disks{$disk};
    print "hdd disk: $disk\n";

    open(my $fh, ">", "/sys/class/block/$disk/queue/scheduler") or next;
    print "set $disk sched to $hdd_sched\n";
    print $fh $hdd_sched;
}
