#!/usr/bin/python
# -*- coding: utf-8 -*-

description = """
парсит переданные yaml-файлы (см. каталог /data) и генерит картинку file.png

./bin/adv-draw-graph.py data/*.yaml && gwenview file.png

скрипту для работы нужны graphviz и pygraphviz

TODO
+ параметр -- в какой файл писать
+ генерировать svg, png, другие типы

"""

import sys
import argparse
import json
import re
import os
import yaml

import pygraphviz as pgv


def parse_options():
    parser = argparse.ArgumentParser(add_help=False)
    parser.add_argument("-h", "--help", dest="help", help="Справка", action="store_true")
    parser.add_argument("-o", "--output", dest="output_file", help="В какой файл записать сгенерированную картинку", type=str)
    parser.add_argument("-c", "--color-rules", dest="color_rules_file", help="Файл с правилами раскраски", type=str)
    parser.add_argument("-i", "--include-tag", dest="include_tag", help="теги, которые надо включить в схему", type=str, action='append')
    opts, extra = parser.parse_known_args()
    opts.map_files = extra

    if 'all' in opts.include_tag:
        opts.include_tag = []

    if opts.help:
        print description
        print parser.format_help()
        exit(0)

    if len(extra) <= 0:
        exit("expecting action (check|force|skip|show|clear-cmd|clear-last-try|schedule)")

    if len(extra) <= 0:
        exit("expecting at least 1 file with objects")

    return opts



def read_files(files):
    res = []
    for f in files:
        with open(f, 'r') as fd:
            m = yaml.load(fd)
            res.append(m)
    return res


def read_color_rules(f):
    if not f:
        return {}
    with open(f, 'r') as fd:
        c = yaml.load(fd)
    return c


def calc_color(obj, rules):
    color = '#eeeeee'
    if 'type' in rules and 'type' in obj and obj['type'] in rules['type']:
        color = rules['type'][obj['type']]['color']
    if 'parent' in obj and obj['parent'] in rules:
        color = rules[obj['parent']]['color']
    if obj['slug'] in rules:
        color = rules[obj['slug']]['color']
    return color


def generate_node_label(node):
    """
    собрать html-метку для вершины
    если есть атрибут href -- со ссылкой
    """
    if 'href' in node:
        title_tr = (u"<TR><TD TARGET='_blank' HREF='%s' TITLE='href'>"
        + u"<FONT COLOR='#00008B' FACE='boldfontname'><U>%s</U></FONT></TD></TR>"
        ) % (node['href'], node['name'])
    else:
        title_tr = (u"<TR><TD>"
        + u"<FONT COLOR='#00008B' FACE='boldfontname'>%s</FONT></TD></TR>"
        ) % (node['name'])

    table_parts = [u"<<TABLE BORDER='0'>", title_tr]

    if 'tags' in node:
        for t in node['tags']:
            table_parts.append(u"<TR><TD>%s</TD></TR>" % t)

    table_parts.append("</TABLE>>")

    label = "".join(table_parts)
    return label


def maps2mygraph(maps):
    mg = {
            'nodes': [],
            'edges': [],
            'subgraphs': [],
            }
    for m in maps:
        if not 'objects' in m:
            continue
        for obj in m['objects']:
            mg['nodes'].append(obj)
            if 'tags' not in obj:
                obj['tags'] = []

    for m in maps:
        if not 'links' in m:
            continue
        for l in m['links']:
            mg['edges'].append(l)
    return mg


def mygraph2graph(mygraph, color_rules):
    G = pgv.AGraph(
            strict=False,
            directed=True,
            rankdir="LR"
            )
    #c1 = G.add_subgraph(name='cluster_apps-c', label='label 12345', shape='rect', style='filled', color='grey')
    #c1 = G.add_subgraph(name='cluster_apps-c', label='label 12345', shape='rect', style='filled', color='grey')
    #c1.graph_attr['rank']='same'
    #c1.add_node(obj['slug'],
    #cluster_apps = G.add_subgraph(name='cluster_apps', label='Behavior', shape='rect', style='empty')
    #cluster_storages = G.add_subgraph(name='cluster_storages', label='Storages', shape='rect', style='empty')
    #cluster_LB = G.add_subgraph(name='cluster_LB', label='LogBroker', shape='rect', style='empty')
    #cluster_outer_world = G.add_subgraph(name='cluster_outer_world', label='Other services', shape='rect', style='empty')

    for n in mygraph['nodes']:
        color = calc_color(n, color_rules)
        node_label = generate_node_label(n)
        dest = G
        #if n['type'] in ['app', 'code', 'app-group']:
            #dest = cluster_apps
        #elif n['type'] in ['storage']:
            #dest = cluster_storages
        #elif n['type'] in ['lb-topic']:
            #dest = cluster_LB
        #elif n['type'] in ['outer-world']:
            #dest = cluster_outer_world
        dest.add_node(n['slug'],
                label=node_label,
                style="rounded,filled",
                shape="box",
                color=color
                )
    for l in mygraph['edges']:
        edge_label = l['label'] if 'label' in l else ''
        G.add_edge(l["from"], l["to"], label=edge_label)
    return G


def find_absent_nodes(mygraph):
    present = {}
    absent = {}
    for n in mygraph['nodes']:
        present[n['slug']] = True
    for e in mygraph['edges']:
        to_add = []
        for s in [ e['from'], e['to'] ]:
            if s not in present:
                n = {
                        'slug': s,
                        'name': s,
                        'type': 'type',
                        'tags': [],
                        }
                absent[s] = n
    result = absent.values()
    print yaml.dump(result)
    return

def filter_nodes_by_tags(mygraph, tags):
    print "tags: %s" % tags
    if len(tags)<=0:
        return mygraph
    res = {
            'nodes': [],
            'edges': [],
            'subgraphs': [],
            }
    tags_set = set(tags)
    nodes = {}
    for n in mygraph['nodes']:
        if set(n['tags']) & tags_set:
            res['nodes'].append(n)
            nodes[n['slug']] = True
    for e in mygraph['edges']:
        if e['from'] in nodes and e['to'] in nodes:
            res['edges'].append(e)
    return res


def run():
    opts = parse_options()
    maps = read_files(opts.map_files)
    color_rules = read_color_rules(opts.color_rules_file)
    mg = maps2mygraph(maps)
    mg = filter_nodes_by_tags(mg, opts.include_tag)
    #find_absent_nodes(mg)
    G = mygraph2graph(mg, color_rules)
    G.layout(prog="dot")
    if opts.output_file:
        G.draw(opts.output_file)

    exit(0)

if __name__ == '__main__':
    run()
