package Base;

use strict;
use warnings;

use LWP::Simple;
use Time::HiRes;
use Data::Dumper;
use Term::ANSIColor;
use SOAP::Lite;
use JSON;
use Encode qw/encode_utf8 decode_utf8/;
use HTTP::Request::Common;
use HTTP::Response;
use LWP::UserAgent;
use Socket;

BEGIN {
    $Net::HTTPS::HTTPS_SSL_SOCKET_CLASS = "Net::SSL"
}

=head2 call_method(method, data)

    Функция вызывает указанный метод интерфейса программирования приложений Яндекс.Директ ( API )

=cut

sub call_method(%)
{
    my %REQ = @_;

    my $ltm = Time::HiRes::time();
    my $request_method = $REQ{options}->{method} || 'json';

    my $pretify = $REQ{options}->{pretty} ? { pretty => 1 } : {};
    print STDERR "Call method: ".$REQ{cmd}."\n" if $REQ{options}{debug};
    print STDERR "Params:".(ref $REQ{request} ? to_json($REQ{request}, $pretify) : $REQ{request})."\n"
        if $REQ{request} && $REQ{options}{debug};

    my $result = $REQ{options}{method} eq 'json' ? call_method_json(%REQ) : call_method_soap(%REQ);

    print STDERR sprintf("Process time: %0.4f sec\n", Time::HiRes::time() - $ltm) if $REQ{options}{debug};

    return $result;
}

sub get_url_by_options
{
    my $options = shift;

    my $domain = $options->{server} || 'api.direct.yandex.ru';
    my $api_version = $options->{api_version} || 999;

    my $url_suffix = join "/", grep {$_} ( $options->{live} ? 'live' : '' )
                        , ( $api_version ? "v$api_version" : '' )
                        , ( $options->{method} eq 'json' ? "json" : "soap" );

    my $scheme = 'https';
    my $port = $options->{port}; # default = '443';

    if ($options->{beta}) {
        my $hostname = `hostname`;
        if($hostname =~ /-precise/) {
            $domain = $options->{beta} . ".beta1.direct.yandex.ru";
        } else {
            $domain = $options->{beta} . ".beta2.direct.yandex.ru";
        }
        $port ||= '14443';
    }

    if ($options->{test}) {
        $port ||= '14443';
        $domain = 'test-direct.yandex.ru';
    } elsif ($options->{test2}) {
        $port ||= '14443';
        $domain = 'test2-direct.yandex.ru';
    }

    if ($options->{http}) {
        $scheme = 'http';
        $port ||= '14080';
    }

    my $url = "$scheme://$domain".($port ? ":$port" : '')."/$url_suffix";
    return $url;
}

sub call_method_json
{
    my %OPT = @_;
    my ($url, $cmd, $headers, $data, $options) = @{\%OPT}{qw/url cmd headers request options/};

    my $json_api_data = {};
    $json_api_data->{param} = $data;
    $json_api_data->{method} = $cmd;


    $json_api_data->{locale} = $options->{locale} || "ru";
    $json_api_data->{agcl} = 1;

    $json_api_data->{$_} = $headers->{$_} foreach (qw/login token application_id fake_login/);

    my $ua = LWP::UserAgent->new( timeout => 600, ssl_opts => { verify_hostname => 0 });

    my $http_headers = HTTP::Headers->new;
    # $http_headers->push_header('Content-Type' => 'application/x-www-form-urlencoded');
    # $http_headers->push_header('Content-Type' => 'text/xml');

    my $dreq = $OPT{options}{method} eq 'json'
            ? encode_utf8(to_json($json_api_data))
            : to_json($json_api_data);

    print STDERR "Request: ".$dreq."\n" if $options->{debug} || $options->{shortdebug};

    my $request = post_request($url, $http_headers, $dreq);
    my $response = $ua->request($request);

    print STDERR "---\n".$request->as_string, "\n\n" if $options->{debug};
    print STDERR "----\n".$response->as_string, "\n\n" if $options->{debug};

    if ($response->is_success) {
        my $content = $response->decoded_content();
        if($options->{pretty} && $options->{debug}) {
            $content = to_json( from_json($content), { pretty => 1} );
        }
        print STDERR "Response: ".$content."\n" if $options->{debug} || $options->{shortdebug};

        if ($OPT{cmd} ne 'GetCampaignsSpecialStat') {
            my $data = from_json(Encode::decode_utf8($content));
            return $data->{data};
        } else {
            return $content;
        }
    } else {
        die $response->status_line;
    }
}

sub call_method_soap
{
    my %OPT = @_;
    my ($url, $cmd, $headers, $data, $options) = @OPT{qw/url cmd headers request options/};

    local %ENV;
    $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;
    my $soap = SOAP::Lite
        -> uri('API')
        -> proxy($url);

    $soap->on_debug(sub {print STDERR @_}) if $options->{debug};

    my $request = $soap->call($cmd
                            , @{ $headers || []}
                            , $data);

    unless ($request->fault) {
        # в случае успешного выполнения запроса - возвращаем результат
        return $request->result;
    } else {
        # если возникла ошибка - выводим сообщение в STDERR и завершаем выполнение программы
        my $error = join "\n", "Error code: ".$request->faultcode
                                , "Describe error: ".($request->faultstring || '')
                                , "Details: ".($request->faultdetail || '');

        die $error;
    }
}

=head2 get_report(reportURL)

    Фукнкция загружает указанный URL( полученный с помощью метода GetReportURL ) и возвращает содержимое страницы( готовый XML отчет )

=cut

sub get_report
{
    return get(shift);
}

sub check_wsdl($)
{
    my $url = shift;
    
    my $res;
    
    my $response = LWP::UserAgent->new()->request(HTTP::Request->new('GET', $url));
    if ($response->code != 200) {
        die Encode::decode_utf8($response->message);
    } else {
        $res = $response->content();
    }

    return $res;
}

sub process_external_value_by_spec_syntax
{
    my $value = shift;
    
    return undef unless defined $value;

    if ($value eq 'empty_string') {
        $value = '';
    } elsif ($value eq 'empty_array') {
        $value = [];
    } elsif ($value eq 'empty_hash') {
        $value = {};
    } elsif ($value eq 'undef') {
        $value = undef;
    }

    return decode_utf8($value);
}

sub call
{
    my %REQ = @_;
#warn Dumper {REQ => \%REQ};
    my $result = call_method(%REQ);

    print STDERR ProcessResult::short_info_about_result($result)."\n" if $REQ{options}{debug};

    my $res1 = ProcessResult::process_result_by_cmd(result => $result, %REQ);

    return $res1;
}

sub _http_post
{
    my $url = shift;
    my $post_content = shift;
    my %options = @_;

    my $response = LWP::UserAgent->new(%options)->post($url, Content => $post_content);

    if ($response->is_success()) {
        return $response->content();
    } else {
        print STDERR "$0 error \nresponse-code: " . $response->code . "\ncontent: " . $response->content();
        return;
    }
}

sub post_request {
    my $url = shift;
    my $http_headers = shift;
    my $data = shift;
    my $hostname = host_from_url($url);
    my $ipv4 = get_ipv4_host_by_name($hostname);
    my $port = port_from_url($url);
    $url =~ s/$hostname/$ipv4/g;
    $http_headers->header(host => $hostname . ($port? ":$port" :'') );
    return HTTP::Request->new('POST', $url, $http_headers, $data);
}

# not sure how long it'll be ipv4 only but when it's not, perl probably will be
# fixed to work with ipv6 properly anyway
sub get_ipv4_host_by_name {
    my $name = shift or return;
    my $ipaddr_packed = gethostbyname($name);

    return defined $ipaddr_packed
        ? inet_ntoa($ipaddr_packed)
        : undef;
}

sub host_from_url {
    my $url = shift;
    $url =~ m!^https?\://(.+?)[:/]! or die "wrong url format $url";
    return $1;
}

sub port_from_url {
    my $url = shift;
    $url =~ m!^https?\://(.+?)/! or die "wrong url format $url";
    my ($host, $port) = split(':', $1);
    return $port;
}
 


1;
