#!/usr/bin/perl -w

use strict;

=pod

=head1 NAME

    Скрипт для тестирования задач API
    
=head1 DESCRIPTION

    

=head1 USAGE

    1. Нужен файл в /home/your_login/direct-api.oauth - в котором для ваших логинов прописаны токены и application_id приложений
    2. Пример запуска:
    
    ./api.pl --login="mirage-super" --fake="holodilnikru" --beta=8038 --live --cmd="GetRetargetingGoals" --ulogin="holodilnikru"
    ./api.pl --login="mirage-agency4" --test --live --cmd="PingAPI" --AnyCustomParameter=AnyValue
    ./api.pl --login="mirage59" --soap --live --cmd="PingAPI"
    
    Стандартные параметры:
        --debug -   распечатать запрос-ответ в STDERR
        --pretty -  форматировать вывод json в debug-сообщениях
        --live  -   запрос в live-версию
        --version=\d    - номер версии
        --login=\s  - логин оператора запроса, данные для авторизации которого есть в файле авторизации
        --cmd=\s    - название метода
        --json  - сделать запрос в json (default)
        --usetoken  - авторизоваться по токенам (default)
        --cert  -   авторизоваться по сертификатам (должны лежать по специальному пути)
        --soap  -   сделать запрос по soap
        --server=\s -   адрес сервера
        --fake=\s   -   фейково авторизоваться на бете/ТС под любым логином (при условии, что оператор запроса - валидный супер-пользователь)
        --test
        --test2
        --beta=\d

    Параметры для методов:        
        --json_params=s -  Параметры задаются в JSON, все остальные ключи параметров игнорируются
        --json_params_file=s - Тоже, что и json_params но параметры беруться из файла
        --cid=\d,\d -   номер кампании
        --bid=\d,\d -   номер объявления
        --ulogin=\s -   логин клиента
        --param=\s
        --param2=\s
        --currency=\s
        + любые кастомные параметры, которые требуются для метода и которые поддерживаются в RequestParams.pm

=cut

use SOAP::Lite;
use Crypt::SSLeay;

use YAML::Syck;
use JSON::Syck;
use JSON;

use Time::HiRes;
use Text::Iconv;
use Data::Dumper;
use Getopt::Long qw/:config pass_through/;
use File::Slurp;
use Term::ANSIColor;

use Encode qw/encode_utf8/;

use Base qw/call/;
use ProcessResult;
use RequestParams;

binmode(STDOUT, ":utf8");

$YAML::Syck::ImplicitUnicode = 1;

my ($cmd, $beta, $debug, $testing, $operator_login, $version, $short, $server);
my ($usetoken, $json, $cert_flag, $get_token_flag, $testing2, $http);

our ($cid, $bid, $ulogin, $param, $param2);
my ($empty_hash, $empty_array, $empty_string, $null_value);
my ($json_params, $json_params_file, $pretty);
my ($locale, $live, $shortdebug, $fake_auth, $soap, $currency);
my %universal_params = ();

my %get_opt_long_config = (
    'cmd=s' => \$cmd,
    'pretty' => \$pretty,
    'beta=s' => \$beta,
    'debug' => \$debug,
    'shortdebug' => \$shortdebug,
    'test' => \$testing,
    'test2' => \$testing2,
    'login=s' => \$operator_login,
    'version=s' => \$version,
    'usetoken' => \$usetoken,
    'short' => \$short,
    'param=s' => \$param,
    'param2=s' => \$param2,

    'cid=s' => \$cid,
    'bid=s' => \$bid,
    'ulogin=s' => \$ulogin,
    'json' => \$json,
    'soap' => \$soap,
    'http' => \$http,
    'cert' => \$cert_flag,
    'get_token=s' => \$get_token_flag,
    'server=s' => \$server,
    'empty_string' => \$empty_string,
    'empty_array' => \$empty_array,
    'empty_hash' => \$empty_hash,
    'null_value' => \$null_value,

    'fake=s' => \$fake_auth,
    'json_params=s' => \$json_params,
    'json_params_file=s' => \$json_params_file,
    'locale=s' => \$locale,
    'live' => \$live,
    'currency=s' => \$currency,
#     'custom=s%' => \%universal_params,
);

# находим и заполняем кастомные поля
foreach my $arg (@ARGV) {
    my ($name) = $arg =~ /^\-+(.*?)(?:\=|$)/;
    next if !$name || defined $get_opt_long_config{"$name=s"} || defined $get_opt_long_config{"$name"};
    warn $name;
    $universal_params{ $name } = "";
    $get_opt_long_config{ "$name=s" } = \$universal_params{ $name };
}

GetOptions(
    %get_opt_long_config    
) || exit(1);

$version ||= 4;

$json = 1 if !$soap;

my %REQ;
$REQ{$_} = $universal_params{$_} foreach (keys %universal_params);
$REQ{cid} = Base::process_external_value_by_spec_syntax($cid);
$REQ{bid} = Base::process_external_value_by_spec_syntax($bid);
$REQ{ulogin} = Base::process_external_value_by_spec_syntax($ulogin);
$REQ{param} = Base::process_external_value_by_spec_syntax($param);
$REQ{param2} = Base::process_external_value_by_spec_syntax($param2);
$REQ{cmd} = $cmd || 'PingAPI';
$REQ{Currency} = $currency;

my $AUTH_CONF_FILE = "/home/".( [getpwuid($<)]->[0] )."/direct-api.oauth";

unless (-f $AUTH_CONF_FILE) {
    die "Don't exists auth file: $AUTH_CONF_FILE";
}

my $TOKENS = from_json(read_file($AUTH_CONF_FILE));

unless ($operator_login) {
    die "Do not specified operator login";
}

my %OAUTH = ();
$OAUTH{token} = $TOKENS->{$operator_login}{prod}{token};
$OAUTH{application_id} = $TOKENS->{$operator_login}{prod}{application_id};

# базовые параметры запроса
my $request_options = {
    debug => $debug,
    pretty => $pretty,
    shortdebug => $shortdebug,
    
    beta => $beta,
    test => $testing,
    test2 => $testing2,
    server => $server,
    
    http => $http,
    api_version => $version,
    live => $live,

    method => $soap ? 'soap' : 'json',
    locale => $locale || 'en',
};

if ($get_token_flag) {
    # показать урл, по которому можно получиться токен

    my $OAUTH_URL = "https://oauth.yandex.ru";

    my $application_client_id = $OAUTH{application_id};
    my $get_token_url = "$OAUTH_URL/authorize?client_id=$application_client_id&response_type=code";
    print $get_token_url, "\n";
    
    my $OAUTH_URL_CODE = "$OAUTH_URL/token";
    my $req_vars = {
        grant_type => 'authorization_code'
        , code => ($get_token_flag > 1 ? $get_token_flag : 0),
        , client_id => $application_client_id
    };  

    my $json_tokens = _http_post($OAUTH_URL_CODE, $req_vars);
    print $json_tokens ? Dumper(from_json($json_tokens)) : 'empty';

    exit(1);
}

my $request_headers;

if ($cert_flag) {
    my $base_cert_url = "/home/".( [getpwuid($<)]->[0] )."/certs/$operator_login";

    $ENV{HTTPS_CERT_FILE} = "$base_cert_url/cert.crt";
    $ENV{HTTPS_KEY_FILE}  = "$base_cert_url/private.key";

    # общие для всех сертификатов данные
    $ENV{HTTPS_CA_FILE}   = "$base_cert_url/cacert.pem";
    $ENV{HTTPS_CA_DIR}    = $base_cert_url;
    
} else {

    my $login = $operator_login;
    my $token = $OAUTH{token};
    my $application_client_id = $OAUTH{application_id};

    if ($soap) {
        my @headers = (
            SOAP::Header->name("token")->value($token)->type("")
            , SOAP::Header->name("login")->value($login)->type("")
            , SOAP::Header->name("application_id")->value($application_client_id)->type("")
            , SOAP::Header->name("locale")->value($locale)->type("")
        );
        
        push @headers, SOAP::Header->name("fake_login")->value($fake_auth)->type("") if $fake_auth;
        
        $request_headers = \@headers;

    } elsif ($json) {

        $request_headers = {
            token => $token,
            login => $login,
            application_id => $application_client_id,
            locale => $locale,
            # TODO: finance tokens
        };

        $request_headers->{fake_login} = $fake_auth if $fake_auth;
    }
}

RequestParams::set_method($request_options->{method});

if($json_params_file) {
    $json_params = read_file($json_params_file);
}

my $request_data = RequestParams::get_request_data_by_params(
    %REQ
    , 'empty_string' => $empty_string
    , 'empty_array' => $empty_array
    , 'empty_hash' => $empty_hash
    , 'null_value' => $null_value
    , 'json_params' => $json_params
);

# check_wsdl("http://soap.direct.yandex.ru/wsdl/v4/");

my $api_url = Base::get_url_by_options($request_options);
print STDERR "URL: ".$api_url."\n" if $debug;

if ($cmd) {
    my $result = Base::call(url => $api_url
                                , cmd => $cmd
                                , request => $request_data
                                , headers => $request_headers
                                , options => $request_options
                                , custom_options => \%REQ);
}
