#!/usr/bin/perl -w

=head1 NAME


=head1 DESCRIPTION

    Скрипт для поддержания "автоматических" рабочих копий в стиле Директа. 
    Инструкции для создания/удаления бет читае из stdin'а или ходит за ними по http

    + аккуратно обновляет рабочие копии
    + если бранч, на который смотрела РК, удалили, аккуратно удаляет рабочую копию

    "аккуратно" == с выполнением необходимых дополнительных действий (postupdate, prerm, postcreate)

    Опции:
        -h, --help
            показать справку

    Параметры
    

    $Id$

=cut



use strict;
use warnings;

use Pid::File::Flock qw/:auto/;
use Cwd qw/getcwd chdir/;
use Getopt::Long;
use List::Util qw(min);
use List::MoreUtils qw{ uniq firstidx};
use JSON;
use Yandex::Hostname;
use EV;

use Yandex::HTTP qw/http_parallel_request/;
use Yandex::JugglerQueue;
use Yandex::Log;

use BetaPorts;
use ProjectSpecific qw/svn_url tabula_url/;

use Data::Dumper;

use utf8;
use open ':std' => ':utf8';

our $VERBOSE = 0;
our $UPDATE_TIMEOUT = 600;

#.......................................................

run() unless caller();

sub run
{
    umask 022;
    $ENV{PATH} = "$ENV{PATH}:/usr/local/bin";
    $ENV{LC_ALL} = "C";

    my $tabula_autobeta_url = tabula_url()."/autobeta";

    my %O = parse_options();

    # Список имеющихся рабочих копий получаем в начале, чтобы не удалять синхронно созданные автобеты, о которых воркер "не знает"
    my $existing_wc = BetaPorts::get_used_paths(for => 'autobeta');

    ### Добываем инструкции (список автобет и действий для них)
    my $todo_json = "";
    if ($O{use_tabula}){
        my $url = "$tabula_autobeta_url/instructions";
        my $resp = Yandex::HTTP::http_parallel_request(
            GET => { 0 => { url => $url } },
            num_attempts => 1,
            timeout => 15,
            soft_timeout => 0,
        )->{0};

        die "$resp->{headers}->{Status} $resp->{headers}->{Reason}" unless $resp->{is_success};

        $todo_json = $resp->{content};
    } else {
        $todo_json = join "", <>;
    }

    my $todo;
    eval {
        $todo = from_json($todo_json);
    };
    if ($@){
        print "$@\ninstructions:\n$todo_json";
        exit 1;
    }

    write_log("todo ".to_json($todo));

    my $hostname = Yandex::Hostname::hostfqdn();
    write_log("current hostname: $hostname");

    # для определённых fqdn преобразуем их в имена, под которыми эти хосты известны в Табуле
    # лучше всего в Табуле завести дополнительное поле "целевой хост осознает себя таким-то", и сравниваться с этим полем
    my $tabula_hostname = {
        'ppcdev1.da.yandex.ru' => 'ppcdev1.yandex.ru',
        'ppcdev2.da.yandex.ru' => 'ppcdev2.yandex.ru',
        'ppcdev3.da.yandex.ru' => 'ppcdev3.yandex.ru',
    }->{$hostname};
    if ($tabula_hostname && ($tabula_hostname // '') ne $hostname) {
        write_log("filtering for hostname: $tabula_hostname");
    }
    $tabula_hostname //= $hostname;
    my $autobetas = [grep { $_->{target_host} eq $tabula_hostname} @{$todo->{autobetas}}];
    $todo->{autobetas} = $autobetas;

    write_log("filtered todo ".to_json($autobetas));

    ### Собираем дополнительные данные и готовимся к основной работе: 
    # * Для новых автобет создаем каталоги
    # * Для существующих -- добавляем в описание данные про рабочую копию (бранч, ревизия и т.п.)
    for my $ab (@$autobetas){
        next if $ab->{status} eq 'creating';

        if ($ab->{status} eq 'to create'){
            $ab->{port} = eval{ BetaPorts::get_free_beta(for => 'autobeta') } unless $ab->{port} > 0;
            unless ( $ab->{port} ) {
                push @{$ab->{errors}}, "can't get free port";
                next;
            }
            unless ( BetaPorts::is_port_free( $ab->{port}) ){
                push @{$ab->{errors}}, "port $ab->{port} already in use";
                next;
            }
            $ab->{path} = compose_autobeta_dirname(source_branch => $ab->{source_branch}, port => $ab->{port});
            unless( defined $ab->{path} ){
                push @{$ab->{errors}}, "can't compose dirname for autobeta";
                next;
            }
            mkdir $ab->{path};
            -d $ab->{path} or push @{$ab->{errors}}, "can't mkdir $ab->{path}";
        }

        if ( ! -d $ab->{path} && $ab->{status} ne 'to delete' ){
            push @{$ab->{errors}}, "path $ab->{path} doesn't exist";
            next;
        }

        next if $ab->{status} =~ /^(to create)/;

        my $svn_info = `svn info $ab->{path}`;
        push(@{$ab->{errors}}, "can't perform `svn info $ab->{path}`"), next if $?;

        ($ab->{svn_url}, $ab->{svn_root}, $ab->{revision}, ) = $svn_info =~ /^URL:\s*(.*?)\n.*Repository Root:\s*(.*?)\n.*^Revision:\s*(.*?)\n/sm;

        my $svn_info_branch = `svn info $ab->{svn_url}`;
        ($ab->{last_changed_rev}, ) = $svn_info_branch =~ /^Last Changed Rev:\s*(.*?)\n/sm;

        ($ab->{relative_svn_path} = $ab->{svn_url}) =~ s/^\Q$ab->{svn_root}\E//;
        my $ignore_relative_path_difference;
        if ($ab->{status} eq 'to delete' && $ab->{svn_root} =~ m!^svn\+ssh://svn\.yandex\.ru/direct$!) {
            # беты со старым (до переезда в Аркадию) репозиторием можно удалить
            $ignore_relative_path_difference = 1; 
        }

        push @{$ab->{errors}}, "relative svn path and source_branch differ ($ab->{relative_svn_path}, $ab->{source_branch})" if $ab->{relative_svn_path} ne $ab->{source_branch} && !$ignore_relative_path_difference;
    }
    write_log("autobetas ".to_json($autobetas, { pretty => 1 }));

    ### В список "что делать" добавляем удаление "неожиданных" каталогов (которые отсутствуют в инструкции с сервера)
    # не делаем удаление раньше для того, чтобы в одну итерацию не мог освободиться порт и тут же оказаться занятым новой бетой --
    # это странноватая ситуация для сервера
    my %known_path = map {$_->{path} => 1} grep {$_->{path}} @$autobetas;
    for my $wc (grep {!$known_path{$_}} @$existing_wc ){
        my $ctime = (stat($wc))[10];
        my $now = time();
        if ($now - $ctime > 1800) {
            # Помечаем пути как 'unexpected' только если с момента создания прошло более чем полчаса. Таким образом, не удаляются автобеты, находящиеся в процессе создания (по синхронному запросу) на момент запуска воркера.
            push @$autobetas, {path => $wc, status => 'unexpected'};
        }
    }

    # Смотрим, что изменилось в репозиториях, на которые ссылаются автобеты 
    # (svn up будем делать только там, где по svn log видно, что что-то обновится)
    # Обновление устроено так, что обновится бета, смотрящая на любой репозиторий
    # при этом создается автобета только с основным репозиторием проекта
    # Если с сервера присылать не только относительный путь, но и адрес репозитория -- 
    # автобеты смогут обслуживать разные проекты на одном сервере
    my @repos = uniq map {$_->{svn_root}} grep { $_->{status} !~ /^(to create|to delete|unexpected|creating)$/ && !@{$_->{errors} || []} } @$autobetas;
    write_log("repositories: ", \@repos);
    my %CHANGES;
    for my $repository ( @repos ){
        my $svn_info = `svn info $repository`;
        my ($head_rev, ) = $svn_info =~ /^Revision:\s*(.*?)\n/sm;
        my @wcs = grep {($_->{svn_root}||'') eq $repository} @$autobetas; 
        my $r = min map {$_->{revision}} @wcs;
        my $svn_log_cmd = "svn log -r $r:$head_rev -v $repository";
        if ($repository eq 'svn+ssh://arcadia.yandex.ru/arc') {
            # в Аркадии смотрим только на нужное
            my @paths = ('trunk/arcadia/direct/perl', 'branches/direct/perl');
            $svn_log_cmd .= ' ' . join(' ', @paths);
        }
        write_log("performing $svn_log_cmd");
        my $log = `$svn_log_cmd`;
        my @changed_paths = split "\n", join "\n", ($log =~ m/^Changed paths:\n(.*?)\n\n/gsm);
        # удалённые ветки могут быть созданы заново, поэтому проходим по всему логу, чтобы определить окончательный статус
        my %is_deleted;
        for my $p (@changed_paths) {
            my (undef, $status, $path) = split /\s+/, $p;
            $is_deleted{$path} = 1 if $status eq 'D';
            delete $is_deleted{$path} if $status eq 'A';
        }
        my @deleted_paths = keys %is_deleted;
        s/^.*?([^ ]*)$/$1/ for @changed_paths;
        @changed_paths = uniq @changed_paths;

        $CHANGES{$repository} = {
            #changed => \@changed_paths,
            deleted => { map {$_ => 1} @deleted_paths },
            HEAD => $head_rev,
        };
    }
    write_log("CHANGES: ".to_json(\%CHANGES, { pretty => 1 }));

    # выполняем нужные svn-действия для каждой рабочей копии: svn co, svn up. Или rm -rf, если надо удалить бету
    my $workpath = getcwd();
    for my $ab ( @$autobetas ){
        next if $ab->{status} eq 'creating';

        if( $ab->{status} eq 'to delete' && (! $ab->{path} || ! -e $ab->{path} ) ){
            # автобету еще не успели запустить, а ее уже удалили
            # или катлог удалили руками, и пришла инструкция удалять -- подтверждаем, что удалили 
            $ab->{result} = "deleted";
            next;
        }

        next if @{$ab->{errors} || []} > 0;

        if ( !$ab->{path} || ! -d $ab->{path} ){
            push @{$ab->{errors}}, "wrong path '$ab->{path}'";
            next;
        }

        my $time = localtime;
        chdir $workpath;
        chdir $ab->{path};
        ### Создание
        if ( $ab->{status} eq 'to create' ){
            my $svn_url = svn_url('root').$ab->{source_branch};
            write_log("creating beta $ab->{path} with $svn_url");

            `svn co $svn_url .`;
            ($ab->{revision}, ) = `svn info .` =~ /^Revision:\s*(.*?)$/sm;
            # postcreate
            my $stdout =  `direct-mk beta_postcreate --lock beta-$ab->{port} 2>&1`;
            write_log("beta_postcreate output tail: \n".substr($stdout, -1000)."\n\n");
            push(@{$ab->{errors}} , "postcreate for $ab->{path} failed\n"), next if $?;
            `chmod -R go-w .`;
            $ab->{result} = "created";
            next;
        } 
        ### Удаление
        if ( $ab->{status} =~ /^(to delete|unexpected)$/ || $CHANGES{$ab->{svn_root}}->{deleted}->{$ab->{relative_svn_path}} ){
            write_log("deleting $ab->{path}");
            # prerm
            my $stdout = `direct-mk beta_prerm --lock beta-$ab->{port} 2>&1`;
            write_log("beta_prerm output: \n$stdout\n\n");

            # костыль: удаляем пустые директории со статусом "unexpected" несмотря на то, что prerm завершается с ошибкой
            # такие директории возникают, если чекаут рабочей копии не случился из-за сбоя соединения с SVN
            # операция создания автобеты завершается с ошибкой, но пустая директория остаётся, и prerm не выполняется никогда
            my @ab_contents = grep { ! /^\.\.?$/ } `ls -a $ab->{path}`;
            my $force_delete = $ab->{status} eq 'unexpected' && !@ab_contents;

            push(@{$ab->{errors}}, "prerm for $ab->{path} failed ($?, $!)"), next if $? && !$force_delete;
            chdir $workpath;
            `rm -rf $ab->{path}`;
            $ab->{result} = "deleted";
            next;
        }
        ### Обновление
        my $revision_count_for_force_update = ($ab->{svn_root} =~ /\barcadia\.yandex\.ru\b/) ? 10_000 : 200;
        if ( $ab->{revision} < $ab->{last_changed_rev} || $ab->{revision} < $CHANGES{$ab->{svn_root}}->{HEAD} - $revision_count_for_force_update ){
            write_log("updating $ab->{path}: rev $ab->{revision} < last changed rev $ab->{last_changed_rev} || rev $ab->{revision} < HEAD $CHANGES{$ab->{svn_root}}->{HEAD} - 200");
            #($ab->{revision}, ) = `svn info .` =~ /^Revision:\s*(.*?)$/sm;
            my $stdout = `timeout $UPDATE_TIMEOUT direct-mk --lock beta-$ab->{port} up 2>&1`;
            my $up_exit_code = $? >> 8;
            write_log("direct-mk up output tail: \n".substr($stdout, -1000)."\n\n");
            if ($up_exit_code == 124) {     # такой код отдаёт timeout, если команда не завершилась вовремя
                push(@{$ab->{errors}}, "up for $ab->{path} timed out");
            } elsif ($up_exit_code != 0) {
                push(@{$ab->{errors}}, "up for $ab->{path} failed");
            }
            $ab->{result} = "updated";
            next;
        }
    }

    ### Еще раз проходимся по бетам, и выполняем дополнительные действия (direct-mk), если требуется
    for my $ab ( @$autobetas ){
        next if @{$ab->{errors} || []} > 0 || $ab->{status} eq 'to delete' || $ab->{status} eq 'creating';
        
        my @actions = split /[,;:]+/, $ab->{actions}||'';
        next unless @actions > 0;

        chdir $workpath;
        chdir $ab->{path};

        for my $action ( @actions ){
            if ( $action !~ /^[\-\w\s]+$/ ){
                push @{$ab->{errors}}, "illegal action '$action'";
                last;
            }
            my $stdout = `direct-mk --lock beta-$ab->{port} $action 2>&1`;
            write_log("action $action output: \n$stdout\n\n");
            push(@{$ab->{errors}}, "$action failed for $ab->{path}") if $?;
        }
        $ab->{actions_result} = @{$ab->{errors} || []} == 0 ? 'ok' : 'fail'; 
        $ab->{result} ||= 'updated';
    }

    @$autobetas = grep { $_->{result} || @{$_->{errors} || []} > 0 } @$autobetas;

    ### Отправляем результаты работы на сервер
    my $result_status_json = to_json($todo);
    write_log("result ".$result_status_json);
    
    if ($O{use_tabula}){
        my $url = "$tabula_autobeta_url/done";
        my $resp = Yandex::HTTP::http_parallel_request(
            POST => { 0 => { url => $url, body => {report => $result_status_json} } },
            headers => { 'Content-type' => 'application/x-www-form-urlencoded' },
            num_attempts => 1,
            timeout => 15,
            soft_timeout => 0,
        )->{0};

        die "$resp->{headers}->{Status} $resp->{headers}->{Reason}" unless $resp->{is_success};
    } else {
        print "$result_status_json\n";
    }

    # Мониторинг
    queue_juggler_event(service => 'autobeta-worker', status => 'OK', description => 'OK');

    exit;
}

#......................................................

# разбирает параметры командной строки
sub parse_options
{
    my %O = (
    );

    $Yandex::Log::LOG_ROOT = '/var/log/autobeta'; 
    GetOptions (
        "h|help"        => \&usage,
        "t|tabula"      => \$O{use_tabula},
        "path-to-svn-url=s" => \$O{path_to_svn_url}, # ? массив
        "postcreate=s"  => \$O{postcreate},
        "prerm=s"       => \$O{prerm},
        "postupdate=s"  => \$O{postupdate},
        "verbose"       => \$VERBOSE,
        "log-dir=s"     => \$Yandex::Log::LOG_ROOT,
    ) or die "can't parse options";

    return %O;
}

# Печатает usage-сообщение
sub usage {
    system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}

{
    my $log;
sub write_log 
{
    $log ||= Yandex::Log->new(
        log_file_name => 'autobeta.log', 
        date_suf => '%Y%m%d', 
        tee => $VERBOSE,
#        msg_prefix => "[$$]",
    );
    $log->out(@_);
}
}

sub compose_autobeta_dirname
{
    my %O = @_;

    return undef if !$O{source_branch} || $O{port} !~ /^\d+$/;
    # сокращаем путь для удобства и совместимости с автобетами из старого репизтория
    $O{source_branch} =~ s!^/branches/direct/perl!/branches!;
    $O{source_branch} =~ s!^/trunk/arcadia/direct/perl!/trunk!;

    (my $branch = $O{source_branch}) =~ s![/\.]!-!g;

    my $dirname;
    if ($ProjectSpecific::PROJECT eq 'Directmod') {
        $dirname = "$BetaPorts::BETAS_BASE_DIR/mod.auto$branch.$O{port}";
    } else {
        $dirname = "$BetaPorts::BETAS_BASE_DIR/beta.auto$branch.$O{port}";
    }
    return $dirname;
}
