#!/usr/bin/perl -w

=head1 NAME

    update-svn-working-copies.pl

=head1 DESCRIPTION

    Скрипт для поддержания "автоматических" рабочих копий. 
    Рабочие копии для обработки принимает в параметрах

    + аккуратно обновляет рабочие копии
    + если встретился пустой каталог и задано правило для вычисления svn url'а -- аккуратно создает новую рабочую копию
    + если бранч, на который смотрела РК, удалили, аккуратно удаляет рабочую копию
    + если в корне рабочей копии есть файл auto-delete -- аккуратно удаляет рабочую копию

    "аккуратно" == с выполнением необходимых дополнительных действий (postupdate, prerm, postcreate)

    Опции:
        -h, --help
            показать справку

        --path-to-svn-url
            регулярное выражение для преобразование имени каталога в svn url, который надо чекаутить

        TODO
        лок, чтобы не запускать одновременно два экземпляра

        --prerm
            действия при удалении бранча (остановка apache, например)
            будут выполнены shell'ом в корне рабочей копии перед rm -rf

        --post-create
            действия при создании беты (генерация файлов, запуск apache)
            будут выполнены shell'ом в корне рабочей копии после svn checkout

        --post-update
            (необязательно) действия при обновлении беты (например, рестарт apache)
            будут выполнены shell'ом в корне рабочей копии после svn up 
            (и только в том случае, если делался svn up)

    Параметры
        имена каталогов, которые надо обработать
    
    Пример
    update-svn-working-copies.pl --postupdate './apache/init.sh restart'  /opt/www/beta.svn-multicurrency-p0.8994

    update-svn-working-copies.pl --postupdate './apache/init.sh restart' --postcreate 'mkdir protected/logs; ./protected/maintenance/beta_httpd_conf.pl --init --user=lena-san;direct-sch devtest;./protected/get_geo_js.pl' --path-to-svn-url 's#.*/beta.svnauto-(.*)\.\d+$#svn+ssh://svn.yandex.ru/direct/branches/$1#' /var/www/beta.svnauto*    

    $Id$

=cut



use strict;
use warnings;

use Cwd qw/getcwd chdir/;
use Getopt::Long;
use List::Util qw(min);
use List::MoreUtils qw{ uniq firstidx};
use File::Slurp;

use Data::Dumper;

use utf8;
use open ':std' => ':utf8';

#.......................................................


my $LOGFILE = "auto-update.log";

my %O = parse_options();

exit unless @ARGV;

# собираем данные о каталогах, которые обрабатываем
my @dirs;
for my $path (@ARGV){
    next unless -d $path;

    my $dir = { path => $path};
    push @dirs, $dir;

    my $svn_info = `svn info $path`;
    $dir->{empty} = 1, next if $?;

    $dir->{to_delete} = 1, next if -f "$path/auto-delete";

    ($dir->{svn_url}, $dir->{svn_root}, $dir->{revision}, ) = $svn_info =~ /^URL:\s*(.*?)\n.*Repository Root:\s*(.*?)\n.*^Revision:\s*(.*?)$/sm;
    ($dir->{relative_svn_path} = $dir->{svn_url}) =~ s/^\Q$dir->{svn_root}\E//;
}

# репозитории, на которые ссылаются рабочие копии
my @repos = uniq map {$_->{svn_root}} grep { !$_->{empty} && !$_->{to_delete} } @dirs;

# что изменилось в репозиториях
my %CHANGES;
for my $repository ( @repos ){
    my @wcs = grep {$_->{svn_root} eq $repository} @dirs; 
    my $r = min map {$_->{revision}} @wcs;
    my $log = `svn log -r $r:HEAD -v $repository`;
    my @changed_paths = split "\n", join "\n", ($log =~ m/^Changed paths:\n(.*?)\n\n/gsm);
    my @deleted_paths = grep {/^ {3}D/} @changed_paths;
    s/^.*?([^ ]*)$/$1/ for @changed_paths, @deleted_paths;
    @changed_paths = uniq @changed_paths;

    $CHANGES{$repository} = {
        changed => \@changed_paths,
        deleted => { map {$_ => 1} @deleted_paths },
    };
}

# выполняем нужные действия для каждой рабочей копии 
my $workpath = getcwd();
for my $wc ( @dirs ){
    my $time = localtime;
    chdir $workpath;
    chdir $wc->{path};
    if ( $wc->{empty} ){
        append_file( $LOGFILE, "$time\tgoing to create working copy\n" );
        my $svn_url = $wc->{path};
        print(STDERR "no rule to convert path to svn url\n"), next if !$O{path_to_svn_url};
        eval "\$svn_url =~ $O{path_to_svn_url}";
        print STDERR "can't convert path to svn url\n", next if $svn_url eq $wc->{path};

        `svn co $svn_url .`;
        # postcreate
        `$O{postcreate}` if $O{postcreate};
        print STDERR "something wrong with postcreate for $wc->{path}\n", next if $?;
        next;
    } 
    if ( $wc->{to_delete} || $CHANGES{$wc->{svn_root}}->{deleted}->{$wc->{relative_svn_path}} ){
        append_file( $LOGFILE, "$time\tgoing to delete working copy\n" ) ;
        # prerm
        `$O{prerm}` if $O{prerm};
        print STDERR "something wrong with prerm for $wc->{path}\n", next if $?;
        chdir $workpath;
        `rm -rf $wc->{path}`;
        next;
    }
    my $idx = firstidx { $_ =~ /^$wc->{relative_svn_path}/ } (@{$CHANGES{$wc->{svn_root}}->{changed}});
    if ( $idx != -1 ){
        append_file( $LOGFILE, "$time\tgoing to update working copy\n" ) ;
        `svn up`;
        # postupdate
        `$O{postupdate}` if $O{postupdate};
        print STDERR "something wrong with postupdate for $wc->{path}\n" if $?;
        next;
    }
}

exit;

#......................................................

# разбирает параметры командной строки
sub parse_options
{
    my %O = (
    );

    GetOptions (
        "h|help"        => \&usage,
        "path-to-svn-url=s" => \$O{path_to_svn_url}, # ? массив
        "postcreate=s"  => \$O{postcreate},
        "prerm=s"       => \$O{prerm},
        "postupdate=s"  => \$O{postupdate},
    ) or die "can't parse options";

    return %O;
}

# Печатает usage-сообщение
sub usage {
    system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}

