package BEM::Builder::Make;
use strict;
use warnings;

=head1 NAME

BEM::Builder::Make

=head1 DESCRIPTION

Сборщик bem-builder: собственно занимается проверкой/заполнением кэшей и запуском указанных в файле
конфигурации команд.

=cut

use autodie;
use Digest::MD5 qw( md5_hex );
use English qw( -no_match_vars );
use File::Basename qw( basename );
use File::Find qw( find );
use File::Path qw( make_path remove_tree );
use File::Slurp qw( read_file );
use File::Spec;
use FindBin;
use Getopt::Long;
use Guard;
use List::MoreUtils qw( any );
use Readonly;
use Try::Tiny;

use Yandex::Shell;
use Yandex::Svn;

use BEM::Builder::Tools;

Readonly my $TIMEOUT => 300; # 5 минут

$Yandex::Shell::PRINT_COMMANDS = 1;

=head1 SUBROUTINES/METHODS

=head2 hashsum

=cut

sub hashsum {
    my ( $rootdir, $files, $hash_skip_dirs ) = @_;

    my %skip_dir = map { $_ => 1 } @$hash_skip_dirs;

    my @flattened_sums;
    foreach my $file (@$files) {
        if ( -f $file || -l $file ) {
            my $relpath = File::Spec->abs2rel( $file, $rootdir );
            my $sum     = md5_hex( read_file($file) );

            my @dirs = File::Spec->splitdir($relpath);
            next if any { $skip_dir{$_} } @dirs;

            push @flattened_sums, "$relpath $sum\n";

            next;
        }

        if ( -d $file ) {
            find( sub {
                my $subfile = $File::Find::name;
                return unless -f $subfile;

                my $relpath = File::Spec->abs2rel( $subfile, $rootdir );
                my $sum     = md5_hex( read_file($subfile) );

                my @dirs = File::Spec->splitdir($relpath);
                return if any { $skip_dir{$_} } @dirs;

                push @flattened_sums, "$relpath $sum\n";
            }, $file );

            next;
        }

        die "Invalid file type for $file";
    }

    @flattened_sums = sort @flattened_sums;

    return md5_hex(@flattened_sums);
}

=head2 cleanup_directory

=cut

sub cleanup_directory {
    my ( $directory, $preserved_files ) = @_;

    local $Yandex::Shell::PRINT_COMMANDS = 0;

    my %preserve_file = map { $_ => 1 } @$preserved_files;

    my $status = svn_file_status($directory);
    $status->{ignored} ||= [];

    foreach my $fullname ( @{ $status->{ignored} } ) {
        my $relname = File::Spec->abs2rel( $fullname, $directory );
        next if $preserve_file{$relname};
        yash_system( 'rm', '-rf', $fullname );
    }

    return;
}

=head2 build_in_directory

=cut

sub build_in_directory {
    my ( $directory, $actionspec ) = @_;

    my $action_name  = $actionspec->{action_name};
    my $command_dir  = $actionspec->{command_dir};
    my $environment  = $actionspec->{environment} || {};
    my $command      = $actionspec->{command};
    my $check_files  = $actionspec->{check_files} || [];

    chdir BEM::Builder::Tools::resolve_path( $directory, $command_dir );

    local %ENV = %ENV;

    while ( my ( $k, $v ) = each %$environment ) {
        $ENV{$k} = $v;
    }

    yash_system("$command >$directory/_item-$action_name.log 2>&1");

    if ( my @missing_files = grep { ! -e "$directory/$_" } @$check_files ) {
        die "Missing files: " . join( ', ', @missing_files ) . "\n";
    }

    return 0;
}

=head2 process_action_stack

=cut

sub process_action_stack {
    my %args = @_;

    my $actions            = $args{actions};
    my $use_cache          = $args{use_cache};
    my $is_global_cache    = $args{is_global_cache};
    my $cache_root         = $args{cache_root};
    my $cache_access_group = $args{cache_access_group};
    my $builddir           = $args{builddir};
    my $data_dirs          = $args{data_dirs};
    my $hash_skip_dirs     = $args{hash_skip_dirs};

    unless ($use_cache) {
        foreach my $actionspec (@$actions) {
            my $build_errorcode = build_in_directory( $builddir, $actionspec );
            return $build_errorcode if $build_errorcode;
        }

        return 0;
    }

    unless ( -d $cache_root ) {
        yash_system("mkdir $cache_root");
    }

    my $actionspec = $actions->[-1];

    my $action_name  = $actionspec->{action_name};
    my $cache_group  = $actionspec->{cache_group};
    my $source       = $actionspec->{source};
    my $dest         = $actionspec->{dest};
    my $command_dir  = $actionspec->{command_dir};
    my $command      = $actionspec->{command};

    $dest = [$dest] unless ref $dest eq 'ARRAY';

    my $source_fullpath = BEM::Builder::Tools::resolve_path( $builddir, $source );
    my $hashsum = hashsum( $builddir, [$source_fullpath], $hash_skip_dirs );

    my $cache_file = "$cache_root/$cache_group/$hashsum.tar.gz";

    unless ( -f $cache_file ) {
        if ($ENV{DIRECT_BEM_MAKE_NO_NEW_CACHE}) {
            # если нужного кэша нет -- не кэшировать заново (т.е. новый не создавать)
            process_action_stack( %args, use_cache => 0 );
            return 0;
        }

        if ( @$actions > 1 ) {
            process_action_stack( %args, actions => [ @$actions[ 0 .. $#$actions - 1 ] ] );
        }

        unless ( -d "$cache_root/$cache_group" ) {
            mkdir "$cache_root/$cache_group";
            yash_system("chgrp $cache_access_group $cache_root/$cache_group");
            yash_system("chmod g+w $cache_root/$cache_group");
        }

        my $tmpdir = File::Temp->newdir( "$cache_root/$cache_group/$hashsum-newXXXXX", CLEANUP => 0 );

        scope_guard {
            # уйдём из временного каталога, прежде чем его удалять
            chdir '/';

            # приводим к строке, потому что File::Temp возвращает объект, который не обрабатывается File::Path:
            remove_tree("$tmpdir");
        };

        try {
            for my $subdir (@$data_dirs) {
                make_path("$tmpdir/$subdir");
                yash_system("cp -aT $builddir/$subdir $tmpdir/$subdir");
            }

            build_in_directory( "$tmpdir", $actionspec );

            if ($is_global_cache) {
                foreach my $destdir (@$dest) {
                    yash_system("chgrp -R $cache_access_group $tmpdir/$destdir");
                    yash_system("chmod -R g+w $tmpdir/$destdir");
                }
            }

            chdir $tmpdir;
            yash_system("tar czf item.tar.gz " . join( ' ', @$dest ) . " _item-$action_name.log");

            if ($is_global_cache) {
                yash_system("chgrp $cache_access_group item.tar.gz");
                yash_system("chmod g+w item.tar.gz");
            }

            rename "$tmpdir/item.tar.gz", "$cache_file";
        } catch {
            my $error = $ARG;

            my $backup_tmpdir = File::Temp->newdir( "/tmp/bem-make-error-$cache_group-$hashsum-XXXXX", CLEANUP => 0 );
            yash_system("cp -aT $tmpdir $backup_tmpdir");
            yash_system("chmod -R a+rX $backup_tmpdir");

            print STDERR "Failed to build $action_name.\n";
            print STDERR "The log file at $backup_tmpdir/_item-$action_name.log has information about occured problem.\n";

            die $error;
        };
    }

    foreach my $destdir (@$dest) {        
        remove_tree("$builddir/$destdir");
    }

    yash_system("touch -c $cache_file");
    yash_system("tar xzf $cache_file -C $builddir");
    unlink "$builddir/_item-$action_name.log";

    return 0;
}

=head2 run

=cut

sub run {
    my ($config) = @_;

    $SIG{ALRM} = sub {
        print STDERR "Timed out\n";
        kill -9, $$;
    };
    alarm( $config->{'timeout'} || $TIMEOUT );

    my $cache_info = BEM::Builder::Tools::get_cache_info($config);
    my $cache_root = $cache_info->{cache_root};
    my $rootdir    = BEM::Builder::Tools::get_rootdir($config);
    my $data_dirs  = $config->{data_directories};

    my $use_cache = $cache_info->{have_cache_dir};

    my $is_global_cache = $cache_info->{is_global_cache};
    unless ($is_global_cache) {
        print STDERR "auto-cleaning cache in $cache_root\n";

        BEM::Builder::Tools::clean_cache(
            cache_root => $cache_root,
            groups     => [ map { $_->{cache_group} } @{ $config->{make_actions} } ],
            logdir     => $config->{logdir},
            exp_days   => $config->{cache_expire_after_days},
        );
    }

    my $need_sandbox = $use_cache;

    my $build = sub {
        my ($builddir) = @_;

        return process_action_stack(
            actions            => $config->{make_actions},
            use_cache          => $use_cache,
            is_global_cache    => $is_global_cache,
            cache_root         => $cache_root,
            cache_access_group => $config->{cache_access_group},
            builddir           => $builddir,
            data_dirs          => $config->{data_directories},
            hash_skip_dirs     => $config->{hash_skip_directories},
        );
    };

    unless ($need_sandbox) {
        foreach my $dir (@$data_dirs) {
            my $fullpath = BEM::Builder::Tools::resolve_path( $rootdir, $dir );
            cleanup_directory( $fullpath, $config->{preserved_files} );
        }

        return $build->($rootdir);
    }

    my $tmpdir = File::Temp->newdir( "/tmp/bem-build-XXXXX", CLEANUP => 0 );

    scope_guard {
        chdir '/';
        remove_tree("$tmpdir");
    };

    foreach my $dir (@$data_dirs) {
        make_path("$tmpdir/$dir");

        my $file_status = svn_file_status("$rootdir/$dir");
        $file_status->{ignored} ||= [];

        my @ignored_paths = map { File::Spec->abs2rel( $_, "$rootdir/$dir" ) } @{ $file_status->{ignored} };
        my $exclude_args = join( ' ', map { "--exclude $_" } @ignored_paths );

        yash_system("rsync -a $exclude_args $rootdir/$dir/ $tmpdir/$dir/");

        foreach my $preserved_file ( @{ $config->{preserved_files} } ) {
            if (-e "$rootdir/$dir/$preserved_file") {
                yash_system("cp -a $rootdir/$dir/$preserved_file $tmpdir/$dir/$preserved_file");
            }
        }
    }

    my $result = $build->("$tmpdir");
    return $result if $result;

    foreach my $dir (@$data_dirs) {
        yash_system("rsync -a --delete $tmpdir/$dir/ $rootdir/$dir/");
    }

    return $result;
}

1;
