package BEM::Builder::Tools;
use strict;
use warnings;

=head1 NAME

BEM::Builder::Tools

=head1 DESCRIPTION

Библиотека функций bem-builder.

=cut

use Cwd;
use File::Path qw( remove_tree );
use File::Spec;
use File::stat;
use Readonly;
use Try::Tiny;

use Yandex::Log;

Readonly my $DEFAULT_EXP_DAYS => 7;

=head1 SUBROUTINES/METHODS

=head2 get_rootdir($config)

=cut

sub get_rootdir {
    my ($config) = @_;
    return resolve_path( $config->{rootdir} );
}

=head2 resolve_path(@components)

=cut

sub resolve_path {
    my (@components) = @_;

    my $path = File::Spec->catfile(@components);
    return Cwd::realpath($path);
}

=head2 get_cache_info($config)

Возвращает следующую информацию о том, какой кэш нужно использовать:

    have_cache_dir  - удалось извлечь из конфигурации каталог кэша (0 - значит "пользователь попросил не кэшировать)
    cache_root      - путь к корневому каталогу кэша (например: /var/cache/ppc)
    is_global_cache - является ли кэш общесистемным

Если пользователь указал в качестве параметра --cache несуществующий каталог, функция порождает исключение.

=cut

sub get_cache_info {
    my ($config) = @_;

    my $cache_spec = $config->{cache} || 'global';

    my $cache_root;
    if ( $cache_spec eq 'none') {
        $cache_root = ''
    } elsif ( $cache_spec eq 'global' ) {
        $cache_root = $config->{cache_root}->{global};
    } elsif ( $cache_spec eq 'user' ) {
        $cache_root = resolve_path( $ENV{HOME}, $config->{cache_root}->{user} );
    } elsif ( -d $cache_spec ) {
        $cache_root = $cache_spec;
    } else {
        die "Недопустимое значение --cache $cache_spec: " .
            "должно быть global, user, none или путь к существующему каталогу\n";
    }

    my $have_cache_dir = $cache_root ne '';

    return {
        cache_root      => $cache_root,
        have_cache_dir  => $have_cache_dir,
        is_global_cache => $cache_spec eq 'global',
    };
}

=head2 get_log_func

=cut

sub get_log_func {
    my ( $logdir, $logname ) = @_;

    if ($logdir) {
        $Yandex::Log::LOG_ROOT = $logdir;

        my $log = Yandex::Log->new( log_file_name => $logname, date_suf => '%Y%m%d', lock => 1 );
        return sub {
            my ($msg) = @_;
            return $log->out($msg);
        };
    }

    return sub {
        my ($msg) = @_;
        print STDERR "$msg\n";
    };
}

=head2 clean_cache

=cut

sub clean_cache {
    my %args = @_;

    my $cache_root = $args{cache_root};
    my $groups     = $args{groups};
    my $logdir     = $args{logdir};
    my $exp_days   = $args{exp_days} || $DEFAULT_EXP_DAYS;

    my $log_func = get_log_func( $logdir, 'bem-clean-cache.log' );

    $log_func->('start');

    $log_func->( "cleaning cache in $cache_root, user = $ENV{USER}, groups = " . join( ', ', @$groups ) );

    my @cache_subdirs = map { "$cache_root/$_" } @$groups;

    my $error_count = 0;

    foreach my $cache_filename ( sort map { glob("$_/*") } @cache_subdirs ) {
        my $mtime     = stat($cache_filename)->mtime;
        my $mtime_out = scalar( localtime $mtime );

        if ( $mtime > time - $exp_days * 86400 ) {
            $log_func->("skipping: $cache_filename ($mtime_out)");
            next;
        }

        try {
            if ( -f $cache_filename ) {
                $log_func->("removing: $cache_filename ($mtime_out)");
                unlink $cache_filename;
            } elsif ( -d $cache_filename ) {
                $log_func->("removing directory: $cache_filename ($mtime_out)");
                remove_tree($cache_filename);
            } else {
                die "$cache_filename: invalid file type";
            }

            # дополнительная проверка: если по каким-то причинам элемент не удалился, но исключение
            # не сгенерировалось, оно сгенерируется здесь;
            # это актуально, потому что обработка ошибок в File::Path сделана странно
            die "Couldn't delete $cache_filename" if -e $cache_filename;

            1;
        } catch {
            $error_count++;
            $log_func->("error handling $cache_filename: $_");
        };
    }

    $log_func->('finish');

    if ( $error_count > 0 ) {
        die "$error_count errors occured during cache cleanup\n";
    }
}

1;
