#!/usr/bin/perl -w

# $Id$

=head1 NAME

    beta-ports.pl -- cli-интерфейс к BetaPorts.pm

    Предстоят доработки, параметры могут меняться

=head1 DESCRIPTION

    Терминология: 
    
    зарезервированные (reserved) порты -- указанные в %BetaPorts::RESERVED, 
    то есть приписанные к определенному логину или функциональности.
    Соответствующая бета может еще не быть создана

    доступные порты (available) -- все порты, отведенные под беты, 
    но не зарезервированные ни за кем

    занятые (used) порты -- порты, для которых созданы беты

    свободные (free) порты -- не занятые, на них можно создавать новые беты (в пределах своих зарезервированных диапазонов)

    Опции:

    -h, --help
    справка

    -a, --available
    показать свободные диапазоны портов, с -n -- только диапазоны не меньше указанной длины

    -f, --for-user
    для аллокации/освобождения: для какого пользователя выполняем действие

    -n <number>
    длина диапазона
    в режиме отчета будут показываться свободные диапазоны длины не меньше, чем этот параметр
    в режиме аллокации/освобождения будет обрабатываться диапазон указанной длинны с началом из параметра -s

    -s, --start-port 
    стартовый порт диапазона для аллокации/освобождения, использовать вместе с -n
    для аллокации/освобождения одного порта можно пользоваться -p

    -p, --port
    порт для точечной аллокации/освобождения
    для аллокации/освобождения диапазона есть -s -n 

    -t, --target-host <hostname>
    УСТАРЕЛО, не поддерживается. 
    Если хочется посмотреть про другой сервер, можно ssh другой-сервер beta-ports.pl
    Было:
    резерв на каком хосте (сервере) просматривать, по умолчанию -- текущий хост
    если имя не имеет суффикса .yandex.ru, то он будет добавлен

    Примеры: 
    
    свободные порты, не менее 10 подряд, на текущем сервере:
    beta-ports.pl -a -n 10

    Таким образом можно выбрать диапазон для себя или для коллеги. 

    Зарезеривировать порты:
    для себя, дапапзон
    beta-ports.pl alloc -s 14035 -n 5
    для себя, точечно
    beta-ports.pl alloc -p 14050
    для другого пользователя, диапазон
    beta-ports.pl alloc -s 14035 -n 5 -f lena-san
    для другого пользователя, точечно
    beta-ports.pl alloc -p 14050 -f lena-san

    Освободить:
    свои, диапазон
    beta-ports.pl free -s 14035 -n 5
    свой, точечно
    beta-ports.pl free -p 14050
    другого пользователя, диапазон
    beta-ports.pl free -s 14035 -n 5 -f lena-san
    другого пользователя, точечно
    beta-ports.pl free -p 14050 -f lena-san

=head1 CODE

=cut

use strict;
use Getopt::Long;
use Data::Dumper;
use List::Util qw/sum/;
use Yandex::Hostname;

use BetaPorts;

use utf8;
use open ':std' => ':utf8';

my %O;

run() unless caller();

sub run
{
    parse_options();
    my $exit_code = 0;
    if (      $O{cmd} eq 'report' ){
        cmd_report();
    } elsif ( $O{cmd} eq 'alloc' ) {
        my $success_cnt = cmd_reserve();
        $exit_code = 1 if $success_cnt == 0;
    } elsif ( $O{cmd} eq 'free' ) {
        my $success_cnt = cmd_release();
        $exit_code = 1 if $success_cnt == 0;
    } else {
        die "unknown command $O{cmd}, stop";
    }

    exit $exit_code;
}

# разбирает параметры и складывает в глобальный %O
sub parse_options
{
    GetOptions(
        "h|help" => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0;},
        "a|show-available" => \$O{show_available},
        "r|show-reserved" => \$O{show_reserved},
        "n=i" => \$O{N},
        "s|start=i" => \$O{port_start},
        "p|port=i" => \$O{port},
        "f|for-user=s" => \$O{for_user},
        "t|target-host=s" => \$O{target_host},
    ) || die "can't parse options";

    if ( $O{target_host} ){
        die "--target-host not supported anymore, try this: ssh $O{target_host} beta-ports.pl";
    }
    if ( scalar @ARGV == 1 ){
        $O{cmd} = shift @ARGV;
    } elsif ( scalar @ARGV == 0 ) {
        $O{cmd} = 'report';
    } else {
        die "too many commands, stop\n";
    }

    if ( !$O{for_user} ){
        $O{for_user} = getlogin || getpwuid($<) || die "can't get current login";
    }
}

sub cmd_report 
{
    my $current_host = BetaPorts::current_host();

    my @base_options = qw/show_available show_reserved/;
    if (!grep {$O{$_}} @base_options){
        $O{$_} = 1 for @base_options;
    }

    my $betas = BetaPorts::get_betas_info();
    #print Dumper($betas);

    my $title = "current host: $current_host";
    print $title."\n";
    print (("=" x length($title)) . "\n\n");

    if ( $O{show_reserved} ){
        print "reserved ports\n";
        print "==============\n";
        for my $o (@{$betas->{owners}}){
            my $count = sum( 0, map {$_->{length}} 
                @{$betas->{ranges_by_owners}->{$o}}
            ); 
            next if $O{N} && $count < $O{N};
            print join "", (
                $o, 
                " - ",
                join(", ", 
                    map { BetaPorts::range_to_str($_)} 
                    @{$betas->{ranges_by_owners}->{$o}}
                ), 
                " ($count)\n",
            );
        }
        print "\n";
    }

    if ($O{show_available}){
        my $title = "available (unreserved) ports";
        $title .= $O{N} ? ", ranges >= $O{N}" : "";
        print "$title\n";
        print "=" x length($title) . "\n";
        print join "", map { BetaPorts::range_to_str($_, count => 1)."\n" }grep {$O{N} ? $_->{end} - $_->{start} + 1 >= $O{N} : 1} @{$betas->{unreserved_ranges}};
        print "\n";
    }
}


sub cmd_reserve
{
    my $arr = make_ports_arr();

    die "error number 49405, stop" if @$arr > 10 || @$arr <= 0;
    my $res = BetaPorts::reserve_ports( $arr, for => $O{for_user} );

    print "Successfully reserved: ".(join(", ", @{$res->{success}}) || '-')."\n";
    if (keys $res->{errors}){
        # здесь можно покрасивее, с номерами портов и отсортировать по ним
        print "Errors:\n".join("\n", map {" - $_"} values %{$res->{errors}});
        print "\n";
    }

    return scalar @{$res->{success}}
}



sub cmd_release
{
    my $arr = make_ports_arr();
    die "error number 49406, stop" if @$arr <= 0;
    my $res = BetaPorts::release_ports( $arr, for => $O{for_user} );

    print "Successfully cleared: ".(join(", ", @{$res->{success}}) || '-')."\n";
    if (keys $res->{errors}){
        # здесь можно покрасивее, с номерами портов и отсортировать по ним
        print "Errors:\n".join("\n", map {" - $_"} values %{$res->{errors}});
        print "\n";
    }

    return scalar @{$res->{success}}
}


=head2 make_ports_arr

    из параметров собирает массив портов, которые пользователь хочет резервировать/освобождать

=cut
sub make_ports_arr
{
    my @arr;
    if ( $O{port} ) {
        die "incompatible options -p and -s; use -p <...> xor -s <...> -n <...>" if $O{port_start};
        die "incompatible options -p and -n; use -p <...> xor -s <...> -n <...>" if $O{N};
        @arr = ( $O{port}  );
    } else {
        die "illegal beginning of port range, stop" if !$O{port_start}; 
        die "illegal port range length, stop" if !$O{N}; 
        die "it's not recommended to reserve more than 5 ports at once, stop" if $O{N} > 5; 
        @arr = ( $O{port_start} .. $O{port_start} + $O{N} - 1 );
    }

    return \@arr;
}


