package Beta::Service;

=head1 INFO

Базовый класс, описывающий "сервис" - прогамму, которая должна работать на бете
Любой сервис должен уметь:

start()
    запуститься

stop()
    остановиться. Здесь же нужно, по необходимости, позвать wait_until_stop()

started()
    вернуть 0 или 1 в зависимости от состояния

show()
    вернуть строку с кратким описанием сервиса (имя команды)


Опционально сервис может реализовать:

restart()
    по-умолчанию вызывается stop() и start()

status()
    вывести на экран текущее состояние сервиса. По умолчанию
    берется значение started()

wait_until_stop()
    Подождать, пока сервис не остановится


Новый класс имеет смысл заводить, если обнаружилось несколько
скриптов, способ запуска/остановки которых отличается только
параметрами.

=cut

use Mouse;
use feature 'state';

use YAML;
use Module::Load qw//;
use List::Util qw/max/;
use Cwd;
use Beta::Service::group;
use Aspect qw//;

our $CONFIG = "";

has name => (
    is => 'ro',
    isa => 'Str',
    required => 1,
);
has ROOT => (
    is => 'ro',
    isa => 'Str',
    required => 1,
);
has PORT => (
    is => 'ro',
    isa => 'Int',
    required => 1,
);
has type => (
    is => 'ro',
    isa => 'Str',
    required => 1,
);
has hidden => (
    is => 'ro',
    isa => 'Bool',
    default => 0,
);

has disabled => (
    is => 'rw',
    isa => 'Bool',
    default => 0,
);

sub BUILDARGS
{
    my ($self, $args) = @_;
    state $CWD = Cwd::cwd();
    ($args->{ROOT}) = ($CWD =~ m!(/var/www/[^/]+)!);
    $args->{ROOT} //= '';
    ($args->{PORT}) = ($args->{ROOT} =~ m!/var/www/.+?(\d+)$!);
    $args->{PORT} //= 0;
    return $args;
}

our %SERVICES = ();
my $PAD_LEN = 0;

## "static" methods

sub load
{
    my ($config) = @_;
    $config //= $CONFIG;
    my $services = YAML::LoadFile($config);

    my @ALL = ();
    for my $name (keys %$services) {
        my $desc = $services->{$name};
        $PAD_LEN = max($PAD_LEN, length($name));
        $desc->{name} = $name;
        my $class = "Beta::Service::$desc->{type}";
        Module::Load::load $class;
        my $service = $class->new($desc);
        _expand_vars($desc);
        $SERVICES{$name} = $service;
        push @ALL, $name;
    }
    @ALL = sort @ALL;
    $SERVICES{ALL} = Beta::Service::group->new({
        name => 'ALL',
        type => 'group',
        hidden => 1,
        childs => \@ALL,
    });

    # обрабатыаем disabled
    Aspect::around {
        my $self = ($_->args)[0];
        return if $self->disabled;
        $_->proceed;
    } Aspect::call qr/Beta::Service::.*(start|stop)/;
}

# рекурсивно обойти переменную и заменить в ней все вхождения $NAME (в строках) на $var->{NAME}
# костыль, чтобы в service.yaml можно было использовать $PORT, $ROOT etc.
sub _expand_vars
{
    my $var = shift;
    my $parent = shift || $var;
    if (ref $var eq 'HASH' || (ref $var) =~ /Beta::Service/) {
        for my $k (keys %$var) {
            $var->{$k} = _expand_vars($var->{$k}, $parent);
        }
        return $var;
    }
    elsif (ref $var eq 'ARRAY') {
        return [ map { _expand_vars($_, $parent) } @$var ];
    }
    elsif (ref $var eq '') {
        $var =~ s!\$(\w+)!exists $parent->{$1} ? $parent->{$1} : die "unknown var $1"!eg;
        return $var;
    }
    else {
        die "dont know how to expand ".(ref $var);
    }
}

sub service
{
    my ($args) = shift;
    if (!%SERVICES) {
        load($CONFIG);
    }
    if (@$args == 1 && $args->[0] eq 'LIST') {
        service_list();
        return;
    }
    if (@$args != 2) {
        die "invalid args";
    }

    my ($name, $action) = @$args;
    my $service = $SERVICES{$name};
    unless ($service) {
        die "unknown service '$name'\n";
    }
    unless ($service->can($action)) {
        die "unknown action '$action'\n";
    }
    $service->$action;
}

## class methods

sub log
{
    my $self = shift;
    for (@_) {
        print "$_\n";
    }
}

sub pad_name
{
    my $self = shift;
    return $self->name . (' ' x ($PAD_LEN - length($self->name)));
}

sub service_list
{
    for my $s (sort { $a->name cmp $b->name } values %SERVICES) {
        next if $s->hidden;
        printf "%s : %s [ %s ]%s\n", $s->pad_name, $s->show(), $s->type, 
            ($s->disabled ? " (disabled)" : "");
    }
}

# for service_list
sub show
{
    my $self = shift;
    die "not implemented\n";
}

sub start
{
    my $self = shift;
    die "not implemented\n";
}

sub stop
{
    my $self = shift;
    die "not implemented\n";
}

sub restart
{
    my $self = shift;
    if ($self->started) {
        $self->stop()
    }
    $self->start();
}

sub status
{
    my $self = shift;
    return if $self->hidden;
    printf "%s : %s\n", $self->pad_name, $self->started ? "started" : "not started";
}

sub started
{
    my $self = shift;
    die "not implemented\n";
}

sub wait_until_stop
{
    my ($self) = shift;
    my $start = time;
    while ($self->started) {
        if (time - $start > 120) {
            $self->log("failed to stop ".$self->name);
            return;
        }
        sleep 1;
    }
}

__PACKAGE__->meta->make_immutable();


