package Beta::Service::daemon;


=head1 DESCRIPTION

Скрипт, работающий как демон.

Параметры:
cmd - команда, которую нужно запустить
args - массив строк - параметры скрипта
match - ищем среди процессов совпадение с match (регулярка); Если не указан - пытаемся матчить по имени скрипта
stop_method => 'stop' | 'kill' - как останавливать: --stop или kill $pid
pid_file - пид файл. Если не указан - $ROOT/protected/run/`basename`.pid
readonly_pid_file - Настройка "не писать pid-файл". Если флаг взведён, ожидается, что демон сам запишет свой pid-файл.

=cut

use Mouse;
use File::Slurp;
use Yandex::ListUtils qw/as_array/;

extends 'Beta::Service';

has cmd => (
    is => 'ro',
    isa => 'Str',
    required => 1,
);

has args => (
    is => 'ro',
    isa => 'ArrayRef',
);

has match => (
    is => 'ro',
    isa => 'Str',
);

has pid_file => (
    is => 'rw',
    isa => 'Str',
);

has readonly_pid_file => (
    is => 'rw',
    isa => 'Bool',
);

has stop_method => (
    is => 'ro',
    isa => 'Str',
);

sub BUILDARGS
{
    my ($self, $args) = @_;
    $self->SUPER::BUILDARGS($args);

    my $fn = $args->{cmd} =~ s!.*/!!r;
    if (!$args->{pid_file}) {
        $args->{pid_file} = $args->{ROOT} . "/protected/run/" . $fn . ".pid";
    }
    if (!$args->{match}) {
        $args->{match} = $fn;
    }
    if (!$args->{stop_method}) {
        $args->{stop_method} = 'kill';
    }
    if ($args->{args}) {
        $args->{args} = as_array($args->{args});
    }

    return $args;
}

sub _run
{
    my ($self, $cmd) = @_;
    my $pid = fork();
    unless (defined $pid) {
        die "can not fork: $!";
    }
    if ($pid == 0) {
        # $self->log("starting: $cmd");
        close STDOUT;
        close STDERR;
        close STDIN;
        exec $cmd or die "failed to exec: $!";
    }
    else {
        $self->log("started ".$self->name.", pid = $pid\n");
        unless ( $self->readonly_pid_file ) {
            write_file($self->pid_file, $pid);
        }
    }
}

sub start
{
    my $self = shift;
    if ($self->started) {
        $self->log($self->name." : already started");
        return;
    }
    my $cmd = $self->cmd;
    if ($self->args) {
        $cmd .= join ' ', ('', @{$self->args});
    }
    $self->_run($cmd);
}

sub stop
{
    my $self = shift;
    if (!$self->started) {
        $self->log($self->name." already stopped");
        return;
    }
    if ($self->stop_method eq 'kill') {
        my $pid = $self->pid();
        $self->log("kill $pid");
        if ($pid) {
            kill 15 => $pid;
        }
    }
    elsif ($self->stop_method eq 'stop') {
        my $cmd = $self->cmd;
        if ($self->args) {
            $cmd .= join ' ', (' ', @{$self->args}, '--stop');
        }
        $self->log($cmd);
        system($cmd);
    }
    $self->wait_until_stop();
}

sub pid
{
    my ($self) = shift;
    if (-f $self->pid_file) {
        my $pid = read_file($self->pid_file);
        chomp $pid;
        return $pid;
    }
    return;
}

sub show
{
    my $self = shift;
    my $cmd = $self->cmd;
    if ($self->args) {
        $cmd .= join ' ', ' ', @{$self->args};
    }
    return $cmd;
}

sub started
{
    my $self = shift;
    if (! -f $self->pid_file) {
        return 0;
    }
    my $pid = $self->pid;
    if (kill 0 => $pid) {
        # system("ps axuww | grep $pid");
        my $cmdline = eval { read_file("/proc/$pid/cmdline"); };
        return 0 unless $cmdline;
        $cmdline =~ s!\x00! !g;
        $cmdline =~ s!\s+$!!g;
        $cmdline =~ s!^/usr/bin/perl\s+(\-\w+\s*)*!!;
        my $match = $self->match;
        if ($cmdline =~ /$match/) {
            return 1;
        }
        return 0;
    }
    else {
        return 0;
    }
}

1;
