#!/usr/bin/perl

=head1 NAME

    direct-callchains.pl - анализ цепочек вызовов функций

=head1 SYNOPSIS

    direct-callchains.pl /var/www/beta.zhur.8404 cmd_showCamp
    direct-callchains.pl /var/www/beta.zhur.8404 /var/www/beta.zhur.8404/protected/bsExportMaster.pl

=head1 DESCRIPTION    

    Изначально скрипт предназначался для статического анализа perl-кода на предмет шардинга.

    См. также direct-callchains-to-freemind.pl

=cut 

use warnings;
use strict;

use Carp qw/croak/;
use Getopt::Long;
use JSON;
use File::Slurp;
use Digest::SHA2;
use Data::Dumper;
use DDP;

use PPI;
#use PPI::Cache path => "$ENV{HOME}/tmp/ppi_cache";

use Yandex::Svn;
use Yandex::HashUtils;

$|++;

my $CACHE = 1;
my $DEBUG = 0;
my $CACHE_DIR = "/var/cache/ppc/ppi.pl";

GetOptions(
    "halp" => \&usage,
    "cache!" => \$CACHE,
    "debug!" => \$DEBUG,
    ) || die "can't parse options: $!";

my $dir = shift;
usage("Incorrect dir") if !$dir || !-d $dir;
my @items = @ARGV;
usage("No subroutines or files defined") if !@items;

my $subs_info = process_modules($dir);

my @funcs;
for my $item (@items) {
    if ($item =~ /^((?:\w+::)*)(\w+)$/) {
        croak "Can't find function $2" if !$subs_info->{$2};
        push @funcs, $item;
    } else {
        croak "File not exists: $item" if !-f $item;
        my $doc = PPI::Document->new($item) || croak "Can't parse $item";
        my $info = get_ppi_calls($doc);
        print "$item".($info->{is_bad} ? ":BAD" : ""), "\n";
        push @funcs, @{$info->{calls}};
    }
}

for my $func (@funcs) {
    my $chains = get_calls_chains($subs_info, $func);
    print_chains($chains);
}



sub print_chains {
    my ($chains) = @_;
    for my $chain (@$chains) {
        print join(" -> ", map {"$_"} @$chain)."\n";
    }
}

{
    my %CHAIN_CACHE;
sub get_calls_chains {
    my ($subs_info, $full_name, $was) = @_;
    if (!exists $CHAIN_CACHE{$full_name}) {
        my ($pack, $name) = $full_name =~ /^(.*)::(.*)$/ ? ($1, $2) : (undef, $full_name);

        return [] if !exists $subs_info->{$name} || $name =~ /^(new|init|load|save|add|delete)$/;


        $was ||= {};
        if (exists $was->{$name}) {
            return [["CYCLE:$name"]];
        }
        local $was->{$name} = undef;

        my $self_pref = $subs_info->{$name} && $subs_info->{$name}->{is_bad} ? "BAD:" : "";
        my @chains = ();
        my @variants = @{$subs_info->{$name}->{variants}};
        for my $variant (@variants) {
            next if $pack && ($variant->{pack}||'') ne $pack;
            my $desc_name = ($variant->{pack} ? "$variant->{pack}".($pack || @variants == 1 ? '' : '?')."::" : '').$name.($variant->{is_bad} ? ":BAD" : "");
            push @chains, [$desc_name];
            for my $call (@{$variant->{calls}}) {
                for my $chain (@{get_calls_chains($subs_info, $call, $was)}) {
                    push @chains, [$desc_name, @$chain];
                }
            }
        }
        $CHAIN_CACHE{$full_name} = \@chains;
    }
    return $CHAIN_CACHE{$full_name};
}
}

sub process_modules {
    my $dir = shift;
    my %info = ();
    my @files = svn_files($dir);
    for my $file (grep {/\.pm$/} @files) {
        my $pf = process_file($file);
        for my $sub_name (keys %{$pf->{subs}}) {
            push @{$info{$sub_name}->{variants}}, $pf->{subs}->{$sub_name};
        }
    }
    for my $sub_name (keys %info) {
        $info{$sub_name}{is_bad} = grep {$_->{is_bad}} @{$info{$sub_name}->{variants}};
    }
    return \%info;
}

{
sub hash_func {
    my $sha2obj = new Digest::SHA2;
    $sha2obj->add(@_);
    return $sha2obj->hexdigest();
}
my $my_hash;
sub get_cache_file {
    my $file = shift;
    $my_hash ||= hash_func(read_file($0));
    return "$CACHE_DIR/".hash_func($my_hash, read_file($file));
}
}

sub process_file {
    my $file = shift;

    print STDERR "$file\n" if $DEBUG;

    my $cache_file = get_cache_file($file);
    if ($CACHE && -f $cache_file) {
        return from_json(scalar read_file($cache_file));
    }
    
    my $doc = PPI::Document->new($file) || croak "Can't parse $file";

    my $pack;
    my %sr2pack;
    my $subs = $doc->find( sub { 
        if ($_[1]->isa('PPI::Statement::Package')) {
            $pack = $_[1]->namespace;
        }
        if ($_[1]->isa('PPI::Statement::Sub') and $_[1]->name) {
            $sr2pack{$_[1]} = $pack;
            return 1;
        } else {
            return 0;
        } }) || [];

    my %ret = (
        file => $file,
        subs => {},
        );
    for my $s (@$subs) {
        $ret{subs}{$s->name} = process_sub($s, $sr2pack{$s});
    }

    if (-d $CACHE_DIR) {
        write_file($cache_file, {atomic => 1}, to_json(\%ret));
    }
    return \%ret;
}

sub process_sub {
    my ($s, $pack) = @_;
    my %ret = (
        name => $s->name,
        pack => $pack,
        );
    hash_merge \%ret, get_ppi_calls($s);
    return \%ret;
}


sub get_ppi_calls {
    my ($ppi) = @_;
    my %ret = (
        is_bad => 0,
        calls => [],
        );
    my $tokens = $ppi->find( sub { $_[1]->isa('PPI::Token') } ) || [];
    my %calls;
    for my $token (@$tokens) {
        if ($token->isa('PPI::Token::Symbol') && $token->content =~ /^\$.*dbh/) {
            $ret{is_bad} = 1;
        } elsif ($token->isa('PPI::Token::Word')) {
            my $snext = $token->snext_sibling;
            if ($token->content =~ /^PPC(_.*)?$/ && $snext) {
                my $is_good_ppc = $snext->isa('PPI::Structure::List') && $snext->content !~ /^\(\s*\)$/;
                $ret{is_bad} = 1 if !$is_good_ppc;
            } elsif ($snext) {
                if ($snext->isa('PPI::Structure::List')
                    && $token->content !~ /^(my|while|for|foreach)$/
                ) {
                    $calls{$token->content} = 1;
                }
            }
        }
    }
    $ret{calls} = [sort keys %calls];
    return \%ret;
}

sub usage {
    print @_, "\n" if @_;
    system("podselect -section NAME -section SYNOPSIS $0 | pod2text-utf8 >&2");
    exit @_ ? 1 : 0;
}
