#!/usr/bin/env python
# *-* encoding: utf-8 *-*

import OpenSSL
import json
import logging
import os
import socket
import smtplib
import ssl
import time
import telnetlib
import yaml
import urllib
import unicodedata
from datetime import datetime
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from optparse import OptionParser

'''  Программа для проверки сертификатов сервисов. По-умолчанию проверяет только
    машины из списка LIST_SERVERS. При указании флага --juggler, -j добавляет к нему машины
    из проверок juggler, полученных c service=https_cert. 
     Ничего не выводит, если проблем с сертификатами не наблюдается. При желании получить весь 
    вывод, можно добавить флаг -v или --verbose.
     Для отладки программы используется флаг --debug, -d.
     Результат можно получить письмом, указав в --mail-list, -m списки email адресов. В этом
    случае ничего выводиться не будет.
'''
    
CRIT_DAYS = 14 #критическое время для напоминания о сертификате в днях
LIST_SERVERS = [ "direct.yandex.ru", 
                 "api.direct.yandex.ru",
                 "direct-mod.yandex.ru",
                 "intapi.direct.yandex.ru",
                 "gorynych.yandex.ru",
                 "partner.yandex.ru",
                 "partner2.yandex.ru",
                 "soap-sandbox.direct.yandex.ru",
                 "cert.partner.yandex.ru",
                 "catmedia.yandex.ru",
                 "advq.yandex.ru",
                 "ppcgraphite.yandex.ru",
                 "intapi.terminal.yandex.ru",
                 "ipv6.api.direct.yandex.ru",
                 "overapi.yandex.ru",
                 "ppcautoadmin.yandex.net",
                 "terminal.yandex.ru"
]
CATYPES = { "external": "Yandex CA",
            "internal": "YandexInternalCA"
}
VHOSTPATH = "/etc/yandex-direct/direct-vhosts.yaml"

def getLogSetup(log_level):
    ''' Функция для задания уровня логирования. Принимает параметр INFO, WARNING, CRITICAL.
        В выводе возвращается интерфейс для логирования.
    '''
    logger = logging.getLogger('steam logs to console')
    logger.setLevel(level=getattr(logging, log_level))
    # create console handler and set level to LEVEL
    ch = logging.StreamHandler()
    ch.setLevel(level=getattr(logging, log_level))
    # create formatter
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    # add formatter to ch
    ch.setFormatter(formatter)
    # add ch to logger
    logger.addHandler(ch)
    return logger

def parseTime(str_time):
    ''' Функция для перевода даты из строкового представления в дату. 
        Например: 20170918172100Z -> date(years=2017, month=09, day=18, h=17, m=21, s=00).
    '''
    str_time = str_time.replace('Z', 'GMT')
    date = datetime.strptime(str_time, '%Y%m%d%H%M%S%Z')
    return date

def convTimestamp(date):
    ''' Преобразует дату в timestamp.
    '''
    return time.mktime(date.timetuple())

def getCertificate(sock, ssl_version=ssl.PROTOCOL_TLSv1):
    ''' Функция для получения сертификата. Пробует сначала скачать по ipv6, потом по ip4. 
        Если ответ ответ от машин не получен - возвращает None. Ранее использовался метод 
        ssl.get_server_certificate, но на старых версиях он не работает.
    '''
    def openCert(ctx, sock_type):
        s = socket.socket(sock_type, socket.SOCK_STREAM)
        s.connect(sock)
        cnx = OpenSSL.SSL.Connection(ctx, s)
        cnx.set_tlsext_host_name(sock[0])
        cnx.set_connect_state()
        cnx.do_handshake()
        raw = cnx.get_peer_certificate()
        s.close()
        return raw
    ctx = OpenSSL.SSL.Context(ssl_version)
    x509 = None
    try:
        sock_type = socket.AF_INET6
        x509 = openCert(ctx, sock_type)
    except Exception as err:
        logger.debug("error get {0} certificate on ipv6".format(sock[0]))
    try:
        sock_type = socket.AF_INET
        x509 = openCert(ctx, sock_type)
    except Exception as err:
        logger.debug("error get {0} certificate on ipv4".format(sock[0]))
    return x509

def checkCAName(x509, hostname):
    if not os.path.exists(VHOSTPATH):
        msg = "{0} vhost {1} not exist. Skip checkCAname".format(hostname, VHOSTPATH)
        return (2, msg)
    with open(VHOSTPATH) as fd:
        raw = fd.read()
    vhostData = yaml.load(raw)

    extData = None
    for num in range(x509.get_extension_count()):
        if x509.get_extension(num).get_short_name().find("subjectAltName") < 0: continue
        extData = x509.get_extension(num).get_data()
    if extData is None:
        msg = "{0} not found subjectAltName in certificate"
        return (2, msg)

    decodeData = unicode(extData, errors='replace')
    listData = ''.join([i for i in decodeData if unicodedata.category(i)[0] != 'C']).split(u'\ufffd')
    listDNS = [ i.lower() for i in listData if i.find(".") > -1 ]

    caType = None
    vhosts = vhostData.get("vhosts", {})
    for name in vhosts:
        if name.lower() not in listDNS: continue
        caType = vhosts[name].get("ca_type", "internal")
    if caType is None:
        msg = "{0} certificate subjectAltNames {1} not found in {2}".format(hostname, listDNS, VHOSTPATH)
        return (2, msg)

    commonName = x509.get_issuer().commonName
    caName = CATYPES.get(caType, None)
    logger.debug("certificate commonName(CN)/{0} commonName(CN): {1}/{2}".format(caType, commonName, caName))

    if commonName.find(caName) < 0:
        msg = "{0} bad commonName(CN) (certificate CN = {1}, needing CN = {2})".format(hostname, commonName, caName)
        return (2, msg)
    return (0, "")

def checkCert(hostname, port=443):
    ''' Функция проверяет сертификат. Принимает DNS имя сервиса и порт(по-умолчанию 443).
        Перед проверкой сертификата, смотрится наличие DNS записи и доступность сервиса.
        На выходе возвращается кортеж с кодом выполнения(ошибки) и сообщение.
        Например: (0, "certificate was good") или (2, "certificate expired").
    '''
    sock = (hostname, port)
    ct_time = time.time()
    logger.debug("socket {0}".format(sock))
    try:
        telnetlib.Telnet(hostname, port, timeout=2)
    except Exception as err:
        return (2, "{0} error connection: {1}".format(hostname, err))
    x509 = getCertificate(sock)
    if not x509:
        return (2, "{0} error get or read certificate: {1}".format(hostname, err))
    if opts.checkCA:
        result = checkCAName(x509, hostname)
        if result[0] !=0:
            return result
    str_after, str_before = (x509.get_notAfter(), x509.get_notBefore())
    logger.debug("string time after {0}, string time before {1}". format(str_after, str_before))
    date_after, date_before = (parseTime(str_after), parseTime(str_before))
    logger.debug("parse time after {0}, parse time before {1}". format(date_after, date_before))
    ts_after, ts_before = (convTimestamp(date_after), convTimestamp(date_before))
    if ts_before > ct_time:
        msg = "{0} was younger than current time (certificate time = {1}, current time {2})".format(
                hostname, date_before, datetime.fromtimestamp(ct_time))
        return (2, msg)
    crit_time = CRIT_DAYS*3600*24
    delta_time = ts_after - ct_time
    if crit_time > delta_time:
        msg = "{0} was expire after {1} hours (certificate time = {2}, current time {3})".format(
                hostname, delta_time/3600, date_before, datetime.fromtimestamp(ct_time))
        return (2, msg)
    if x509.has_expired():
        msg = "{0} was expired!! (certificate time = {2}, current time {3})".format(
                hostname, delta_time/3600, date_before, datetime.fromtimestamp(ct_time))
        return (2, msg)
    msg = "{0} was good (certificate time = {1}, current time {2})".format(
            hostname, date_before, datetime.fromtimestamp(ct_time))
    return (0, msg)

def checkDNS(host):
    ''' Принимает DNS запись и проверяет резолв адреса по IPv6 и IPv4 адресам.
    '''
    try:
        dns6 = socket.getaddrinfo(host, None, socket.AF_INET6)
        if dns6: return True
    except Exception as err:
        pass
    try:
        dns4 = socket.getaddrinfo(host, None, socket.AF_INET)
        if dns4: return True
    except Exception as err:
        logger.warning("bad response {0}: {1}".format(host, err))
    return False

def getJugglerHosts(namespaces):
    ''' Ходит в Juggler и получает список проверок с service=https_cert.
    '''
    url = 'http://juggler-api.search.yandex.net:8998/api/checks/checks'
    json_data = dict()
    for namespace in namespaces:
        parms = urllib.urlencode({'service_name': 'https_cert',
                                  'namespace_name': namespace,
                                  'do': 1
                                 })
        url = "{0}?{1}".format('http://juggler-api.search.yandex.net:8998/api/checks/checks', parms)
        try:
           raw = urllib.urlopen(url).read()
           json_data.update(json.loads(raw))
        except Exception as err:
           logger.info("func getJugglerHosts for namespace {1} error {0}".format(err, namespace))
    return [ i for i in json_data.keys() if checkDNS(i) ]

def sendMail(result, mail_list):
    ''' Приимает результат проверок сертификатов и список email адресов для
        отправки. Например: ("все очень плохо у direct.yandex.ru", ["ppc@ya.ru", "direct@ya.ru"].
    '''
    prefix = "В ходе проверки сертификатов обнаружены следующие проблемы:"
    plain_msg = "{0}\n\n{1}".format(prefix, '\n'.join(result))
    html_msg = "<html><head></head><body><p><i>{0}</i><br><br>{1}</p></body></html>".format(prefix, '<br>'.join(result))
    msg = MIMEMultipart('alternative')
    msg.attach(MIMEText(plain_msg, 'plain'))
    msg.attach(MIMEText(html_msg, 'html'))
    msg['Subject'] = "Certificate Checker Services(CCS)"
    msg['From'] = "root"
    msg['To'] = mail_list[0]
    if len(mail_list)>1: 
        msg['Cc'] = ','.join(mail_list[1:])
    s = smtplib.SMTP('localhost')
    s.sendmail("root", mail_list, msg.as_string())

def runSendOrPrint(result):
    ''' Функция принимает кортеж результатов проверок сертификатов, агрегирует сообщения
        и в зависимости от наличия установленного флага --mail-list выполняет либо отправку письма, либо
        печать сообщения. Например: result=[(0, "certificate was good"), (2, "certificate expired")]
        преобразуется в "certificate expired\n certificate was good\n" при выставлении флага --verbose.
    '''
    bad_msgs = [ i[1] for i in result if i[0]>0 ]
    good_msgs = [ i[1] for i in result if i[0]==0 ]
    result_msgs = bad_msgs + good_msgs if opts.verbose else bad_msgs
    if opts.monrun:
        if bad_msgs:
            print "2; {0}".format(','.join(bad_msgs))
        else:
            print "0; {0}".format(','.join(good_msgs))
    if not result_msgs:
        return
    if opts.mails:
        sendMail(result_msgs, opts.mails)
    elif not opts.monrun:
        [ logger.critical(i) for i in bad_msgs ]
        [ logger.info(i) for i in good_msgs ]
    return

def callList(option, opt, value, parser):
    ''' Преобразует строку вида "ppc@ya.ru,direct.ya.ru" в словарь.
    '''
    value = [ os.path.basename(v) for v in value.split(',') ]
    setattr(parser.values, option.dest, value)

if __name__ == '__main__':
    USAGE = "usage: %prog --help"
    VERSION = "1.0"

    parser = OptionParser(usage=USAGE, version=VERSION)
    
    parser.add_option(  "-d", "--debug",
                        action="store_true",
                        dest="debug", help="enable debug mode"
                        )
    parser.add_option(  "-v", "--verbose",
                        action="store_true",
                        dest="verbose", help="enable verbose mode"
                        )
    parser.add_option(  "-j", "--juggler",
                        action="callback", default="",
                        type="string", callback=callList,
                        dest="juggler", help="list namespaces for juggler hosts with service https_cert. \
                                              DONT WORKING with flag --hosts."
                        )
    parser.add_option( "--check-ca",
                        action="store_true",
                        dest="checkCA", help="check certificate CA type"
                        )
    parser.add_option(  "-m", "--mail-list",
                        action="callback", default = "",
                        type="string", callback=callList,
                        dest="mails", help="mail list for send messages"
                        )
    parser.add_option(  "--monrun",
                        action="store_true",
                        dest="monrun", help="print monrun format"
			)
    parser.add_option(  "--hosts",
                        action="callback", default = "",
                        type="string", callback=callList,
                        dest="hosts", help="list services for check"
			)
    (opts, args) = parser.parse_args()

    if opts.debug:
        level = "DEBUG"
    elif opts.verbose:
        level = "INFO"
    else:
        level = "CRITICAL"
    logger = getLogSetup(level)

    filter_hosts = []
    if opts.hosts and opts.juggler:
        logger.warning("flag 'juggler' dont working with flag 'hosts'. Ignore 'juggler'")
    elif opts.juggler:
        filter_hosts = getJugglerHosts(opts.juggler)
        logger.debug(filter_hosts)
    list_servers = opts.hosts if opts.hosts else LIST_SERVERS 
    check_hosts = list(set(filter_hosts + list_servers))

    result = []
    for i in check_hosts:
        result.append(checkCert(i))
    runSendOrPrint(result)
