#!/usr/bin/python
# -*- coding: utf-8 -*-

'''
Сравнивает список прав и пароль пользователя в бд и в конфиге.
Возвращает инструкции mysql для того чтобы права в бд сошлись с конфигом

Пример запуска для просмотра необходимых изменений:
/usr/local/bin/mysql-compare-grants --grants-config direct.yaml --socket /var/run/mysqld.ppcdata0/mysqld.sock --user root --destructive | perl -lne 'if (/^(.*")@"([^"]+)";$/) { $ip = qx(host $2 | grep "pointer"); chomp($ip); $ip = (split(/\s+/, $ip))[-1]; $ip ||= $2; print qq($1\@"$ip";) } else { print }'

Применяем абсолютно все изменения в грантах, вывод сохраняем в лог:
/usr/local/bin/mysql-compare-grants --grants-config direct.yaml --socket /var/run/mysqld.ppcdata0/mysqld.sock --user root --destructive --apply --log /var/log/mysql.ppcdata0/compare-grants.log

Просмотр логов с резолвингом ip:
cat /var/log/mysql.ppcdata0/compare-grants.log | perl -lne 'if (/^(.*")@"([^"]+)";$/) { $ip = qx(host $2 | grep "pointer"); chomp($ip); $ip = (split(/\s+/, $ip))[-1]; $ip ||= $2; print qq($1\@"$ip";) } else { print }' | less
'''

import sys
from socket import getaddrinfo, gethostname, gaierror, getnameinfo, \
                    AF_UNSPEC, SOCK_STREAM, AI_ADDRCONFIG, AI_CANONNAME
from collections import defaultdict, namedtuple
import argparse
from datetime import datetime
import time
import fcntl
import pickle
import os
from stat import *
import subprocess
import yaml
import requests
import jsonschema
import MySQLdb

SCRIPT_NAME = sys.argv[0] #'compare_grants.py'
Grant = namedtuple('Grant', ['object_type', 'priv_type', 'grant_option'])

class MyException(Exception):
    '''
    Класс для того чтобы ловить только свои исключения
    '''
    pass

MY_FQDN = getaddrinfo(gethostname(), None, AF_UNSPEC, SOCK_STREAM, 0, AI_CANONNAME)[0][3]

def resolve_hosts(*host_names, **kwargs):
    '''
    Возвращает список ip для имен хостов
    '''
    ip_set = set()
    for host_name in host_names:
        try:
            flags = AI_ADDRCONFIG if host_name == MY_FQDN else 0
            res = getaddrinfo(host_name, None, AF_UNSPEC, SOCK_STREAM, 0, flags)
        except gaierror as exc:
            raise MyException(kwargs.get('additional_message', '')
                              + 'Cannot resolve {}.'.format(host_name), exc.errno)
        for addr in res:
            ip_set.add(addr[4][0])
    return list(ip_set)


def read_from_cache(file_name, max_cache_age):
    '''
    Если кэш доступен и актуален возвращает его содержимое, иначе None
    '''
    if not os.path.exists(file_name):
        return None
    cache_age = time.time() - os.stat(file_name).st_mtime
    if cache_age > max_cache_age:
        return None
    try:
        response = pickle.load(open(file_name))
    except (OSError, IOError) as exc:
        sys.stderr.write(SCRIPT_NAME
                         + ': Warning. Cannot read from cache file. '
                         + file_name + ': ' + exc.strerror + '\n')
        sys.stderr.flush()
        return None
    except pickle.UnpicklingError:
        sys.stderr.write(SCRIPT_NAME
                         + ': Warning cannot deserialize conductor response from file '
                         + file_name + '\n')
        return None
    return response


def get_host_names_from_conductor(group_name, params):
    '''
    Возвращает список имен хостов в группе
    '''
    # запросить список хостов по группе в кондукторе
    # если кондуктор не доступен то попробуем сходить в кэш
    cache_path = '/var/cache/mysql-compare-grants'
    cache_file_path = os.path.join(cache_path, '{}.dump'.format(group_name))
    try:
        res = requests.get(
            '{}/{}?format=json'.format(params.conductor_url, group_name),
            timeout=5
        )
        # conductor api умеет &format=json
        res.raise_for_status()
        # возвращает список словарей: [{"fqdn":"ppcstandby05e.yandex.ru"},{},...]
    except requests.exceptions.Timeout as exc:
        res = read_from_cache(cache_file_path, params.max_cache_age)
        if res is None:
            raise MyException('Cannot resolve conductor group'
                              + '"{}". Timeout.'.format(group_name) + exc.message, 1)
    except requests.exceptions.ConnectionError as exc:
        res = read_from_cache(cache_file_path, params.max_cache_age)
        if res is None:
            raise MyException('Cannot connect to {}'.format(params.conductor_url), -1)
    except requests.exceptions.RequestException as exc:
        # res = read_from_cache(cache_file_path)
        # if res is None:
        raise MyException(exc.message, 2)
    else:
        if not params.write_no_cache:
            if not os.path.exists(cache_path):
                try:
                    os.makedirs(cache_path)
                except (IOError, OSError) as exc:
                    sys.stderr.write('An error ocured while creating cache dir. '
                                     + cache_path + ': ' + exc.strerror + '\n')
            try:
                pickle.dump(res, open(cache_file_path, 'w'))
            except (IOError, OSError) as exc:
                sys.stderr.write(SCRIPT_NAME
                                 + ': Warning cannot write to cache file: '
                                 + cache_file_path + ' ' + exc.strerror + '\n')
            except pickle.PicklingError:
                sys.stderr.write(SCRIPT_NAME
                                 + ': Warning cannot serialize conductor response for group '
                                 + group_name + '\n')
    # хотим только непустые fqdn (на случай пустого списка и "fqdn": "")
    try:
        res = res.json()
    except ValueError:
        raise MyException('Cannot decode json object from conductor', 41)
    res = [x["fqdn"] for x in res if x.get("fqdn")]
    return res


def resolve_conductor_group(group_name, params):
    '''
    Возвращает список всех ip для группы
    '''
    res = get_host_names_from_conductor(group_name, params)
    return resolve_hosts(*res,
                         additional_message='In resolving conductor group {}: '.format(group_name))


def get_host_names_from_sky(expr):
    '''
    Возвращает имена хостов после вычисления выражения калькулятором Блинова
    '''
    path_to_sky = 'sky'
    try:
        proc = subprocess.Popen([path_to_sky, 'list', expr],
                                stdout=subprocess.PIPE,
                                stderr=sys.stderr)
        exit_code = proc.wait()
    except (IOError, OSError) as exc:
        raise MyException(path_to_sky + ': ' + exc.strerror, exc.errno)
    if exit_code != 0:
        raise MyException('sky exited with non zero-code with expression: {}'.format(expr), 31)
    return [s.strip() for s in proc.stdout.readlines() if s.strip()]


def perform_sky_calc(expr, params=argparse.Namespace()):
    '''
    Возвращает результат вычислений sky.
    '''
    res = get_host_names_from_sky(expr)
    return resolve_hosts(*res,
                         additional_message="In resolving Blinov's calc expression '{}' : ")


def check_config(config):
    '''
    Проверяет соответствие конфига схеме и в случае ошибки кидает исключение
    '''
    schema = {
        'type': 'object',
        'additionalProperties': False,
        'required': ['users'],
        'definitions': {
            'ipContainer': {
                'type': 'array',
                'minItems': 1,
                'uniqueItems': True,
                'items': {
                    'type': 'string'
                }
            }
        },
        'properties': {
            'mysql_version': {
                'enum': ['5.5', '5.6', '5.7']
            },
            'users': {
                'type': 'object',
                'additionalProperties': False,
                'patternProperties': {
                    '^.+$': {
                        'type': 'string'
                    },
                }
            },
            'hosts_aliases': {
                'type': 'object',
                'additionalProperties': False,
                'patternProperties': {
                    '^.+$': {
                        'type': 'object',
                        'additionalProperties': False,
                        'properties': {
                            'special': {
                                'type': 'array',
                                'minItems': 1,
                                'uniqueItems': True,
                                'items': {
                                    'enum': ['localhost', '%']
                                }
                            },
                            'ip': {
                                'type': 'array',
                                'minItems': 1,
                                'uniqueItems': True,
                                'items': {
                                    'type': 'string',
                                    'pattern': r'^(\d|[a-f]|[.:])+$'
                                }
                            },
                            'hosts': {
                                '$ref': '#/definitions/ipContainer'
                            },
                            'conductor_groups': {
                                '$ref': '#/definitions/ipContainer'
                            },
                            'sky_list': {
                                '$ref': '#/definitions/ipContainer'
                            },
                        },
                    },
                },
            },
            'grants': {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'required': ['user',
                                 'host',
                                 'object_type',
                                 'priv_level',
                                 'priv_type',
                                 'grant_option'],
                    'additionalProperties': False,
                    'properties': {
                        'user': {
                            'type': 'string',
                        },
                        'host': {
                            'type': 'string'
                        },
                        'object_type': {
                            'enum': ['TABLE', 'FUNCTION', 'PROCEDURE']
                        },
                        'priv_level': {
                            'type': 'array',
                            'items': {
                                'type': 'string'
                            }
                        },
                        'priv_type': {
                            'type': 'array',
                            'items': {
                                'type': 'string'
                            }
                        },
                        'grant_option': {
                            'type': 'boolean'
                        }
                    }
                }
            }
        }
    }

    try:
        jsonschema.validate(config, schema)
    except jsonschema.exceptions.ValidationError as exc:
        raise MyException('In configuration file: ' + exc.message
                          + ' in [' + ']['.join(map(str, exc.path)) + ']. '
                          + 'Value is ' + str(exc.instance), 39)
    except TypeError as exc:
        if str(exc) == 'expected string or buffer':
            raise MyException(str(exc), 40)
        raise exc


def parse_config(config_path, is_need_mysql_version):
    '''
    Парсит конфиг и заодно проверяет корректность данных. Возвращает:
    1) Пользователи (user->passwd)
    2) Описание алиасов
    3) Словарь с грантами в конфиге (user, ip) -> {(priv_level) -> Grant(...)}
    4) Версия mysql из конфига (если она не требуется, то None)
    '''
    # get usernames and passwords

    try:
        config = yaml.load(open(config_path), Loader=yaml.Loader)
    except (IOError, OSError) as exc:
        raise MyException(config_path + ': ' + exc.strerror, exc.errno)
    except yaml.YAMLError, exc:
        raise MyException('Error in confuguration file {}: '.format(config_path)
                          + str(exc), 1)

    check_config(config)
    # Словарь словарей (user, host_alias): {priv_level1: Grant(...), priv_level2: Grant(...)]
    grant_requests = defaultdict(dict)

    grant_req_users = set()
    for grant_record in config['grants']:
        user = grant_record['user']
        grant_req_users.add(user)
        alias = grant_record['host']
        if user not in config['users']:
            raise MyException(user + ' not found in section users.', 32)
        if alias not in config['hosts_aliases']:
            raise MyException(alias + ' not found in section hosts_aliases.', 26)
        for lvl in [s.strip() for s in grant_record['priv_level']]:
            if lvl in grant_requests[(user, alias)]:
                raise MyException('priv_level ' +  lvl
                                  + ' occurs twice for "{}"@"{}"'.format(user, alias), -4)
            if grant_record['priv_type'] or grant_record['grant_option']:
                # записываем только если есть гранты
                grant_requests[(user, alias)][lvl] = Grant(
                    grant_record['object_type'],
                    grant_record['priv_type'],
                    grant_record['grant_option']
                )
    # считаем, что удалять только юзера, или только все его гранты не хотим
    missed_grant_req_users = set(config['users']) - grant_req_users
    missed_users = grant_req_users - set(config['users'])

    if missed_grant_req_users:
        raise MyException('Users {} in users section'.format(', '.join(missed_grant_req_users))
                          + ', but not in grants section.', 4)

    if missed_users:
        raise MyException('Users {} in grants section'.format(', '.join(missed_users))
                          + ', but not in users section.', 4)

    if is_need_mysql_version and config.get('mysql_version') is None:
        raise MyException('In configuration file: Not found mysql version. '
                          + 'Use --ignore-version to ignore mysql_version in config.', 19)
    return config['users'], config['hosts_aliases'], grant_requests, config.get('mysql_version')


def resolve_aliases(hosts_aliases, params):
    '''
    Получает словарь с описанием алиасов как в конфиге. И namespace с параметрами.
    Возвращает два словаря: alias -> list(ip) и ip -> alias
    '''
    for key in 'log', 'conductor_url', 'write_no_cache', 'max_cache_age', 'ignore_empty_groups':
        if key not in params:
            raise MyException('In resolve aliases: '
                              + '{} not found in input params'.format(key), 33)
    hosts = {}
    ip_to_alias = {}
    def resolving_dec(func):
        def df(*args, **kwargs):
            if params.log is not None:
                sys.stderr.write('[{date}] {action} [{alias}, {args}]: '.format(
                    date=datetime.now(),
                    action=func.__name__,
                    alias=alias,
                    args=', '.join(args)))
            result = func(*args, **kwargs)
            if params.log is not None:
                sys.stderr.write(str(result) + '\n')
            if not result and not params.ignore_empty_groups:
                raise MyException(key + ' is empty', 37)
            return result
        return df

    for alias, content in hosts_aliases.items():
        if params.log is not None:
            sys.stderr.write('[{date}] Start resolving alias {alias}\n'.format(
                date=datetime.now(),
                alias=alias))
        try:
            if content.get('ip') is None:
                content['ip'] = []
            content['ip'] += content.get('special', [])
            func_dict = {'conductor_groups': resolve_conductor_group,
                         'hosts': resolve_hosts,
                         'sky_list': perform_sky_calc}
            for key, func in func_dict.items():
                for value in content.get(key, []):
                    content['ip'] += resolving_dec(func)(value, params=params)
        except MyException as exc:
            message, errno = exc.args
            raise MyException('In alias {} : {}'.format(alias, message), errno)
        if not content['ip']:
            raise MyException('Alias {} is empty'.format(alias), 38)
        hosts[alias] = set(content['ip'])
        if params.log is not None:
            sys.stderr.write('[{date}] '.format(date=datetime.now())
                             + 'Resolving alias {alias} finished\n'.format(alias=alias)
                            )
        for ip in hosts[alias]:
            if ip in ip_to_alias:
                raise MyException(ip + ' in two aliases: '
                                  + '{old} and {new}'.format(old=ip_to_alias[ip], new=alias), 5)
            ip_to_alias[ip] = alias
    return hosts, ip_to_alias


def parse_grant_message(message):
    '''
    Парсит строку выданную mysql'ем. Возвращает на что выданы гранты, и Grant(...)
    '''
    def parse_mysql_priv_type(message):
        '''
        нельзя просто разбить строку по ", " т. к. могут встретиться гранты на отдельные столбцы.
        Например: SELECT (c1, c2), INSERT
        Считаем кол-во открытых закрытых скобок и если мы не в них то тогда уже бьем по ","
        Возвращает список грантов
        '''
        tokens = []
        balance = 0
        cur = ''
        for symbol in message:
            if symbol == ',':
                if balance == 0:
                    tokens.append(cur)
                    cur = ''
                else:
                    cur += symbol
                continue
            cur += symbol
            if symbol == '(':
                balance += 1
            elif symbol == ')':
                balance -= 1
        if cur:
            tokens.append(cur)
        return [t.strip() for t in tokens]

    strip_split = lambda s, sep: [x.strip() for x in s.split(sep)]
    head, tail = message.strip().split(' ON ', 1)
    priv_type = parse_mysql_priv_type(head[len('GRANT'):])
    tail = strip_split(tail, ' TO ')
    mid = strip_split(tail[0], ' ')
    object_type = 'TABLE'
    if len(mid) > 2:
        # it's strange
        raise MyException('bad grant message from mysql "{}"'.format(message), 6)
    elif len(mid) == 2:
        object_type = mid[0]
    elif len(mid) == 1:
        pass
    else:
        # it's very strange
        raise MyException('bad grant message from mysql "{}"'.format(message), 6)
    lvl = mid[-1].replace('`', '')
    grant_option = tail[1].endswith('WITH GRANT OPTION')
    if priv_type[0] == 'USAGE':
        # usage синоним отсутствия прав
        priv_type = []
    elif priv_type[0] == 'PROXY':
        # на прокси пока забили
        return None
    elif priv_type[0] == 'ALL PRIVILEGES':
        # при ипользовании Grant пишется просто ALL. Зачем в вывод добавили лишнее слово?...
        priv_type = ['ALL']
    if not priv_type and not grant_option:
        #прав нет значит нет смысла записывать
        return None
    return lvl, Grant(object_type, priv_type, grant_option)


def get_user_grants(user, ip, cursor):
    '''
    Берет список грантов mysql для пользователя и возвращает словарь {priv_level1: Grant(...), ...}
    '''
    host_user_name = '"{user}"@"{ip}"'.format(user=user, ip=ip)
    all_grants = {}
    try:
        cursor.execute('show grants for {name};'.format(name=host_user_name))
    except MySQLdb.InternalError:
        raise MyException('db internal error', 11)
    except MySQLdb.ProgrammingError:
        raise MyException('wrong show grants query', 12)
    for message_tuple in cursor.fetchall():
        for message in message_tuple:
            parsed_message = parse_grant_message(message)
            if parsed_message is not None:
                all_grants[parsed_message[0]] = parsed_message[1]
                # mysql указывает priv_level только один раз. Стоит ли это проверять?
    return all_grants


def get_user_table(cursor, *columns):
    '''
    Возвращает таблицу с пользователями из бд. Колонки задаются
    '''
    cursor.execute('SELECT {columns} FROM mysql.user;'.format(columns=', '.join(columns)))
    return {record[:2]: record[2:] for record in cursor.fetchall()}


def get_db_grants(user_table, cursor):
    '''
    Получает гранты для всех пользователей
    '''
    db_grants = defaultdict(dict)
    for user, ip in user_table:
        db_grants[(user, ip)] = get_user_grants(user, ip, cursor)
    return db_grants


def get_mysql_version(cursor):
    '''
    Возвращает версию mysql в формате "major"."minor"
    '''
    cursor.execute('SELECT VERSION();')
    return '.'.join(cursor.fetchall()[0][0].split('.')[:2])


def get_db_grants_state(args, connection_opts, mysql_ver):
    '''
    Возращает текущее состояние бд. гранты, пользователей и версию mysql
    '''
    try:
        with MySQLdb.connect(**connection_opts) as cursor:
            try:
                real_ver = get_mysql_version(cursor)
            except MySQLdb.ProgrammingError:
                raise MyException('Wrong SELECT VERSION query.', 17)
            if real_ver != mysql_ver:
                if args.ignore_version:
                    mysql_ver = real_ver
                else:
                    raise MyException('Mysql version in configuration file '
                                      + "doesn't match real db version.", 16)
            try:
                columns = {'5.5': ['user', 'host', 'password'],
                           '5.6': ['user', 'host', 'password'],
                           '5.7': ['user', 'host', 'authentication_string', 'plugin']}
                user_table = get_user_table(cursor, *columns[mysql_ver])
            except MySQLdb.ProgrammingError:
                raise MyException('Wrong SELECT query.', 13)
            try:
                db_grants = get_db_grants(user_table, cursor)
            except MySQLdb.ProgrammingError:
                raise MyException('Wrong SHOW GRANTS query.', 15)
    except MySQLdb.OperationalError as exc:
        raise MyException(exc.args[1], 14)
    except MySQLdb.InternalError as exc:
        raise MyException('Internal db error. Message: ' + exc.message, 18)
    return db_grants, user_table, mysql_ver


def check_user(user, ip, usr_hash, user_table, mysql_ver):
    '''
    Возвращает инструкцию mysql для создания пользователя или смены пароля
    Если ничего не требуется то None
    '''
    def create_user(user, ip, usr_hash):
        if mysql_ver == '5.5' or mysql_ver == '5.6':
            return 'CREATE USER "{user}"@"{ip}" '.format(user=user, ip=ip) \
                    + 'IDENTIFIED BY PASSWORD "{hash}";'.format(hash=usr_hash)
        elif mysql_ver == '5.7':
            return 'CREATE USER "{user}"@"{ip}" '.format(user=user, ip=ip) \
                    + 'IDENTIFIED WITH mysql_native_password AS "{hash}";'.format(hash=usr_hash)

    # в 5.7 нет password в mysql.user
    # user.authentication_string
    # user.plugin (mysql_native_password)
    if (user, ip) in user_table:
        # check password
        current_hash = user_table[(user, ip)][0]
        if mysql_ver == '5.7':
            if user_table[(user, ip)][1] != 'mysql_native_password':
                # что-то нужно здесь делать. Пока решили просто ругаться и не падать.
                sys.stderr.write(SCRIPT_NAME + ': Warning: "{usr}"@"{ip}" '.format(usr=user, ip=ip)
                                 + 'has unsuppoted plugin "{}"\n'.format(user_table[(user, ip)][1])
                                )
                return None
        if current_hash == usr_hash:
            return None
        return 'SET PASSWORD FOR "{usr}"@"{ip}" = "{hash}";'.format(usr=user, ip=ip, hash=usr_hash)
    else:
        return create_user(user, ip, usr_hash)


def diff(grants_config, grants_mysql, user, ip, destructive):
    '''
    Возвращает список инструкций mysql для приведения грантов конкретного пользователя к конфигу
    '''
    host_user_name = '"{user}"@"{ip}"'.format(user=user, ip=ip)

    # проверить, как будет ругаться если имена баз/таблиц - спецсимволы, или содержат пробел
    def revoke(priv_level, priv_type, object_type):
        return 'REVOKE {priv_type} '.format(priv_type=', '.join(priv_type)) \
                + 'ON {object_type} {priv_level} FROM {user};'.format(priv_level=priv_level,
                                                                      object_type=object_type,
                                                                      user=host_user_name)

    def revoke_grant_opt(priv_level, object_type):
        return 'REVOKE GRANT OPTION ' \
                + 'ON {object_type} {priv_level} FROM {user};'.format(
                    priv_level=priv_level,
                    object_type=object_type,
                    user=host_user_name)

    def grant(priv_level, priv_type, object_type):
        return 'GRANT {priv_type} '.format(priv_type=', '.join(priv_type)) \
                + 'ON {object_type} {priv_level} TO {user};'.format(
                    priv_level=priv_level,
                    object_type=object_type,
                    user=host_user_name)

    def grant_grant_opt(priv_level, object_type):
        return 'GRANT USAGE ON {object_type} '.format(object_type=object_type) \
                + '{priv_level} TO {user} WITH GRANT OPTION;'.format(
                    priv_level=priv_level,
                    user=host_user_name)

    add = []
    remove = []

    for priv_level, grant_config_record in grants_config.items():
        if priv_level in grants_mysql:
            # Могут ли не сойтись object_type при равных priv_level?... Проверить
            sg_conf = set(grant_config_record.priv_type) #set_grant_config
            sg_mysql = set(grants_mysql[priv_level].priv_type) #set_grant_mysql
            up = list(sg_conf - sg_mysql)
            down = list(sg_mysql - sg_conf)

            if up:
                # если есть что добавить в бд
                add.append(grant(priv_level, up, grant_config_record.object_type))
            if down and destructive:
                # есть что удалить
                remove.append(revoke(priv_level, down, grants_mysql[priv_level].object_type))
            if grant_config_record.grant_option != grants_mysql[priv_level].grant_option:
                if grant_config_record.grant_option:
                    add.append(grant_grant_opt(priv_level, grant_config_record.object_type))
                elif destructive:
                    remove.append(revoke_grant_opt(priv_level, grant_config_record.object_type))
        else:
            # в бд нет грантов для этого priv_level
            add.append(grant(priv_level,
                             grant_config_record.priv_type,
                             grant_config_record.object_type
                            )
                      )
            if grant_config_record.grant_option:
                # Добавить grant_option если требуется
                add.append(grant_grant_opt(priv_level, grant_config_record.object_type))

    for priv_level, grant_mysql_record in grants_mysql.items():
        if priv_level not in grants_config:
            # в бд есть гранты которых нет в конфиге
            if grant_mysql_record.priv_type and destructive:
                remove.append(revoke(priv_level,
                                     grant_mysql_record.priv_type,
                                     grant_mysql_record.object_type
                                    )
                             )
            if grant_mysql_record.grant_option and destructive:
                # Если был grant_option то удалить
                remove.append(revoke_grant_opt(priv_level,
                                               grant_mysql_record.object_type
                                              )
                             )
    return add, remove


def compare(grant_requests, db_grants,
            users, user_table,
            hosts, ip_to_alias,
            mysql_ver, destructive):
    '''
    Выводит все изменения которые необходимо сделать в бд чтобы привести ее к конфигу.
    Возвращает
    mysql инструкции:
        1 каких пользователей удалить
        2 каких пользователей добавить
        3 какие гранты выдать
        4 какие гранты удалить
    списки (user, ip)
        5 пользователи которых добавили
        6 пользователи которых удалили
    флаги
        7 менялись ли пользователи (удалены, добавлены, изменен пароль)
        8 менялись ли гранты
    '''

    instructions_users_add = []
    instructions_users_remove = []
    instructions_add = []
    instructions_remove = []

    add_for_human = [] # список добавленных пользователей (user, ip)
    rem_for_human = [] # список удаленных пользователей (user, ip)
    grants_changed = False
    users_changed = False

    req_user_ip = set()
    for (user, alias), grants_config in grant_requests.items():
        for ip in hosts[alias]:
            req_user_ip.add((user, ip))
            user_instruction = check_user(user, ip, users[user], user_table, mysql_ver)
            if user_instruction is not None:
                if user_instruction.startswith('CREATE'):
                    # Пользователя не существует
                    add_for_human.append((user, ip))
                users_changed = True
                instructions_users_add.append(user_instruction)

            add, remove = diff(grants_config, db_grants[(user, ip)], user, ip, destructive)

            if not (add_for_human and add_for_human[-1] == (user, ip)) and (add or remove):
                # если это не новый пользователь и что-то изменилось то меняем значение флага
                grants_changed = True

            instructions_add += add
            instructions_remove += remove

    for (user, ip) in user_table:
        if ip in ip_to_alias and (user, ip_to_alias[ip]) in grant_requests:
            continue
        _, remove = diff({}, db_grants[(user, ip)], user, ip, destructive)

        if remove:
            grants_changed = True

        instructions_remove += remove

        if (user, ip) not in req_user_ip:
            # Пользователь есть в бд но его нет в конфиге. Удалить
            if destructive:
                instructions_users_remove.append('DROP USER "{user}"@"{ip}";'.format(user=user,
                                                                                     ip=ip)
                                                )
                rem_for_human.append((user, ip))
                users_changed = True
        else:
            user_instruction = check_user(user, ip, users[user], user_table, mysql_ver)
            if user_instruction is not None:
                # Пароль изменился
                users_changed = True
                instructions_users_add.append(user_instruction)

    return instructions_users_add, \
            instructions_users_remove, \
            instructions_remove, \
            instructions_add, \
            add_for_human, \
            rem_for_human, \
            users_changed, \
            grants_changed


def write_log(changed, add_for_human, rem_for_human, actions):
    to_mylist_format = lambda lst: ['{ip} to user {usr}'.format(ip=ip, usr=usr) for usr, ip in lst]

    dump = yaml.dump({'changed': changed,
                      'hosts_added': to_mylist_format(add_for_human),
                      'hosts_removed': to_mylist_format(rem_for_human),
                      'actions': actions}, default_flow_style=False, width=1000)
    sys.stderr.write('[{date}]: {scr_name}: comparison over with:\n'.format(date=datetime.now(),
                                                                            scr_name=SCRIPT_NAME))
    sys.stderr.write(dump)


def print_debug(grant_requests, db_grants, users, user_table, hosts, ip_to_alias):
    print "DEBUG: grant_req"
    for (user, alias), grants_config in grant_requests.items():
        print user, alias
        for key, value in grants_config.items():
            print "    ", key, value
    print "\nDEBUG: db_grants"
    for (user, alias), grants_config in db_grants.items():
        print user, alias
        for key, value in grants_config.items():
            print "    ", key, value
    print "\nusers:\n", users, \
            "\n\nusers_table\n", user_table, \
            "\n\nhosts\n", hosts, \
            "\n\nip_to_alias\n", ip_to_alias, "\n"


def apply_changes_to_db(args, connection_opts, actions):
    try:
        dbs = MySQLdb.connect(**connection_opts)
        cursor = dbs.cursor()
    except MySQLdb.OperationalError as exc:
        raise MyException('While changing database: ' + exc.args[1], 23)
    except MySQLdb.InternalError as exc:
        raise MyException('While changing database. '
                          + 'Internal db error. Message: ' + exc.message, 24)
    try:
        def logging_dec(func):
            if args.log is not None:
                sys.stderr.write('Begin to apply grants...\n')
                def log_f(query):
                    sys.stderr.write('Execute: ' + query + '\n')
                    func(query)
                    sys.stderr.write('OK\n')
                return log_f
            return func

        execute = logging_dec(cursor.execute)

        for i in actions:
            execute(i)

    except MySQLdb.Error, exc:
        # Системные таблицы в mysql используют MyISAM, а он не поддерживает транзакции.
        # Поэтому делать rollback(как и commit) бесполезно,
        # но оно здесь в надежде на транзакции в будущем
        dbs.rollback()
        raise MyException('While changing database.\n{} {}'.format(exc[0], exc[1]), 22)
    finally:
        cursor.close()
        dbs.close()


def print_human(add_for_human, rem_for_human, grants_changed, users_changed, ip_to_alias):
    print 'new hosts:'
    for user, ip in add_for_human:
        print '\t{ip} added to {alias} for {user}'.format(ip=ip,
                                                          alias=ip_to_alias[ip],
                                                          user=user)
    if not add_for_human:
        print '\tNo changes'
    print 'removed hosts:'
    for user, ip in rem_for_human:
        print '\t{ip} removed for {user}'.format(ip=ip, user=user)
    if not rem_for_human:
        print '\tNo changes'
    print '\ngrants changed: {}'.format(('no', 'yes')[grants_changed])
    print '\nusers changed: {}'.format(('no', 'yes')[users_changed])


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.RawDescriptionHelpFormatter,
                                     description=__doc__)
    parser.add_argument('--grants-config', action='store', dest='grants_config',
                        required=True, help='Задает путь к конфигу.')
    parser.add_argument('--human', action='store_true', help='Человекочитаемый вывод.')
    parser.add_argument('--destructive', action='store_true',
                        help='Добавляет инструкции на удаление.')
    parser.add_argument('--socket', dest='unix_socket',
                        help='Задает путь к сокету для подключения к бд.')
    parser.add_argument('--host', help='Задает хост для подключения к бд.')
    parser.add_argument('--password', dest='passwd', help='Задает пароль для подключения к бд.')
    parser.add_argument('--user', help='Задает имя пользователя под которых заходим.')
    parser.add_argument('--port', type=int, help='Задает порт для подключения к бд.')
    parser.add_argument('--ignore-version', action='store_true', dest='ignore_version',
                        help='Игнорировать версию mysql в конфиге.')
    parser.add_argument('--debug', action='store_true', help='Вывод отладочной информации.')
    parser.add_argument('--apply', action='store_true', help='Применять требуемые изменения к бд.')
    parser.add_argument('--monrun', action='store_true', help='Вывод для monrun.')
    parser.add_argument('--log', help='Вывод логов в указанный файл')
    parser.add_argument('--write-no-cache', action='store_true', dest='write_no_cache',
                        help='Не записывать ничего в кэш.')
    parser.add_argument('--max-cache-age', type=int, dest='max_cache_age',
                        default=3*60*60, # 3 часа
                        help='Максимальное время жизни кэш файлов (сек).')
    parser.add_argument('--check', action='store_true', help='Проверка формата конфига.')
    parser.add_argument('--ignore-empty-groups', action='store_true',
                        help='Игнорировать пустые группы при резолвинге (можно легко удалить все гранты)')
    parser.add_argument('--conductor-url', dest='conductor_url',
                        default='http://c.yandex-team.ru/api/groups2hosts',
                        help='Задать URL кондуктора.')
    parser.add_argument('--zero-code-on-changes', action='store_true',
                        help='Выходим с нулевым кодом, если в базе пришлось что-то менять (но мы успешно поменяли).' + \
                        ' По-умолчанию выходит с нулевым кодом только если никаких изменений в грантах не требуется')
    return parser.parse_args()


def main():
    # разбираем аргументы командной строки
    args = parse_args()
    # перенаправляем поток ошибок в файл с логами
    if args.log is not None:
        try:
            sys.stderr = open(args.log, 'a')
            sys.stderr.write('[{date}]: '.format(date=datetime.now()) + SCRIPT_NAME + ': '
                             + 'started with args: {args}\n'.format(args=sys.argv[1:]))
        except (IOError, OSError) as exc:
            raise MyException(args.log + ': ' + exc.strerror, exc.errno)
    # запрещаем одновременную работу двух экземпляров скрипта
    script_file = open(sys.argv[0])
    try:
        fcntl.flock(script_file, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        sys.stderr.write('!!! ' + SCRIPT_NAME + ': Error. '
                         + 'Script is already running in another process.\n')
        return 29
    # конфликтующие параметры
    if args.monrun and args.human:
        sys.stderr.write(SCRIPT_NAME + ' : Error. '
                         + 'Conflicting output format parameters. Got --monrun and --human.\n')
        return 27
    # Собираем параметры подключения
    connection_opts = {}
    for opt in ('unix_socket', 'host', 'user', 'passwd', 'port'):
        if args.__getattribute__(opt) is not None:
            connection_opts[opt] = args.__getattribute__(opt)
    # парсим конфиг
    users, \
    hosts_aliases, \
    grant_requests, \
    mysql_ver = parse_config(args.grants_config, not args.ignore_version)
    # Дошли до этого места значит конфиг распарсился нормально.
    if args.check:
        print 'Configuration file is OK'
        exit()
    # резолвим все алиасы в ip
    hosts, ip_to_alias = resolve_aliases(hosts_aliases, args)
    # выкачиваем список пользователей и их гранты
    db_grants, user_table, mysql_ver = get_db_grants_state(args, connection_opts, mysql_ver)
    # тут понятно. просто вывод всего
    if args.debug:
        print_debug(grant_requests, db_grants, users, user_table, hosts, ip_to_alias)
    # основная часть: сравниваем конфиг и гранты в бд. Получаем инструкции mysql
    instructions_users_add, \
    instructions_users_remove, \
    instructions_remove, \
    instructions_add, \
    add_for_human, \
    rem_for_human, \
    users_changed, \
    grants_changed = compare(grant_requests, db_grants,
                             users, user_table,
                             hosts, ip_to_alias,
                             mysql_ver, args.destructive
                            )
    # согласен выглядит не очень здорово
    # Собираем все инструкции вместе
    actions = []
    actions += instructions_users_add
    actions += instructions_remove + instructions_users_remove
    actions += instructions_add
    if actions:
        actions = ['SET SQL_LOG_BIN=0;', 'SET autocommit=0;'] + actions \
                + ['FLUSH PRIVILEGES;', 'COMMIT;']
    # Вывод результатов сравнения в лог
    if args.log is not None:
        write_log(users_changed or grants_changed, add_for_human, rem_for_human, actions)

    # Применяем изменения к бд
    if args.apply:
        apply_changes_to_db(args, connection_opts, actions)

    # Вывод результата
    if args.monrun:
        if users_changed or grants_changed:
            print '1;some grants changed, run with --human'
        else:
            print '0;OK'
    elif args.human:
        print_human(add_for_human, rem_for_human, grants_changed, users_changed, ip_to_alias)
    else:
        for i in actions:
            print i

    if args.log is not None:
        sys.stderr.write('[{date}]: {scr_name}: finished successfully.\n'.format(
            date=datetime.now(),
            scr_name=SCRIPT_NAME))

    exit_zero = args.zero_code_on_changes or not (users_changed or grants_changed)
    return 0 if exit_zero else 1


def main_wrapper():
    '''
    Обертка над main чтобы не писать много раз обработку одного и того же исключения
    '''
    try:
        return main()
    except MyException as exc:
        message, errno = exc.args
        sys.stderr.write(SCRIPT_NAME + ': Error. ' + message + '\n')
        return errno


if __name__ == '__main__':
    exit(main_wrapper())
