#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
import argparse
import subprocess
import shlex
import re
import difflib
import os
import logging

def run_external_utility(cmd, stdin=None):
    args = shlex.split(cmd)
    p = subprocess.Popen(args, stdin=stdin, stdout=subprocess.PIPE)
    if p.wait() != 0:
        exit(1)
    return p.stdout.readlines()

def dump(ver, mysql_params):
    cmd = "mysql {} -e 'select user, host, password from mysql.user'".format(mysql_params)
    if ver == "5.7":
        cmd = "mysql {} -e 'select user, host, plugin, authentication_string from mysql.user'".format(mysql_params)

    users = run_external_utility(cmd)[1:]
    grants = []
    for line in users:
        usr, host = line.split()[:2]
        grants += run_external_utility('mysql {params} -e "show grants for \'{usr}\'@\'{host}\'"'.format(params=mysql_params, usr=usr, host=host))
    return users, grants


def clean_db(mysql_params):
    mysql_cmd = "SET SQL_LOG_BIN=0;\n" \
                + "DELETE FROM mysql.user where user <> 'root' or host <> '%';\n" \
                + "DELETE FROM mysql.db;\n" \
                + "DELETE FROM mysql.tables_priv where user <> 'root' or host <> '%';\n" \
                + "flush privileges;\n"
    run_external_utility('mysql {params} -e "{cmd}"'.format(params=mysql_params, cmd=mysql_cmd))


def init(mysql_params, directory):
    if os.path.exists(directory + 'schema'):
        run_external_utility('mysql {} db'.format(mysql_params), open('{}schema'.format(directory)))
    if os.path.exists(directory + 'init_users.sql'):
        run_external_utility('mysql {}'.format(mysql_params), open('{}init_users.sql'.format(directory)))
    if os.path.exists(directory + 'init_grants.sql'):
        run_external_utility('mysql {}'.format(mysql_params), open('{}init_grants.sql'.format(directory)))


def get_ans(dir_name):
    ans_users = open(dir_name + 'ans_users.sql').readlines()
    ans_grants = open(dir_name + 'ans_grants.sql').readlines()
    ans_script_output = open(dir_name + 'ans_script_output').readlines()
    return ans_users, ans_grants, ans_script_output


def compare(ans_users, users, ans_grants, grants, ans_script_output, script_output):
    def diff(what, l1, l2):
        l1.sort()
        l2.sort()
        l1 = [s.strip() for s in l1]
        l2 = [s.strip() for s in l2]
        rs = difflib.unified_diff(l1, l2)
        try:
            rs.next()
            rs.next()
        except StopIteration:
            return True
        else:
            print
            print what + ": diff from ans ot real output"
            for ln in rs:
                if ln[0] in ('+', '-'):
                    print ln
            return False

    rs1 = diff('users', ans_users, users)
    rs2 = diff('grants', ans_grants, grants)
    rs3 = diff('script_output', ans_script_output, script_output)
    return rs1 and rs2 and rs3


def main():
    logger = logging.getLogger()
    logger.setLevel(logging.getLevelName(os.environ.get('LOGLEVEL', 'INFO')))
    logging.debug('do_one_test started')

    parser = argparse.ArgumentParser(description='Запускает тест из папки $version/$test')
    parser.add_argument('--version', required=True, help='Версия бд.')
    parser.add_argument('--mysql-params', dest='mysql_params', \
                        required=True, help='Параметры для подключения к бд.')
    parser.add_argument('--port', required=True, help='Порт для подключения к фейковому кондуктору.')
    parser.add_argument('--test', required=True, help='Имя теста.')
    args = parser.parse_args()

    logging.debug('with args: ' + str(args))

    directory = re.sub('[^/]*$', '', sys.argv[0]) + args.version + '/' + args.test + '/'
    params = '--conductor-url http://127.0.0.1:{} '.format(args.port) \
            + '--grants-config {}config.yaml '.format(directory) \
            + '--write-no-cache --max-cache-age -1 --ignore-empty-groups --zero-code-on-changes'
    if os.path.exists(directory + 'additional_params'):
        params += ' ' + open(directory + 'additional_params').readline()

    logging.debug('and params: ' + str(params))

    clean_db(args.mysql_params)
    init(args.mysql_params, directory)
    path_to_script = re.sub('[^/]*/[^/]*$', 'bin/mysql-compare-grants', sys.argv[0])
    script_output = run_external_utility(path_to_script + ' ' + args.mysql_params + ' ' + params)
    users, grants = dump(args.version, args.mysql_params)
    clean_db(args.mysql_params)
    ans_users, ans_grants, ans_script_output = get_ans(directory)
    if compare(ans_users, users, ans_grants, grants, ans_script_output, script_output):
        return 0
    else:
        return 1


if __name__ == '__main__':
    exit(main())
