#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
import glob
import re
import yaml
from collections import defaultdict
import os.path
import subprocess
import requests
from argparse import Namespace

dr = os.path.dirname(__file__)
sys.path.append(os.path.join(dr, '..', 'bin'))
sys.path.append(os.path.join(dr, '..', 'functests'))


import unittest
import mysql_compare_grants as cg

def mocked_requests_get(*args, **kwargs):
    class MockResponse:
        def __init__(self, json_data, status_code):
            self.json_data = json_data
            self.status_code = status_code

        def json(self):
            return self.json_data

        def raise_for_status(self):
            if self.status_code != 200:
                raise requests.exceptions.HTTPError

    template = 'cond/%s?format=json'

    if args[0] == template % 'bad3':
        return MockResponse(None, 300)
    elif args[0] == template % 'ok':
        return MockResponse([{'fqdn': 'a'}, {'fqdn': 'b'}], 200)
    elif args[0] == template % 'ok1':
        return MockResponse([{'fqdn': 'some_name'}], 200)
    elif args[0] == template % 'slow':
        raise requests.exceptions.Timeout('too slow')
    return MockResponse(None, 404)

class TestCompareGrantsMethods(unittest.TestCase):

    def setUp(self):
        self.params = Namespace(log=None, 
                                write_no_cache=True, 
                                max_cache_age=0, 
                                ignore_empty_groups=True,
                                conductor_url='cond')

    def make_grants_dict(self, parsed_yaml_dict):
        '''
        Преобразует файл с грантами из тестов к виду используемому в скрипте
        '''
        w = {}
        for priv_level, g in parsed_yaml_dict.items():
            w[priv_level] = cg.Grant(priv_type = g['priv_type'], object_type = g['object_type'], grant_option = g['grant_option'])
        return w

    def parse_grants_sql(self, file):
        '''
        Преобразует файл с грантами из дампа mysql к виду используемому в скрипте
        '''
        db_grants = defaultdict(dict)
        key = ()
        with open(file) as f:
            for line in f:
                if line.startswith('Grants'):
                    user, ip = line.split()[2].rsplit(r'@', 1)
                    key = (user.replace("'", ""), ip.replace("'", ""))
                else:
                    res = cg.parse_grant_message(line)
                    if res is not None:
                        lvl, gr = res
                        db_grants[key][lvl] = gr
        return db_grants

    def parse_users_sql(self, file):
        '''
        Преобразует файл с пользователями из дампа mysql к виду используемому в скрипте
        '''
        w = {}
        with open(file) as f:
            for line in f:
                user, ip, usr_hash = line.split()
                w[(user, ip)] = [usr_hash.replace("'", "")]
        return w

    def test_get_host_names_from_conductor(self):
        old_func = cg.requests.get
        cg.requests.get = mocked_requests_get
        res = cg.get_host_names_from_conductor('ok', self.params)
        res.sort()
        ans = ['a', 'b']
        ans.sort()
        self.assertEqual(ans, res)
        res = cg.get_host_names_from_conductor('ok1', self.params)
        res.sort()
        ans = ['some_name']
        ans.sort()
        self.assertEqual(ans, res)
        for group_name in ('bad3', 'name_not_exists', 'slow'):
            with self.assertRaises(cg.MyException):
                cg.get_host_names_from_conductor(group_name, self.params)
        cg.requests.get = old_func

    def test_parse_config(self):
        '''
        Проверяет что функция parse_config кидает исключения с плохими конфигами
        '''
        with self.assertRaises(cg.MyException):
            cg.parse_config('dtyuilfghjkkiuf', True)
        for file in glob.glob("t.py/tests/bad_config/*"):
            with self.assertRaises(cg.MyException, True):
                cg.parse_config(file)

    def test_parse_grant_message(self):
        '''
        Проверяет правильную работу парсера сообщений с грантами от mysql
        '''
        # useless message
        result = cg.parse_grant_message('GRANT USAGE ON TABLE db.table TO idontcare@1.1.1.1')
        self.assertIsNone(result)
        # only grant option
        lvl, g = cg.parse_grant_message('GRANT USAGE ON db.GRANT TO idonTOcare@1.1.1.1 WITH GRANT OPTION')
        self.assertEqual(lvl, 'db.GRANT')
        self.assertEqual(g, cg.Grant('TABLE', [], True))
        # strange symbols
        lvl, g = cg.parse_grant_message('GRANT INSERT ON TABLE ON.* TO 1234567!@#$%^&*)(_+=~;:@1.1.1.1')
        self.assertEqual(lvl, 'ON.*')
        self.assertEqual(g, cg.Grant('TABLE', ['INSERT'], False))
        # two-word grants and *
        lvl, g = cg.parse_grant_message('GRANT CREATE USER, CREATE VIEW, INSERT, SHOW DATABASES ON * TO 1234567!@#$%^&*)(_+=~;:@1.1.1.1')
        g.priv_type.sort()
        self.assertEqual(lvl, '*')
        self.assertEqual(g, cg.Grant('TABLE', sorted(['CREATE USER', 'CREATE VIEW', 'INSERT', 'SHOW DATABASES']), False))
        # all privileges
        lvl, g = cg.parse_grant_message('GRANT ALL PRIVILEGES ON *.* TO 1234567!@#$%^&*)(_+=~;:@1.1.1.1')
        self.assertEqual(lvl, '*.*')
        self.assertEqual(g, cg.Grant('TABLE', ['ALL'], False))
        # all with grant opt
        lvl, g = cg.parse_grant_message('GRANT ALL PRIVILEGES ON *.* TO 1234567!@#$%^&*)(_+=~;:@1.1.1.1 WITH GRANT OPTION')
        self.assertEqual(lvl, '*.*')
        self.assertEqual(g, cg.Grant('TABLE', ['ALL'], True))
        # function
        lvl, g = cg.parse_grant_message('GRANT ALTER ROUTINE, EXECUTE ON FUNCTION 1234567!@#$%^&*)(_+=~;:.MAnY_symmbols TO 1234567!@#$%^&*)(_+=~;:@1.1.1.1')
        g.priv_type.sort()
        self.assertEqual(lvl, '1234567!@#$%^&*)(_+=~;:.MAnY_symmbols')
        self.assertEqual(g, cg.Grant('FUNCTION', sorted(['EXECUTE', 'ALTER ROUTINE']), False))
        # procedure
        lvl, g = cg.parse_grant_message('GRANT USAGE ON PROCEDURE db.f TO u@h WITH GRANT OPTION')
        self.assertEqual(lvl, 'db.f')
        self.assertEqual(g, cg.Grant('PROCEDURE', [], True))
        # columns
        lvl, g = cg.parse_grant_message('GRANT SELECT (c1, c2) ON db.table TO u@h WITH GRANT OPTION')
        self.assertEqual(lvl, 'db.table')
        self.assertEqual(g, cg.Grant('TABLE', ['SELECT (c1, c2)'], True))

    def text_file_diff_tests(self, in_file):
        '''
        Проверка функции diff. Описание каждого теста лежит рядом с самим тестом в формате .txt
        '''
        tc_name = re.sub('^t.py/tests/diff/(.*)_in.yaml$', '\g<1>', in_file)
        out_file = "t.py/tests/diff/%s_out.yaml" % tc_name

        in_yaml = yaml.load(open(in_file), Loader = yaml.Loader)

        grants_config = self.make_grants_dict(in_yaml['config'])
        grants_mysql = self.make_grants_dict(in_yaml['mysql'])

        add, remove = cg.diff(grants_config, grants_mysql, 'user', 'ip', in_yaml['destructive'])
        add.sort()
        remove.sort()

        out_yaml = yaml.load(open(out_file), Loader = yaml.Loader)
        add_ans = out_yaml['add']
        remove_ans = out_yaml['remove']
        add_ans.sort()
        remove_ans.sort()
        self.assertEqual(add, add_ans, "test '%s' on diff add" % tc_name)
        self.assertEqual(remove, remove_ans, "test '%s' on diff remove" % tc_name)

    def test_diff(self):
        for file in glob.glob("t.py/tests/diff/*_in.yaml"):
            self.text_file_diff_tests(file)

    def text_file_compare_tests(self, directory):
        '''
        Проверка функции compare. Описание каждого теста лежит рядом с самим тестом в формате .txt
        '''
        tc_name = re.sub('^t.py/tests/compare/(.*)$', '\g<1>', directory)
        config_file = "t.py/tests/compare/%s/config.yaml" % tc_name
        users_sql_file = "t.py/tests/compare/%s/users.sql" % tc_name
        grants_sql_file = "t.py/tests/compare/%s/grants.sql" % tc_name

        users, hosts_aliases, grant_requests, mysql_ver = cg.parse_config(config_file, True)

        hosts, ip_to_alias = cg.resolve_aliases(hosts_aliases, self.params)

        db_grants = self.parse_grants_sql(grants_sql_file)

        user_table = self.parse_users_sql(users_sql_file)

        returned_names = ('instructions_users_add', \
                            'instructions_users_remove', \
                            'instructions_remove', \
                            'instructions_add', \
                            'add_for_human', \
                            'rem_for_human', \
                            'users_changed', \
                            'grants_changed')
        for destructive, file_suf in [(True, 'destructive'), (False, 'add_only')]:
            out_file = "t.py/tests/compare/%s/result_%s.yaml" % (tc_name, file_suf)
            out_yaml = yaml.load(open(out_file), Loader = yaml.Loader)
            l = list(cg.compare(grant_requests, db_grants, users, user_table, hosts, ip_to_alias, mysql_ver, destructive))
            for i in range(len(returned_names)):
                ans = out_yaml[returned_names[i]]
                if i < len(returned_names) - 2:
                    l[i].sort()
                    ans.sort()
                    if i >= len(returned_names) - 4:
                        ans = [tuple(t) for t in ans]
                self.assertEqual(l[i], ans, "test '%s' on compare %s with %s" % (tc_name, returned_names[i], file_suf))


    def test_compare(self):
        for file in glob.glob("t.py/tests/compare/*"):
            self.text_file_compare_tests(file)

    def test_functests(self):
        '''
        запуск тестов с подключением к бд
        '''
        p = subprocess.Popen([os.path.join(dr, '..', 'functests', 'run_all_tests.sh')], stdout = sys.stdout)
        a = p.wait()
        self.assertEqual(a, 0)
