package main

import (
	"bytes"
	"encoding/csv"
	"encoding/json"
	"fmt"
	"io"
	"log"
	"net/http"
	"os"
	"strconv"
)

type Camp struct {
	cid, yndx_shows, moat_imp int
	ctype, name, dt           string
}

type Ticket struct {
	Key string `json:"key"`
}

func main() {
	camps := loadCamps()
	if len(camps) > 0 {
		subj := fmt.Sprintf("На дату %s обнаружены расхождения за сутки с МОАТ у %d РК", camps[0].dt, len(camps))
		fmt.Println(subj)
		var body string
		for _, camp := range camps {
			body += printCamp(camp)
		}
		createTicket(subj, body, camps[0].dt)
	}
}

func loadCamps() []Camp {
	cur := readCSV("moat.csv")
	prev := readCSV("prev.csv")
	prevcids := make(map[int]bool)
	var camps []Camp
	for _, camp := range prev {
		prevcids[camp.cid] = true
	}
	for _, camp := range cur {
		if !prevcids[camp.cid] {
			camps = append(camps, camp)
		}
	}
	return camps
}

func createTicket(subj string, body string, dt string) {
	client := &http.Client{}
	jsonReq := map[string]string{"summary": subj,
		"queue":       "DIRECT",
		"parent":      "DIRECT-148632",
		"assignee":    "andreypav",
		"description": body,
		"unique":      "moat_diff" + dt}
	jsonString, err := json.Marshal(jsonReq)
	var jsonData = []byte(jsonString)
	token := os.Getenv("ST_ROBOT")
	req, err := http.NewRequest("POST", "https://st-api.yandex-team.ru/v2/issues", bytes.NewBuffer(jsonData))
	req.Header.Add("Authorization", "OAuth "+token)
	req.Header.Set("Content-Type", "application/json; charset=UTF-8")
	resp, err := client.Do(req)
	if err != nil {
		panic(err)
	}
	defer resp.Body.Close()
	fmt.Println("Response status:", resp.Status)
	ticket_body, _ := io.ReadAll(resp.Body)
	ticket := Ticket{}
	jsonErr := json.Unmarshal(ticket_body, &ticket)
	if jsonErr != nil {
		log.Fatal(jsonErr)
	}
	fmt.Println("Ticket:", ticket.Key)
	createComment(ticket.Key)
}

func createComment(ticket string) {
	client := &http.Client{}
	jsonReq := map[string]interface{}{"text": "Нужен анализ расхождений",
		"summonees": [...]string{"kochurovvs", "mamonovvitaly"},
	}
	jsonString, err := json.Marshal(jsonReq)
	var jsonData = []byte(jsonString)
	token := os.Getenv("ST_ROBOT")
	req, err := http.NewRequest("POST", "https://st-api.yandex-team.ru/v2/issues/"+ticket+"/comments", bytes.NewBuffer(jsonData))
	req.Header.Add("Authorization", "OAuth "+token)
	req.Header.Set("Content-Type", "application/json; charset=UTF-8")
	resp, err := client.Do(req)
	if err != nil {
		panic(err)
	}
	defer resp.Body.Close()
	fmt.Println("Response status:", resp.Status)
}

func readCSV(filename string) []Camp {
	var camps []Camp
	file, err := os.Open(filename)
	if err != nil {
		panic(err)
	}
	defer file.Close()

	reader := csv.NewReader(file)
	reader.FieldsPerRecord = 6
	reader.Comment = '#'

	for {
		r, e := reader.Read()
		if e != nil {
			break
		}
		cid, _ := strconv.Atoi(r[1])
		yndx_shows, _ := strconv.Atoi(r[3])
		moat_imp, _ := strconv.Atoi(r[4])
		c := Camp{cid: cid, yndx_shows: yndx_shows, moat_imp: moat_imp,
			ctype: r[5], name: r[2], dt: r[0]}
		camps = append(camps, c)
	}
	return camps
}

func printCamp(c Camp) string {
	diff := 100 - 100.*float64(c.moat_imp)/float64(c.yndx_shows)
	s := fmt.Sprintf("\n\nВ РК %d (%s) расхождение: %.2f", c.cid, c.ctype, diff)
	s += fmt.Sprintf("\nВидимых показов в Директ без учёта inApp и SSP: %d", c.yndx_shows)
	s += fmt.Sprintf("\nВидимых показов в МОАТ: %d", c.moat_imp)
	s += "\nПодробнее https://datalens.yandex-team.ru/w123ic2twecap-verificator-dashboard?tab=5G"
	return s
}
