#!/usr/bin/perl -w

=encoding utf8

=head1 NAME

    run-schema-spy.pl - построение визуальной схемы баз данных
    по файлам из db_schema/

    возможный опции:
    --out - директория, в которой будут создаваться html файлы
    --db-schema-dir - директория, в которой лежит описание базы
    --db-schema-svn - ИЛИ адрес директории в svn, где лежит описание базы

=cut

use strict;
use warnings;

use POSIX qw/strftime/;
use Getopt::Long;
use Data::Dumper;
use File::Slurp;
use File::Find;
use File::Path;
use File::Temp qw/tempdir/;
use Text::Markdown qw/markdown/;

use XML::LibXML;

use Yandex::Shell;
use Yandex::DBTools;
use Yandex::DBSchema;
use Yandex::DBUnitTest qw/:no_init/;

use ProjectSpecific qw/svn_url/;

use utf8;

my ($OUT_DIR, $DB_SCHEMA_DIR, $DB_SCHEMA_SVN);
GetOptions(
    "help" => \&usage,
    "out=s" => \$OUT_DIR,
    "db-schema-dir=s" => \$DB_SCHEMA_DIR,
    "db-schema-svn=s" => \$DB_SCHEMA_SVN,
    ) || die "incorrect options: $!";
usage() unless defined $OUT_DIR && (defined $DB_SCHEMA_SVN xor defined $DB_SCHEMA_DIR);
die "Out directory $OUT_DIR does not exists" if !-d $OUT_DIR;

# получаем схему из svn, если её нет локально
if (defined $DB_SCHEMA_SVN) {
    $DB_SCHEMA_DIR = tempdir(CLEANUP => 1);
    my $trunk_url = svn_url('trunk');
    yash_system("svn export --quiet --force $trunk_url/db_schema/ $DB_SCHEMA_DIR");
}
$Yandex::DBSchema::DB_SCHEMA_ROOT = $DB_SCHEMA_DIR;

# для того, чтобы не попортить данные будем создавать базы с уникальными префиксами.
# потом пройдёмся по всем файлам и удалим префиксы
my $DB_PREFIX = "schemaspy_${<}_".strftime("%Y%m%d%H%M%S", localtime)."_";

my $METADATA_DIR = tempdir(CLEANUP => 1);
my $CSS_ADDITION = qq#
    td.comment p {padding: 0px; margin: 0px;}
    td.detail {padding: 6px 4px;}
#;

cleanup_out_dir($OUT_DIR);

# получаем все данные из db_schema
my $db_schema = get_db_schema_data();
add_tags_databases($db_schema);

create_databases_from_schema($db_schema);

create_databases_metadata($db_schema);

my $cfg = get_db_config(UNIT_TESTS);
die "No host or port defined for unittest db" if !$cfg->{host} || !$cfg->{port};

# самая главная часть - запускаем schemaspy
# пока mysql < 5.5 - schemaspy выдаёт ненулевой exit code, поэтому делаем странную проверку
# а системная java слишком старая, поэтому указываем путь
my $cmd = join ' ', map {yash_quote($_)}
    "java",
    '-jar' => "/usr/share/schemaspy/schemaSpy.jar",
    '-d64', '-XX:-DoEscapeAnalysis', '-Xms16G', '-Xmx16G',
    '-dp' => "/usr/share/java/mysql-connector-java.jar",
    '-t' => "mysql",
    '-host' => "$cfg->{host}:$cfg->{port}", -port => $cfg->{port}, -u => $cfg->{user}, -p => $cfg->{pass},
    '-db' => "${DB_PREFIX}ppc",
    '-o' => $OUT_DIR,
    '-schemas' => join(",", map {"$DB_PREFIX$_"} sort keys %$db_schema),
    #-loglevel => 'finest',
    '-meta' => $METADATA_DIR,
    '-charset' => 'utf-8',
    '-norows',
    '-x' => "banners.uid|bids.cid",
    ;
my $ret = `ulimit -Sv 64000000; $cmd 2>&1`;
if ($? && $ret !~ /Start with/) {
    die "SchemaSpy run failed: $ret";
}

# переименовывем директории,
# дописываем в css нужные нам правила
for my $dbname (sort keys %$db_schema) {
    rmtree("$OUT_DIR/$dbname") if -d "$OUT_DIR/$dbname";
    rename("$OUT_DIR/$DB_PREFIX$dbname", "$OUT_DIR/$dbname") || die "can't rename $OUT_DIR/$DB_PREFIX$dbname => $OUT_DIR/$dbname: $!";
    write_file("$OUT_DIR/$dbname/schemaSpy.css", {append => 1}, $CSS_ADDITION);
}

# удаляем DB_PREFIX из всех html файлов
find(sub {
     my $filename = $_;
     return if !-f $filename || $filename !~ /\.html$/;
     my $cont = read_file($filename, binmode => ":utf8");
     if ($cont =~ s/\Q$DB_PREFIX\E//g) {
         write_file($filename, {binmode => ":utf8"}, $cont);
     }
}, $OUT_DIR);

# подчищаем временные базы
END {
    if ($DB_PREFIX && $db_schema) {
        for my $dbname (keys %$db_schema) {
            do_sql(UNIT_TESTS, "DROP DATABASE IF EXISTS $DB_PREFIX$dbname");
        }
    }
}

sub cleanup_out_dir
{
    my ($dir) = @_;
    for my $tmp (glob("$dir/schemaspy_*")) {
        yash_system "rm", "-rf", $tmp;
    }
}

sub create_databases_metadata {
    my $db_schema = shift;
    while(my ($dbname, $db) = each %$db_schema) {
        my $metadata_doc = XML::LibXML::Document->new('1.0', 'UTF-8');
        my $metadata_el = XML::LibXML::Element->new('schemaMeta');
        $metadata_doc->setDocumentElement($metadata_el);
        my $comments_el = $metadata_el->appendChild(XML::LibXML::Element->new('comments'));
        my $tables_el = $metadata_el->appendChild(XML::LibXML::Element->new('tables'));
        while(my ($table_name, $table_info) = each %{$db->{tables}}) {
            my $table_el = $tables_el->appendChild(XML::LibXML::Element->new('table'));
            $table_el->setAttribute(name => $table_name);
            if ($table_info->{desc}) {
                $table_el->setAttribute(comments => markdown($table_info->{desc}->{table_desc}));
            }
            for my $column_info (@{$table_info->{desc}->{columns}}) {
                my $column_el = $table_el->appendChild(XML::LibXML::Element->new('column'));
                $column_el->setAttribute(name => $column_info->{name});
                if ($column_info->{fk}) {
                    my $fk_el = $column_el->appendChild(XML::LibXML::Element->new('foreignKey'));
                    $fk_el->setAttribute(table => $column_info->{fk}->{table});
                    $fk_el->setAttribute(column => $column_info->{fk}->{column});
                }
                if (defined $column_info->{text}) {
                    $column_el->setAttribute(comments => markdown($column_info->{text}));
                }
            }
            
        }
        write_file("$METADATA_DIR/$DB_PREFIX$dbname.meta.xml", $metadata_doc->toString(2));
    }
}

sub create_databases_from_schema {
    my ($db_schema) = @_;
    while(my ($dbname, $db) = each %$db_schema) {
        do_sql(UNIT_TESTS, "CREATE DATABASE IF NOT EXISTS $DB_PREFIX$dbname DEFAULT CHARSET utf8;");    
        do_sql(UNIT_TESTS, "USE $DB_PREFIX$dbname");
        do_sql(UNIT_TESTS, 'set foreign_key_checks = 0');
        while(my ($table_name, $table_info) = each %{$db->{tables}}) {
            next unless $table_info->{sql};
            (my $sql = $table_info->{sql}) =~ s/CREATE TABLE /CREATE TABLE IF NOT EXISTS /;
            do_sql(UNIT_TESTS, $sql);
        }
        do_sql(UNIT_TESTS, 'set foreign_key_checks = 1');
    }
}

sub get_db_schema_data {
    my $db_schema = {};
    for my $db (Yandex::DBSchema::get_databases()) {
        for my $tbl (Yandex::DBSchema::get_tables($db)) {
            $db_schema->{$db}->{tables}->{$tbl}->{sql} = Yandex::DBSchema::get_create_table_sql(db => $db, table => $tbl);
            my $text_desc = Yandex::DBSchema::get_table_text_desc(db => $db, table => $tbl);
            if (defined $text_desc) {
                $db_schema->{$db}->{tables}->{$tbl}->{desc} = Yandex::DBSchema::parse_table_text_desc($text_desc);
            }
        }
    }
    return $db_schema;
}

# добавляем фэйковые базы для каждой пары база/тэг
sub add_tags_databases {
    my $db_schema = shift;
    # делаем связку: 
    for my $db (sort keys %$db_schema) {
        for my $table_info (values %{$db_schema->{$db}->{tables}}) {
            my $tdesc = $table_info->{desc};
            for my $tag (@{$tdesc->{tags}||[]}) {
                my $normalized_tag = ( $tag =~ s/\s+/_/gr );
                $db_schema->{"${db}__$normalized_tag"}->{tables}->{$tdesc->{table_name}} = $table_info;
            }
        }
    }
}

sub usage {
    my $base_cmd = "podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8";

    if ( my $pager = $ENV{PAGER} ) {
        system("$base_cmd | $pager");
    } else {
        system($base_cmd);
    }

    exit(1);
}
