# -*- coding: utf8 -*-

import logging
import re
from subprocess import Popen, PIPE
from threading import Timer
import direct_juggler.juggler as dj
import jsonpath_rw
import json
import imp


# все считанные конфиги, чтобы не перечитывать
CONFIGS_DICT = {}


def run_cmd(cmd, timeout=4):
    logging.info("run_cmd: %s" % " ".join(cmd))
    stdout = u""
    stderr = u""

    try:
        proc = Popen(cmd, stdout=PIPE, stderr=PIPE)
        timer = Timer(timeout, proc.kill)
        try:
            timer.start()
            stdout, stderr = proc.communicate()
        except:
            logging.error("Exception occurred", exc_info=True)
            stdout = u""
        finally:
            timer.cancel()

        if stderr:
            logging.error(stderr.strip())

    except:
        pass

    logging.info("run_cmd output: " + stdout.replace('\n', '\\n'))
    return stdout


def write_pass_to_file(pass_file, password):
    pass_file.write("[client]\npassword=" + password)
    pass_file.flush()


def make_query(host, port, user, pass_file, query):
    logging.debug("/usr/bin/mysql --defaults-file={0} --host {1} --user {2} -e {3}".format(pass_file.name, host, user, query))
    if not port:
        return run_cmd(["/usr/bin/mysql", "--defaults-file=" + pass_file.name, "--host", host, "--user", user, "-e", query])
    else:
        return run_cmd(["/usr/bin/mysql", "--defaults-file=" + pass_file.name, "--host", host, "--port", str(port), "--user", user, "-e", query])


def get_master(host, port, user, pass_file):
    output = make_query(host, port, user, pass_file, "show slave status\G")
    try:
        host = re.search(ur'Master_Host: (.+)', output, re.I).group(1)
    except:
        host = ''

    try:
        port = re.search(ur'Master_Port: (.+)', output, re.I).group(1)
    except:
        port = ''

    return host, port


def get_trx_max_age(host, port, user, pass_file):
    query = '''SELECT ifnull(max(unix_timestamp() - unix_timestamp(trx_started)), 0)
         trx_max_age FROM INFORMATION_SCHEMA.INNODB_TRX trx LEFT JOIN INFORMATION_SCHEMA.PROCESSLIST proc ON
         trx.trx_mysql_thread_id = proc.id  WHERE (proc.user is null or proc.user != 'loaddb') AND
         (trx_lock_structs > 0 OR trx_query IS NOT NULL)'''

    output = make_query(host, port, user, pass_file, query)
    result = output.split('\n')
    if result < 2:
        raise ValueError("wrong result get_trx_max_age: {0}".format(result))
    return result[1]


def not_have_max_transactions(master_host, port, user, pass_file, max_trx_age):
    try:
        max_age = get_trx_max_age(master_host, port, user, pass_file)
        if int(max_age) > int(max_trx_age):
            msg = u"found old transaction in {0}: {1}".format(master_host, max_age)
            logging.info("{0}: {1}".format(master_host, msg))
            return False, msg
    except Exception as err:
        msg = str(err)
        logging.info("{0}: {1}".format(master_host, msg))
        return False, msg
    return True, u""


def get_freshness_value(project, db_type, host, port, user, pass_file, new_heartbeat_table=False):
    search_result = None
    req = "select UNIX_TIMESTAMP() - timestamp from heartbeat.heartbeat"
    if new_heartbeat_table:
        req = "select NOW() - ts from mysql.mdb_repl_mon"
    if db_type == 'xtradb':
        if is_single_xtradb_ok(host, port, user, pass_file)[0]:
            return 0
        else:
            return -1
    else:
        if project == 'direct' and db_type == 'mysql':
            output = make_query(host, port, user, pass_file, req)
            search_result = re.search(ur"([0-9]+)", output, re.I | re.U)

        if not search_result:
            output = make_query(host, port, user, pass_file, "show slave status\G")
            search_result = re.search(ur"Seconds_Behind_Master: ([0-9]+)", output, re.I | re.U)

    if not search_result:
        return -1

    return int(search_result.group(1))


def get_xtradb_slaves(host, port, user, pass_file):
    output = make_query(
        host, port, user, pass_file,
        'SELECT VARIABLE_VALUE FROM performance_schema.global_variables WHERE VARIABLE_NAME = "wsrep_cluster_address"'
    )

    search_result = re.search(ur'gcomm:\/\/(\S+)', output, re.U | re.M)
    if not search_result:
        return []
    logging.info('xtradb cluster addess: ' + str(search_result.group(1)))
    return [(slave.split(':')[0], port) for slave in search_result.group(1).split(',')]


def get_mysql_props(host, port, user, pass_file, props):
    output = make_query(host, port, user, pass_file, "show status")
    output += make_query(host, port, user, pass_file, "show global variables")

    result = []
    for prop in props:
         try:
             value = re.search(ur'%s\s*([^\s]+)' % prop, output, re.I | re.U).group(1)
         except:
             value = ''

         result.append(value)

    return result

def is_not_read_only_state(host, port, user, pass_file):
    logging.info('%s: start is_read_only check' % host)
    props_desired = [
        ('read_only', 'OFF')
    ]
    props_values = get_mysql_props(host, port, user, pass_file, [key for key, desired in props_desired])
    for (key, desired), value in zip(props_desired, props_values):
        if not value:
            return False, u"%s: can't get property %s" % (host, key)
        if value != desired:
            return False, u"%s: value %s for property %s, need OFF" % (host, value, key)
    return True, u""

def is_semisync_slaves(master_host, slaves, master_port, user, pass_file):
    props_desired = [
        ('Rpl_semi_sync_slave_status', 'ON')
    ]

    max_semisync_slaves = 1
    working_semisync_slaves = 0
    errmsg = list()

    for slave in [x for x in slaves if x['host'] != master_host]:
        slave_host = slave['host']
        slave_port = slave['port']
        logging.info('%s: start is_semisync_slave check' % slave_host)

        props_values = get_mysql_props(slave_host, slave_port, user, pass_file, [key for key, desired in props_desired])
        def check_property():
            for (key, desired), value in zip(props_desired, props_values):
                if not value:
                    msg = u"%s: can't get property %s" % (slave_host, key)
                    logging.info('%s: failed is_semisync_slaves check with message: %s' % (slave_host, msg))
                    return False, msg
                if value != desired:
                    msg = u"%s: value %s for property %s, need ON" % (slave_host, value, key)
                    logging.info('%s: failed is_semisync_slaves check with message: %s' % (slave_host, msg))
                    return False, msg
            logging.info('%s: success is_semisync_slaves check' % slave_host)
            return True, u""
        ok, err = check_property()
        if ok:
            working_semisync_slaves += 1
        else:
            errmsg.append(err)
    if working_semisync_slaves < max_semisync_slaves:
        return False, ','.join(errmsg)
    else:
        return True, ""

def is_semisync_master(master_host, port, user, pass_file):
    logging.info('%s: start is_semisync_master check' % master_host)
    props_desired = [
        ('Rpl_semi_sync_master_status', 'ON')
    ]
    props_values = get_mysql_props(master_host, port, user, pass_file, [key for key, desired in props_desired])
    for (key, desired), value in zip(props_desired, props_values):
        if not value:
            msg = u"%s: can't get property %s" % (master_host, key)
            logging.info('%s: failed is_semisync_master with message: %s' % (master_host, msg))
            return False, msg
        if value != desired:
            msg = u"%s: value %s for property %s, need ON" % (master_host, value, key)
            logging.info('%s: failed is_semisync_master with message: %s' % (master_host, msg))
            return False, msg
    logging.info('%s: success is_semisync_master check' % master_host)
    return True, u""

def is_single_xtradb_ok(host, port, user, pass_file, min_cluster_size=2):
    logging.info('%s: start is_single_xtradb_ok check' % host)
    #output = make_query(host, port, user, pass_file, "show status")
    props_desired = [
        ('wsrep_cluster_size', min_cluster_size), # проверяем, что на машинке wsrep_cluster_size >= 2
        ('wsrep_cluster_status', 'Primary'), # и все следующие параметры строгим сравнением
        ('wsrep_connected', 'ON'),
        ('wsrep_local_state_comment', 'Synced'),
        ('wsrep_sst_donor_rejects_queries', 'ON')
    ]
    props_values = get_mysql_props(host, port, user, pass_file, [key for key, desired in props_desired])

    for (key, desired), value in zip(props_desired, props_values):
        if not value:
            return False, u"%s: can't get property %s" % (host, key)

        if key == 'wsrep_cluster_size' and (not value.isdigit() or int(value) < desired):
            return False, u"%s: wsrep_cluster_size %d < %d" % (host, int(value), desired)
        elif key != 'wsrep_cluster_size' and value != desired:
            return False, u"%s: bad value %s for property %s, need %s" % (host, value, key, desired)

    return True, u""


def has_fresh_slave(project, db_type, master_host, master_port, slaves, user, pass_file, max_value):
    logging.info('%s: start fresh_slave check' % master_host)
    if not slaves:
        msg = u"empty list of slaves for master host: %s" % master_host
        logging.info('%s: failed has_fresh_slave with message: %s' % (master_host, msg))
        return False, msg

    min_freshness = -1
    min_slave = None

    for slave in [x for x in slaves if x['host'] != master_host]:
        slave_host = slave['host']
        slave_port = slave['port']
        freshness_value = get_freshness_value(project, db_type, slave_host, slave_port, user, pass_file, False)
        logging.info('%s: freshness for slave %s - %d sec' % (master_host, slave_host, freshness_value))

        if freshness_value == -1:
            continue

        if freshness_value <= max_value:
            # есть одна реплика с slave_behind (freshness_value) <= max_value - это ОК, выходим
            logging.info('%s: freshness for slave %s < %d - OK, exiting' % (master_host, slave_host, max_value))
            return True, u""

        if min_freshness == -1 or freshness_value < min_freshness:
            min_freshness = freshness_value
            min_slave = slave_host

    if min_freshness == -1:
        message = u"%s: no available slaves" % master_host
    else:
        message = u"%s: most recent replica %s is %d seconds behind master" % (master_host, min_slave, min_freshness)

    logging.info('%s: freshness check failed with message: %s' % (master_host, message))
    return False, message

def is_host_alive(host, port, user, pass_file, crit_suffix="master is unreachable"):
    logging.info('%s: start is_host_alive check' % host)
    result = make_query(host, port, user, pass_file, "select now()")

    if result:
        message = u""
    else:
        message = u"%s %s" % (host, crit_suffix)

    logging.info('%s: is_host_alive check result: %s, msg: %s' % (host, bool(result), message))
    return bool(result), message


def read_file(path):
    return open(path, 'r').read()


def report_to_juggler(service, instance, crit_messages):
    crit_message = u"; ".join(list(filter(lambda x: x, crit_messages)))

    if crit_message:
        event = dict(service="%s_%s" % (service, instance), status='CRIT', description=crit_message)
    else:
        event = dict(service="%s_%s" % (service, instance), status='OK', description='OK')

    logging.info("check result: " + str(event))

    result = dj.queue_events([event])
    logging.info("juggler response: %s" % str(result))


def get_db_config_data(db_config, node_path):
    # пробуем распарсить JSONPath с помощью библиотеки
    # к сожалению, он падает на парсинге путей для ppcdata
    # поэтому далее парсим в ручную
    try:
        return jsonpath_rw.parse(node_path).find(db_config)[0].value
    except Exception as e:
        pass

    if len(node_path) == 0:
        raise ValueError('empty node_path value')

    path = node_path.split(".")
    node = db_config

    for item in path:
        node = node[item]

    return node


def get_db_config(instance_config, instance_name):
    db_config_path = instance_config.get('db_config', '')
    db_config = db_config_path

    if db_config_path:
        if db_config_path in CONFIGS_DICT:
            db_config = CONFIGS_DICT[db_config_path]

        elif instance_config.get('db_config_type', '') == 'json':
            CONFIGS_DICT[db_config_path] = json.loads(read_file(db_config_path))
            db_config = CONFIGS_DICT[db_config_path]

        elif instance_config.get('db_config_type', '').startswith('python'):
            CONFIGS_DICT[db_config_path] = getattr(
                imp.load_source('%s_module' % instance_name, db_config_path),
                instance_config['db_config_type'].split(':')[1]
            )
            db_config = CONFIGS_DICT[db_config_path]

    return db_config

def get_data_from_instance_conf(instance_config, db_config):
    master_host = ""
    master_port = 3306
    if 'db_config_master_node' in instance_config:
	try:
            master_host = get_db_config_data(db_config, instance_config['db_config_master_node'])
        except ValueError as err:
            logging.info("{0}. Use default value {1}".format(err, master_host))
    if 'db_config_port_node' in instance_config:
        try:
            master_port = get_db_config_data(db_config, instance_config['db_config_port_node'])
        except ValueError as err:
            logging.info("{0}. Use default value {1}".format(err, master_port))
    default_slave_port = instance_config.get('mysql_port', '')
    slaves = [
        {
            'host': replica['host'],
            'port': replica.get('mysql_port', default_slave_port)
        } for replica in instance_config.get("replicas", [])
    ]
    max_freshness = instance_config.get('monitoring_max_freshness', 300)
    max_age_transaction = instance_config.get('max_age_transaction', 300)

    return master_host, master_port, slaves, max_freshness, max_age_transaction
