use strict;
use warnings;

package DBConfigExplorer;

use utf8;

use YAML;
use JSON;
use File::Slurp;

=head2 list_properties

    превращает многоуровневый yaml-конфиг в плоский хеш "свойств" вида 
    ppc.connect_timeout => 60,
    ppc:heavy.host => 'ppcdata1.yandex.ru',
    ppcstat:1.ketama_key => 'ppcstat1.yandex.ru'
    ppcstat:1.utf8 => 1

=cut
sub list_properties
{
    my ($dbconf) = @_;

    my @leafs = find_leafs($dbconf->{db_config}, []);

    my $prop = {};

    # для каждого листа собираем соответствующий ему набор параметров и помещаем с нужным именем в список свойств
    for my $leaf ( @leafs ){
        my $v = $dbconf->{db_config};
        my $p;
        $p->{$_} = $v->{$_} for grep {!/^CHILDS$/} keys %$v;
        for my $name_part (@$leaf){
            $v = $v->{CHILDS}->{$name_part};
            $p->{$_} = $v->{$_} for grep {!/^CHILDS$/} keys %$v;
        }
        my $leaf_name = join ":", grep {!/^_$/} @$leaf;
        for my $k (keys %$p){
            serialize_property( $prop, "$leaf_name.$k", $p->{$k} );
        }
    }

    return $prop;
}

=head2 serialize_property

    сериализует названия свойств из сложной структуры db-config и пишет в одноуровневый хеш
    $res -- куда писать результат
    $name_prefix -- префикс для ключей
    $value -- значение, которое надо сериализовать и записать

    в res получается такой хеш:

ppc:1.AutoCommit: 1
ppc:1.compression: 1
ppc:1.connect_timeout: 4
ppc:1.extra_users.direct-sql.pass: ...
ppc:1.host: ppcdata1-01f.yandex.ru
ppc:1.instance: ppcdata1
ppc:1.pass: ...
ppc:1.port: 3406
ppc:1.user: adiuser
ppc:1.utf8: 1
ppc:1.weight: 0
ppc:1:bs.AutoCommit: 1
ppc:1:bs.compression: 1
ppc:1:bs.connect_timeout: 4
...    

=cut
sub serialize_property
{
    my ($res, $name_prefix, $value) = @_;

    if (!ref $value){
        $res->{$name_prefix} = $value;
    } elsif ( ref $value eq 'ARRAY') {
        for my $i ( 0 .. @$value - 1){
            serialize_property($res, "$name_prefix.$i", $value->[$i]);
        }
    } elsif ( ref $value eq 'HASH' ){ 
        for my $k ( keys %$value ){
            serialize_property($res, "$name_prefix.$k", $value->{$k});
        }
    } else {
        die "unknown type of property";
    }


    return;
}


=head2 find_leafs

    возвращает список всех листьев в dbconfig'е. 
    Каждый лист представлен ссылкой на массив имен промежуточных вершин

=cut
sub find_leafs
{
    my ($root, $prefix) = @_;

    if (!exists $root->{CHILDS}){
        return $prefix;
    }

    my @childs = ();
    for my $name (keys %{$root->{CHILDS}}){
        push @childs, find_leafs($root->{CHILDS}->{$name}, [@$prefix, $name]);
    }

    return @childs;
}


sub load_config
{
    my $file = shift;
    if ($file =~ /\.ya?ml$/) {
        return YAML::LoadFile($file);
    } elsif ($file =~ /\.json$/) {
        return JSON::from_json(scalar read_file($file));
    } else {
        die "Unknown file format: '$file'";
    }
}

1;
