#!/usr/bin/env perl

# $Id$


=head1 NAME

=encoding utf8

debsums-check - проверка консистентности определённого deb-пакета (или пакетов)

=head1 DESCRIPTION

    Запускает debsums

    Опции:
    --help
        вывести справку
    --package=s
        обязательный параметр; полное имя проверяемого пакета; допускается повторное использование
    --conf=s
        необязательный параметр; файл с конфигурацией, может содержать ключ exclude-files
    --tries=i
        сколько раз пытаться вызвать debsums, не соглашаясь с тем, что пакет не установлен
    --sleep=i
        сколько секунд ждать между вызовами debsums, не соглашаясь с тем, что пакет не установлен

    debsums-check.pl --package=yandex-direct-frontend --conf=/etc/debsums-monitor/debsums-monitor.ts.conf

    выведет на STDOUT список изменённых или отсутствующих файлов и диффы изменений

=cut

use strict;
use warnings;
use utf8;
use open ':std' => ':utf8';

use File::Temp qw/tempdir/;
use Getopt::Long;
use YAML;

my $DEBSUMS_RETRY_TRIES = 3;
my $DEBSUMS_RETRY_SLEEP = 20;

my ($CONFIG, @packages);
GetOptions(
    "help" => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text >&2"); exit 0;},
    "config=s" => \$CONFIG,
    "package=s" => \@packages,
    "tries=i" => \$DEBSUMS_RETRY_TRIES,
    "sleep=i" => \$DEBSUMS_RETRY_SLEEP,
) or die "can't parse options";
die "no --package option given" unless @packages;
die "bad --sleep option" if $DEBSUMS_RETRY_SLEEP < 0;
die "bad --tries option" if $DEBSUMS_RETRY_TRIES < 0;

my (%exclude_files, @exclude_file_expressions);
if ($CONFIG) {
    die "configuration file not found" unless -r $CONFIG;
    my $CONF = YAML::LoadFile($CONFIG);
    if ($CONF->{'exclude-files'}) {
        if (ref $CONF->{'exclude-files'} && ref $CONF->{'exclude-files'} eq 'ARRAY' && @{$CONF->{'exclude-files'}}) {
            %exclude_files = map {$_ => undef} @{$CONF->{'exclude-files'}};
        } else {
            die "bad exclude-files configuration entry";
        }
    }
    if ($CONF->{'exclude-file-expressions'}) {
        if (ref $CONF->{'exclude-file-expressions'} && ref $CONF->{'exclude-file-expressions'} eq 'ARRAY' && @{$CONF->{'exclude-file-expressions'}}) {
            @exclude_file_expressions = @{$CONF->{'exclude-file-expressions'}};
        } else {
            die "bad exclude-file-expressions configuration entry";
        }
    }
}

foreach my $package (@packages) {
    if ($package !~ /^[a-zA-Z0-9\.\-]+$/) {
        die "unexpected package name: $package\n";
    }
    my @obsolete_config_files = split /\n+/, `dpkg-query -f '\${Conffiles}\n' -W '$package' | awk '\$3 == "obsolete" {print \$1}'`;
    ### лучше Yabdex::Shell; здесь будет пусто, если неудача
    my $changed = `debsums -ac $package 2>&1`;
    if ($changed) {
        if ($DEBSUMS_RETRY_TRIES && $changed =~ /^debsums: package \Q$package\E is not installed/m) {
            for (1 .. $DEBSUMS_RETRY_TRIES) {
                sleep $DEBSUMS_RETRY_SLEEP;
                $changed = `debsums -ac $package 2>&1`;
                last if (!$changed || ($changed !~ /^debsums: package \Q$package\E is not installed/m));
            }

            next unless $changed;
        }
        my @lines = sort split /[\r\n]+/, $changed;
        my ($temp_dir, $dirname);
        foreach my $file (@lines) {
            if ($file =~ /^debsums: missing file .* \(from \Q$package\E package\)/) {
                print "$file\n"; # можно захватить имя пакета в выражении и писать в своём формате
            } elsif ($file =~ /^debsums:/) {
                print "$file\n"; # просто не знаем, можно добавить больше про неизвестность ошибки
            } else {
                unless (-r $file) {
                    print "unexpected debsums line: $file\n"; # это не файл или мы его не можем читать
                    next;
                }

                next if grep { $_ eq $file } @obsolete_config_files;
                next if exists $exclude_files{$file};
                my $skip = 0;
                for my $expression (@exclude_file_expressions) {
                    if ($file =~ $expression) {
                        $skip = 1;
                        last;
                    }
                }
                next if $skip;

                print "changed: $file (from $package package\)\n";
                if ($dirname && $dirname eq 'UNKNOWN') {
                    print "skipping previously failed extraction for package $package\n";
                    next;
                }

                if (!$temp_dir) {
                    my $package_version = `dpkg-query -f='\${Version}' -W $package`;
                    unless ($package_version) {
                        print "couldn't determine package version for $package\n";
                        $dirname = 'UNKNOWN';
                        next;
                    }

                    my @apt_cache_show_lines = split /[\r\n]+/, `apt-cache show $package=$package_version`;
                    my ($filename, $generated);
                    while (my $l = shift @apt_cache_show_lines) { # это firstval
                        if ($l =~ /^Filename:/) {
                            $l =~ s/.*\/([^\/]+)$/$1/;
                            $filename = $l;
                            last;
                        }
                    }
                    unless ($filename) {
                        $filename = $package . '_' . $package_version . '_all.deb';
                        $generated = 1;
                    }

                    unless (-r "/var/cache/apt/archives/$filename") {
                        if ($generated) {
                            print "couldn't determine package filename for $package=$package_version\n";
                        } else {
                            print "package file $filename not found in cache\n";
                        }
                        $dirname = 'UNKNOWN';
                        next;
                    }

                    $temp_dir = File::Temp->newdir();
                    $dirname = $temp_dir->dirname;
                    eval {
                        `dpkg -x /var/cache/apt/archives/$filename $dirname`;
                    };
                    if ($@) {
                        print "couldn't extract /var/cache/apt/archives/$filename";
                        $dirname = 'UNKNOWN';
                        next;
                    }
                }

                print `diff -u $dirname$file $file`;
            }
        }
    }
}
