#!/usr/bin/env python
# -*- coding: utf-8 -*-

import subprocess
import threading
import socket
import zookeeper
import re
import yaml
import argparse
import datetime
import urllib2
import json
import logging
import time

#from juggler import juggler_queue_event
from kazoo.client import KazooClient
from functools import wraps

zk_servers = ['ppcback01f.yandex.ru:2181', 'ppcback01e.yandex.ru:2181', 'ppcback01i.yandex.ru:2181']
zkh = None
service_zk_dpkg = "deploy.version.zk_vs_dpkg"
service_lp_dpkg = "deploy.version.lp_vs_dpkg"

SECONDS_TO_OK = 300
SECONDS_TO_WARN = 1200

CONFIG = '/etc/yandex-direct/direct-apps.conf.yaml'

compare_statuses = {
    'OK' : 1,
    'WARN' : 2,
    'CRIT' : 3
}

def zk_sync_init(hosts):                         
    """
       Подключаемся к zookeeper. Если подключение удалось, то выставляем (0, <zk connect>),
       если нет - (2, <error message>).
    """
    try:
       zk = KazooClient(hosts=hosts, read_only=False, 
                        timeout=1.0, connection_retry=3, 
                        command_retry=3, logger=logger)
       zk.start()
       (status, result) = 0, zk
    except Exception as err:
       (status, result) = 2, str(err)
    return status, result

def get_version_time_from_zk(app, version_node, zkh):
    """
    Возвращает тупл: (статус выполнения, сообщение, доп.сообщение).
    В случае успеха: (0, <версию из ZK>, <время, в которое она была записана>),
    в случае ошибки: (2, <сообщение ошибку, если возникла>.
    """
    try:
        version_node_content = zkh.get(version_node)
    except:
        error = "can't get version node of {0} from zookeeper".format(app)
        return (2, error, "")

    m = re.match('^([^ ]+).*\n([^ ]+)', version_node_content[0])
    if not m:
        error = "can't parse zookeeper node {0}: '{1}'".format(version_node, version_node_content)
        return (2, error, "")
    version = m.group(1)
    ts = m.group(2)
    return (0, version, ts)


def get_version_from_dpkg(package):
    """
    принимает название приложения, возвращает статус выполнения и сообщение. 
    В случае успеха (0, <версию главного пакета этого приложения>),
    в случае ошибки (2, <сообщение ошибкиш>).
    """
    rgxp = re.compile('Package: (.*)\nStatus: (.*)\nPriority: (.*)\nSection: (.*)\nInstalled-Size: (.*)\nMaintainer: (.*)\nArchitecture: (.*)\nVersion: (.*)\n')
    with open('/var/lib/dpkg/status', 'r') as f1le:
        meta_dpkg = f1le.read()
    data = rgxp.findall(meta_dpkg)
    ''' data = [('smartmontools', 'install ok installed', ..., '6.2+svn3841-1.2'), 
                ('lsof', 'install ok installed', ..., '4.86+dfsg-1ubuntu2')]
    '''
    list_pkgs = [ (i[0], i[-1]) for i in data if i[1].lower() == 'install ok installed' ]
    pkgs = dict(list_pkgs)
    version = pkgs.get(package, None)
    if version is None:
        return (2, "can't find the package %s" % (package))
    return (0, str(version))

def cmp_versions(type, app, version_other, dt_other, version_dpkg, dt_now):
    """
    сравнение версий из dpkg и zookeeper|живого процесса
    """
    logger.debug("[cmp_versions()] {0}: {1} {2} dpkg {3}".format(app, type, version_other, version_dpkg))
    if version_other == version_dpkg:
        return (0, "({0}) OK".format(app))
    if (dt_now-dt_other).total_seconds() <= SECONDS_TO_OK:
        return (0, "({0}) OK".format(app))
    if (dt_now-dt_other).total_seconds() <= SECONDS_TO_WARN:
        return (1, "({0}) versions in {1} and dpkg are different ({1} {2} vs dpkg {3})".format(app, type, version_other, version_dpkg))

    return (2, "({0}) versions in {1} and dpkg are different ({1} {2} vs dpkg {3})".format(app, type, version_other, version_dpkg))


def retry(ExceptionToCheck, tries=3, delay=1, backoff=1):
    """
        декоратор ретраев
    """
    def deco_retry(f):
        @wraps(f)
        def f_retry(*args, **kwargs):
            mtries, mdelay = tries, delay
            for i in range(mtries):
               try:
                  return f(*args, **kwargs)
               except ExceptionToCheck as e:
                  msg = "{0}, Retrying in {1} seconds...".format(str(e), mdelay)
                  logger.debug(str(msg))
                  if mtries == i+1: raise
                  time.sleep(mdelay)
                  mdelay *= backoff
            return f(*args, **kwargs)
        return f_retry
    return deco_retry


@retry(Exception, tries=4)
def run_request(url):
    """ 
    принимает URL адрес. Функция обертка для перезапросов в случае ошибок.
    """
    logger.info('[run_request()] get data from url ' + url)
    req = urllib2.Request(url)
    req.add_header('User-agent', 'deploy-consistency-monitor')
    response = json.loads(urllib2.urlopen(req).read())
    logger.info('[run_request()] response from url ' + str(response))
    return response['version']


def get_version_from_lp(app, port):
    """
    принимает название приложения и порт для http-запроса, возвращает статус выполнения и сообщение.
    В случае успеха (0, <версию главного пакета этого приложения>),
    в случае ошибки (2, <сообщение ошибки>)
    """
    try:
        result = run_request('http://127.0.0.1:{0}/admin?action=version'.format(port))
        status = 0
    except Exception as err:
        result = "{0}: {1}".format(app, str(err))
        status = 2
    return (status, result)


def set_logger(debug=False):
    log_level = logging.DEBUG if debug else logging.CRITICAL
    logger = logging.getLogger('ConsoleLogging')
    logger.setLevel(logging.DEBUG)
    ch = logging.StreamHandler()
    ch.setLevel(log_level)
    logger.addHandler(ch)
    return logger


class ListAction(argparse.Action):
    def __init__(self, option_strings, dest, nargs=None, **kwargs):
        super(ListAction, self).__init__(option_strings, dest, **kwargs)
    def __call__(self, parser, namespace, values, option_string=None):
        setattr(namespace, self.dest, values.split(','))


def parse_options():
    parser = argparse.ArgumentParser(formatter_class=argparse.RawDescriptionHelpFormatter,
                                     epilog="Пример использования:\n\t %(prog)s --apps java-api5,java-intapi"
                                    )
    parser.add_argument('--apps', action=ListAction,
            dest="apps", help="приложения для проверки. Например: --app=java-web,java-api5(default: '')")
    parser.add_argument('--conf', action='store', default=CONFIG,
            dest="conf", help="путь до файла с конфиг. Например: /tmp/java.yaml(default: {0})".format(CONFIG))
    parser.add_argument('-d', '--debug', action='store_true',
            dest="debug", help="debug режим(default: False)")
    parser.add_argument('--monrun', action='store_true',
            dest="monrun", help="сгенерировать monrun строку(default: False)")
    parser.add_argument('--checks', action=ListAction, default="",
            dest="checks", help='''сравнение deb версии с зукиперной(zookeeper), 
                                 сравнение deb версии с запущенно(process). Например: --check=zookeeper,live-process''')
    parser.add_argument('--print', action='store_true',
            dest="print", help="распечатать результат проверки")
    parser.add_argument('--version-test', action='store',
            dest="version_test", help="не проверяем установленную версию deb пакета. Например: --version-test=1.0.3-1")
    parser.add_argument('--scan', action='store_true',
            dest="scan", help="ищем установленные пакеты из {0} и выводим результат".format(CONFIG))
    return vars(parser.parse_args())


def load_apps(conf_file='', filters=[]):
    ''' Принимает путь до конфигурационного файла yaml и отфильтровывает интересующие сервисы.
        При пустом фильтре, выводит весь конфиг. Например:
            { 'java-api5': { 'app_name': 'java-api5',
                             'package_name': 'yandex-direct-api5-java'
                             'zookeeper_version_node': "/direct/versions/java-logviewer"
                             'port_for_lp': "10400"
                           },
             'java-web': { ...
            }
    '''
    logger.debug('[load_apps] func(main) {0}, {1}'.format(conf_file, filters))
    try:
        with open(conf_file, 'r') as f:
            data = yaml.load(f)
    except IOError as err:
        return (2, str(err))

    apps = {}
    logger.debug('[load_apps] func(yaml.load) {0}'.format(data))
    for app in data['apps']:
        if (filters and app in filters or not filters) and 'zookeeper-version-node' in data['apps'][app]:
            apps[app] = {'app_name' : app,
                         'package_name' : data['apps'][app]['package'] if 'package' in data['apps'][app] else '',
                         'zookeeper_version_node' : data['apps'][app]['zookeeper-version-node'],
                         'port_for_lp' : data['apps'][app]['port'] if 'port' in data['apps'][app] else '',
                         'ignore-features': data['apps'][app].get('ignore-features', []),
                        }
    return (0, apps)


def convert_dt_zk(dt_zk):
    return datetime.datetime.strptime(dt_zk.strip()[:-5], "time=%Y-%m-%dT%H:%M:%S")

def monrun(juggler_result):
    ''' функция выводит только один тип проверки, т.к. monrun не поймет многострочники.
    '''
    checks = juggler_result.keys()
    if len(checks) > 1:
        err_code = 1
        err_msg = "ошибка в конфигурации, указано несколько ключей для одной проверки: {0}".format(checks)
    else:
        (err_list_codes, err_list_mgs) = zip(*juggler_result[checks[0]].values())
        err_code = max(err_list_codes)
        err_msg = ','.join(err_list_mgs)
    return '{0};{1}'.format(err_code, err_msg)

def run():

    global logger
    
    options = parse_options()
    logger = set_logger(options['debug'])


    # { 'zookeeper': {'java-api5': (0, "description")}, <- сравнивает zookeeper vs deb
    #   'live-process': {'java-web': (2, "description")} } <- сравнивает live process vs deb
    juggler_result = { 'zookeeper': dict(),
                       'live-process': dict()
    }

    status, message = load_apps(options['conf'], filters=options['apps'])
    if status != 0:
        error = {'common': (status, message)}
        [juggler_result[key].update(error) for key in juggler_result]
        apps = dict()
    else:
        apps = message

    # если указаны проверки в --checks, то отфильтровываем ненужные из juggler_result
    if options['checks']:
        all_keys = juggler_result.keys()
        delta_keys = set(juggler_result.keys()) - set(options['checks'])
        [ juggler_result.pop(i) for i in delta_keys if delta_keys ]

    # устанавливаем соединение с zookeeper. В случае ошибки zkh=None
    zkh = None
    if juggler_result.has_key('zookeeper'):
        (status, result) = zk_sync_init(",".join(zk_servers))
        if status != 0:
            error = {'common': (status, result)}
            juggler_result['zookeeper'].update(error)
        else:
            zkh = result
    
    for app in apps:
        if not options['scan'] and app not in options['apps']:
            continue

        # смотрим установленную версию на сервере. Если пакет не найден, то пишем ошибку и переходим
        # к следующему приложению.  При использовании --scan ошибку не пишем и переходим к следующему приложению.
        (status, message) = get_version_from_dpkg(apps[app]['package_name'])
        logger.debug("[run()] func(get_version_from_dpkg juggler_result) app {} status {} message {}".format(app, status, message))
        if status == 0:
            version_dpkg = message
        elif options['version_test']:
            version_dpkg = options['version_test']
        elif options['apps']:
            error = {app: (status, str(message))}
            [juggler_result[key].update(error) for key in juggler_result]
            continue
        else:
            continue

        # если соединение с zookeeper не установилось, то пропускаем проверку версии через него.
        if juggler_result.has_key('zookeeper') and zkh:
            # если конфиг велит пропускать проверку -- пропускаем
            if 'zk-vs-deb' in apps[app]['ignore-features']:
                error = {app: (0, "%s ignored by apps-config" % app)}
            else:
                (status, message, time) = get_version_time_from_zk(app, apps[app]['zookeeper_version_node'], zkh)
                logger.debug("[run()] func(get_version_time_from_zk) app {} status {} message {} {}".format(app, status, message, time.rstrip()))
                if status == 0:
                    (version_zk, dt_zk) = message, time
                    (status, description) = cmp_versions('ZK',
                                                         app,
                                                         version_zk,
                                                         convert_dt_zk(dt_zk),
                                                         version_dpkg,
                                                         datetime.datetime.now()
                    )
                    error = {app: (status, str(description))}
                    juggler_result['zookeeper'].update(error)
                else:
                    error = {app: (status, str(message))}
            juggler_result['zookeeper'].update(error)

        if juggler_result.has_key('live-process'):
            # если конфиг велит пропускать проверку -- пропускаем
            if 'process-vs-deb' in apps[app]['ignore-features']:
                error = {app: (0, "%s ignored by apps-config" % app)}
            else:
                (status, message) = get_version_from_lp(app, apps[app]['port_for_lp'])
                logger.debug("[run()] func(get_version_from_lp) app {} status {} message {}".format(app, status, message))
                if status == 0:
                    version_lp = message
                    (status, description) = cmp_versions('LP',
                                                         app,
                                                         version_lp,
                                                         datetime.datetime.now() - datetime.timedelta(hours=1),
                                                         version_dpkg,
                                                         datetime.datetime.now()
                    )
                    error = {app: (status, str(description))}
                    juggler_result['live-process'].update(error)
                else:
                    error = {app: (status, str(message))}
            juggler_result['live-process'].update(error)

    if zkh is not None:
        zkh.stop()
        zkh.close()

    # если --scan и словарь пустой, то значит не найдено ни одного приложения. 
    if options['scan']:
        error = {'common': (1, str('not found any packages of {0}').format(options['conf']))}
        [juggler_result[i].update(error) for i in juggler_result if len(juggler_result[i]) == 0]

    logger.debug("[run()] var(juggler_result): {0}".format(juggler_result))
    if options['monrun']: print monrun(juggler_result)
    if options['print']: print juggler_result
    return


if __name__ == '__main__':
    run()

