#!/usr/bin/python
# -*- coding: utf-8 -*-
# vim: set expandtab ts=4 sw=4:
# $Id$

import argparse
import logging
import os
import pipes
import re
import subprocess
import sys
import time
import yaml

DRY_RUN=False


def cmd_to_str(cmd):
    return ' '.join([pipes.quote(s) for s in cmd])

def run_cmd(args, retries=1, relax=0, return_instead_of_exit=False):
    global DRY_RUN
    cmd_str = cmd_to_str(args)

    exit_code = 0
    for i in xrange(retries):
        logging.info(u'running: %s' % cmd_str)
        if DRY_RUN:
            return 0

        exit_code = subprocess.call(args)
        if exit_code == 0:
            return 0
        else:
            if return_instead_of_exit:
                logging.warn(u'command exited with non-zero exit code %s: %s' % (exit_code, cmd_str))
            else:
                logging.error(u'command exited with non-zero exit code %s: %s' % (exit_code, cmd_str))

        if i + 1 != retries:
            logging.info(u"sleep for %d seconds" % relax)
            time.sleep(relax)

    if exit_code != 0 and not return_instead_of_exit:
        sys.exit(exit_code)
    return exit_code

def update_package_list(pkg, version, retries=1, relax=0):
    global DRY_RUN
    update_cmd = ['apt-get', 'update']
    update_cmd_str = cmd_to_str(update_cmd)
    version_exists_cmd = ['apt-cache', 'policy', pkg]
    version_exists_cmd_str = cmd_to_str(version_exists_cmd)

    exit_code = 0
    for i in xrange(retries):
        logging.info(u'running: %s' % update_cmd_str)
        if DRY_RUN:
            return

        exit_code = subprocess.call(update_cmd)
        
        if exit_code == 0:
            try:
                logging.info(u'running: %s' % version_exists_cmd_str)
                output = subprocess.check_output(version_exists_cmd)
                if output.find(version) != -1:
                    return
                else:
                    logging.error(u"not found %s=%s" % (pkg, version))

            except subprocess.CalledProcessError as e:
                logging.error(u'command exited with non-zero exit code %s: %s' % (e.returncode, version_exists_cmd_str))
            except:
                logging.error(u'unexpected error')
        else:
            logging.error(u'command exited with non-zero exit code %s: %s' % (exit_code, version_exists_cmd_str))

        if i + 1 != retries:
            logging.info(u"sleep for %d seconds" % relax)
            time.sleep(relax)

    if exit_code != 0:
        sys.exit(exit_code)
    else:
        sys.exit(1)

def update_packages(pkgs_with_versions):
    global facts
    os.environ['DEBIAN_FRONTEND'] = 'noninteractive'
    apt_get_args = ['{}={}'.format(pkg, pkgs_with_versions[pkg]) for pkg in pkgs_with_versions]
    if ('CONFNEW' in os.environ) or not facts['is_production']:
        apt_get_args += ['-o', 'Dpkg::Options::=--force-confnew', '-o', 'Dpkg::Options::=--force-confmiss']
    if facts['is_ts']:
        apt_get_args += ['--auto-remove', '--force-yes']
    run_cmd(['apt-get', 'install'] + apt_get_args, retries=3, relax=10)

def update_perl_direct(version):
    # TODO использовать facts и update_packages
    if os.path.exists('/var/www/ppc.yandex.ru/protected/monitor/limtest.locked'):
        logging.info(u"locked limtest, exiting")
        return

    # собрать список пакетов для обновления
    dpkg_query_output_lines = subprocess.check_output(['dpkg-query', '--showformat', '${db:Status-Abbrev}\\t${Package}\\n', '--show', 'yandex-direct*']).split('\n')
    installed_yandex_direct_packages = [line.split('\t')[1] for line in dpkg_query_output_lines if re.search(r'^ii', line)]
    non_production_conf_packages = [p for p in installed_yandex_direct_packages if re.match(r'yandex-direct-conf-(sandbox-)?(dev|test|test2|testload)$', p)]
    if len(non_production_conf_packages) > 1:
        logging.error('unexpected: more than 1 non-production conf package installed: ' + ', '.join(non_production_conf_packages))
        sys.exit(1)
    is_production = (len(non_production_conf_packages) == 0)
    is_ts = (not is_production) and (re.search(r'-(sandbox-)?(test|test2)$', non_production_conf_packages[0]))
    skip_packages_re = re.compile('|'.join([
        r'tools-clus',
        r'direct-deploy',
        r'direct-commander',
        r'yandex-direct-zookeeper',
        r'yandex-direct-slb-flapgraphs',
        r'java',
        r'push-client-config',
        r'send-logs-to-logbroker',
        r'dscribe-upload-',
        r'queryrecdata',
        r'yandex-direct-dna',
        r'yandex-direct-db-schema',
        r'tvmtool',
        r'cloud-conf',
        r'cloud-v12-conf',
        r'jdk11',
    ]))
    packages_to_update = [pkg for pkg in installed_yandex_direct_packages if not re.search(skip_packages_re, pkg)]
    if len(packages_to_update) == 0:
        logging.error('no packages to update')
        sys.exit(1)

    # установить пакеты
    update_package_list('yandex-direct', version, retries=3, relax=10)
    os.environ['DEBIAN_FRONTEND'] = 'noninteractive'
    apt_get_args = ['{}={}'.format(pkg, version) for pkg in packages_to_update] + ['--force-yes', '--yes']
    if ('CONFNEW' in os.environ) or (not is_production):
        apt_get_args += ['-o', 'Dpkg::Options::=--force-confnew', '-o', 'Dpkg::Options::=--force-confmiss']
    if is_ts:
        apt_get_args += ['--auto-remove']
    run_cmd(['apt-get', 'install'] + apt_get_args, retries=3, relax=10)
    if not is_production:
        # на непродакшеновых средах выполняется postinst для yandex-direct-conf-*, который переписывает конфиги apache/nginx/... с продакшеновых на тестовые
        # иногда этот postinst может выполниться не вовремя -- до установки новых конфигов, которые перепишут тестовые конфиги продакшеновыми.
        # поэтому костыль: после установки всех пакетов ещё раз выполняем postinst
        run_cmd(['/var/lib/dpkg/info/%s.postinst' % non_production_conf_packages[0], 'configure'])

    # перезагрузить код
    possible_init_d_reload_cmds = [
        ['/etc/init.d/direct-accel', 'reload'],
        ['/etc/init.d/soap-direct-accel', 'reload'],
        ['/etc/init.d/intapi-direct-accel', 'reload'],
        ['/etc/init.d/ppc.yandex.ru', 'reload'],
        ['/etc/init.d/soap.direct.yandex.ru', 'reload'],
        ['/etc/init.d/intapi.direct.yandex.ru', 'reload'],
        ['/etc/init.d/fake-services.direct.yandex.ru', 'reload'],
    ]
    for cmd in possible_init_d_reload_cmds:
        if os.path.exists(cmd[0]):
            if not is_production:
                # на непродакшеновых средах лучше делать restart, а не reload, т. к. apache растут по памяти, автоматики по перезапуску как в продакшене нет, а даунтайм не критичен
                cmd[1] = 'restart'
            run_cmd(cmd)
    if os.path.exists('/etc/service/js-templater'):
        run_cmd(['sv', 'restart', 'js-templater'])

def update_java_app(app, version):
    global facts
    is_production = facts['is_production']
    is_sandbox = facts['is_sandbox']
    is_ts = facts['ya_environment_type'] == 'testing'
    pkg = facts['apps_conf']['apps'][app]['package']

    service_restart_timeout = 30

    if pkg not in facts['installed_packages']:
        logging.error('no packages to update')
        sys.exit(1)

    # обновить пакеты
    update_package_list(pkg, version)
    update_packages({pkg: version})

    # собрать список того, что нужно перезапускать
    file_list = subprocess.check_output(['dpkg-query', '--listfiles', pkg]).rstrip().split('\n')
    sv_dirs = [f for f in file_list if re.match(r'^/etc/sv/[^/]+$', f)]
    nginx_init_files = [f for f in file_list if re.match(r'^/etc/init.d/nginx-[^/]+$', f)]
    nginx_config_files = [f for f in file_list if re.match(r'^/etc/nginx/direct-[^/]+$', f)]
    if len(sv_dirs) > 1:
        logging.error("unexpected: more than 1 sv directory in package:\n%s" % '\n'.join(sv_dirs))
        sys.exit(1)
    if len(nginx_init_files) > 1:
        logging.error("unexpected: more than 1 nginx init script in package:\n%s" % '\n'.join(nginx_init_files))
        sys.exit(1)
    if len(nginx_config_files) > 1:
        logging.error("unexpected: more than 1 nginx config file in package:\n%s" % '\n'.join(nginx_config_files))
        sys.exit(1)
    sv_dir = sv_dirs[0]
    nginx_init_file = nginx_init_files[0]
    nginx_config_file = nginx_config_files[0]

    need_to_restart_service = sv_dir is not None
    # на песочнице запросы проксируются прямо в jetty, nginx не нужен
    need_to_restart_nginx = nginx_init_file is not None and not is_sandbox
    need_graceful_restart = need_to_restart_nginx and is_production

    # заменить конфиг nginx на тестовый
    if not is_production and nginx_config_files is not None:
        # закомментировать продакшеновые настройки
        run_cmd(['sed', '-i', r's/^\(\s*\)\(.*###.*environment:production.*\)/\1#\2/', nginx_config_file])
        # раскомментировать непродакшеновые настройки
        run_cmd(['sed', '-i', r's/\(\s*\)#\(.*###.*ya_environment:%s.*\)/\1\2/' % facts['ya_environment_type'], nginx_config_file])

    if need_to_restart_nginx:
        nginx_service_name = os.path.basename(nginx_init_file)
        nginx_pid_file = '/var/run/{}.pid'.format(nginx_service_name)
        run_cmd(['pkill', '--signal', 'QUIT', '--pidfile', nginx_pid_file], return_instead_of_exit=True)
        if need_graceful_restart:
            time.sleep(20)
    if need_to_restart_service:
        restart_return_code = run_cmd(['sv', '-w', str(service_restart_timeout), 'restart', sv_dir], return_instead_of_exit=True)
    if need_to_restart_nginx:
        run_cmd([nginx_init_file, 'restart'])
        if need_graceful_restart:
            time.sleep(5)   # чтобы балансер успел осознать инстанс как живой
    if need_to_restart_service and restart_return_code != 0:
        logging.error('service restart failed with code {}'.format(restart_return_code))
        sys.exit(1)

def update_db_schema(version):
    # TODO использовать facts, update_package_list, update_packages
    run_cmd(['apt-get', 'update'])
    os.environ['DEBIAN_FRONTEND'] = 'noninteractive'
    apt_get_args = ['yandex-direct-db-schema={}'.format(version)]
    run_cmd(['apt-get', 'install'] + apt_get_args, retries=3, relax=10)

def get_facts():
    facts = {}
    dpkg_query_output_lines = subprocess.check_output(['dpkg-query', '--showformat', '${db:Status-Abbrev}\\t${Package}\\n', '--show']).rstrip().split('\n')
    installed_packages = [line.split('\t')[1] for line in dpkg_query_output_lines if line.startswith('ii')]
    facts['installed_packages'] = installed_packages
    facts['is_sandbox'] = len([p for p in installed_packages if p.startswith('yandex-direct-conf-sandbox')]) > 0

    ya_environment_type = open('/etc/yandex/environment.type', 'r').read().rstrip()
    facts['ya_environment_type'] = ya_environment_type
    facts['is_production'] = ya_environment_type == 'production'
    facts['is_ts'] = ya_environment_type == 'testing'

    apps_conf_yaml = open('/etc/yandex-direct/direct-apps.conf.yaml', 'r').read()
    facts['apps_conf'] = yaml.safe_load(apps_conf_yaml)

    return facts

def run():
    global facts
    facts = get_facts()
    sequence_for_app = {
        'direct': update_perl_direct,
        'db-schema': update_db_schema,
    }
    java_apps = []
    for app in facts['apps_conf']['apps']:
        # пока не умеем выкладывать java-b2yt, пропускаем
        if facts['apps_conf']['apps'][app]['type'] == 'arcadia-java' and app not in ['java-b2yt'] + sequence_for_app.keys():
            java_apps.append(app)
    parser = argparse.ArgumentParser()
    parser.add_argument('app_name', help=u'условное имя приложения, одно из: '+', '.join(sequence_for_app.keys() + java_apps))
    parser.add_argument('version')
    parser.add_argument('--dry-run', action='store_true')
    args = parser.parse_args()
    global DRY_RUN
    DRY_RUN = args.dry_run
    if args.app_name not in sequence_for_app.keys() + java_apps:
        sys.exit("error: unknown app '%s'" % args.app_name)
    logging.basicConfig(level=logging.INFO, format='### %(asctime)s [%(levelname)s] %(message)s', datefmt='%H:%M:%S')
    logging.info('START')
    if args.app_name in java_apps:
        update_java_app(args.app_name, args.version)
    else:
        sequence_for_app[args.app_name](args.version)
    logging.info('FINISH')

if __name__ == '__main__':
    run()
