#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Выгружает списки проверок по всем приложениям из direct-apps.conf
Ревизии определяются по версиям приложений в zk direct-versions

Игнорирует приложения с juggler-events в ignore-features
Игнорирует приложения с пустой версией в zookeeper

Поддерживает дополнительные параметры в apps-config:

juggler-events-dir
    директория (относительно общепринятого корня (trunk/arcadia), например direct/intapi/juggler-events)
    если не указано, то primary-svn-dir + '/juggler-events'

juggler-events-ver
    переопределение версии приложения (строка, например, 1.HEAD) из которой брать списки проверок
"""

from __future__ import print_function

import argparse
import json
import logging
import os
import sys
import shutil
import yaml
from subprocess import Popen, check_output, PIPE, STDOUT
from tempfile import mkdtemp

APP_TYPES = ['svn-perl', 'arcadia-java']
TARGET_BY_NAMESPACE = {
    'direct.prod' : 'prod',
    'direct.test' : 'test',
    'direct.dev'  : 'devtest',
    'direct.dev7' : 'dev7'
}

ENV_BY_NAMESPACE = {
    'direct.prod' : 'production',
    'direct.test' : 'testing',
    'direct.dev'  : 'devtest',
    'direct.dev7' : 'dev7'
}

SCRIPT_NAME = 'direct-apps-juggler-make-config'

#Дефолтный набор тегов для непродакшеновых сред
DEFAULT_NP_TAGS = ["direct-appduty-tv_np"]

# svn_app - приложение из direct-apps, для arcadia-java svn_app будет вида web, api5, etc
APP_TYPE_TO_SVN = {
    'svn-perl': {
        'releases': 'svn.yandex.ru/{svn_app}/releases/release-{base_rev}',
        'trunk': 'svn.yandex.ru/{svn_app}/trunk',
    },
    'arcadia-java': {
        'releases': 'arcadia.yandex.ru/arc/branches/direct/release/{svn_app}/{base_rev}/arcadia',
        'trunk': 'arcadia.yandex.ru/arc/trunk/arcadia',
    },
}

def die(msg):
    logging.info(msg + ' - exiting')
    sys.exit(1)


def read_file(fname):
    with open(fname, 'r') as f:
        return f.read()


def get_app_ver(app):
    logging.info('app %s: get version' % (app,))
    out = check_output(['direct-version-get', app])
    logging.info('app %s: zk data:\n%s' % (app, out))
    ver = out.split('\n')[0].strip()
    return ver


def get_events_dir(app_conf):
    if app_conf.get('juggler-events-dir'):
        return app_conf['juggler-events-dir']
    elif app_conf.get('primary-svn-dir'):
        return app_conf['primary-svn-dir'] + '/juggler-events'
    else:
        return 'juggler-events'


def get_app_svn_path(app, app_conf):
    if app_conf.get('type') not in APP_TYPES:
        die('unsupported app type')

    ver = None
    if app_conf.get('juggler-events-ver'):
        ver = app_conf['juggler-events-ver']
    else:
        ver = get_app_ver(app)
    logging.info('app %s: selected version: %s' % (app, ver))
    ver_parts = ver.split('-')[0].split('.')

    base_rev = None
    rev = ver_parts[-1]
    if len(ver_parts) > 2:
        base_rev = ver_parts[-2]

    branch = 'trunk'
    if base_rev is not None:
        branch = 'releases'

    opts = {
        'svn_app': app,
        'rev': rev,
        'base_rev': base_rev,
    }
    opts['prefix'] = APP_TYPE_TO_SVN[app_conf['type']][branch].format(**opts)
    opts['suffix'] = app_conf['juggler-events-dir']

    return '{prefix}/{suffix}'.format(**opts), opts['rev']


def make_svn_url(svn_path, svn_rev, svn_user):
    return 'svn+ssh://%s@%s/@%s' % (svn_user, svn_path, svn_rev)


def get_app_events(svn_path, svn_rev, svn_user, svn_binary, env, target):
    svn_base = os.path.dirname(svn_path.strip('/'))
    svn_base_url = make_svn_url(svn_base, svn_rev, svn_user)

    logging.info('svn: get checks from path %s, rev %s' % (svn_path, svn_rev))

    # хочется различать ситуации, когда svn не работает - тогда не генерим конфиг
    # и когда отсутствует juggler-events-dir в родительской директории (svn_base) - тогда удаляем все проверки этого приложения
    cmd = [svn_binary, 'list', svn_base_url]
    logging.info('svn: run cmd ' + ' '.join(cmd) + ' with env ' + str(env))
    out = check_output([svn_binary, 'list', svn_base_url], env=env)
    logging.info('svn: cmd output:\n' + out)
    base_dirs = [ x.strip().strip('/') for x in out.split('\n') ]

    events_dir = os.path.basename(svn_path.strip('/'))
    if events_dir not in base_dirs:
        logging.info('svn: skipping app, no events dir "%s" in %s from %s' % (events_dir, base_dirs, svn_base))
        return []

    # скачиваем и парсим конфиги проверок
    cwd = mkdtemp()
    path = cwd + '/events'

    svn_url = make_svn_url(svn_path, svn_rev, svn_user)
    cmd = [svn_binary, 'export', svn_url, path]
    logging.info('svn: run cmd ' + ' '.join(cmd) + ' with env ' + str(env))
    p = Popen(cmd, stdout=PIPE, stderr=PIPE, cwd=cwd, env=env)
    out, err = p.communicate()
    logging.info('svn: cmd finished with retcode: %s\nstdout:\n%s\nstderr:\n%s' % (p.returncode, out, err))

    if p.returncode != 0:
        shutil.rmtree(cwd)
        die('bad svn return code')

    app_events = []
    for item in os.listdir(path):
        fpath = path + '/' + item
        logging.info('processing file ' + fpath)
        if not os.path.isfile(fpath) or not fpath.endswith('.yaml'):
            logging.info('skipping file ' + fpath)
            continue

        conf = yaml.load(read_file(fpath))
        logging.info('read conf ' + jdumps(conf))

        #Для непродакшеновых сред вместо tags используем tags_np,
        #а если tags_np - ставим дефолт
        if target != 'prod':
            for event in conf:
                event['tags'] = event.pop('tags_np', DEFAULT_NP_TAGS)

        app_events.extend(conf)

    shutil.rmtree(cwd)

    return app_events


def jdumps(jdict, pretty=False):
    if pretty:
        return '\n' + json.dumps(jdict, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '))
    return json.dumps(jdict, sort_keys=True, ensure_ascii=False)


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument('-c', '--config', required=True, help='конфиг direct-apps.conf')
    parser.add_argument('-o', '--output', help='писать результирующий конфиг с проверками в файл (иначе - в stdout)')
    parser.add_argument('-u', '--svn-user', required=True, help='пользователь, с которым ходить в svn')
    parser.add_argument('-k', '--svn-key', required=True, help='путь до ключа, с которым ходить в svn')
    parser.add_argument('-s', '--svn-binary', default='/usr/bin/svn', help='путь к бинарнику svn')
    parser.add_argument('-n', '--namespace', default='direct.prod', help='namespace, для которого генерируем конфиг')
    args = parser.parse_args()

    return args


def main():
    args = parse_args()

    logging.basicConfig(level=logging.DEBUG, stream=sys.stderr,
                        format='[%(asctime)s]\t%(message)s')
    logging.info('running with args: ' + jdumps(vars(args)))

    apps_conf = yaml.load(read_file(args.config))
    logging.info('running with config: ' + jdumps(apps_conf))

    namespace = args.namespace

    target = TARGET_BY_NAMESPACE.get(namespace)
    environ = ENV_BY_NAMESPACE.get(namespace)
    if target == None:
        die('[' + SCRIPT_NAME + ']' + ' unknown namespace: ' + namespace)

    if target == 'prod':
        conductor_groups_section_name = 'conductor_groups'
    else:
        conductor_groups_section_name = 'conductor_groups' + '_' + target;

    conf = []
    for app, app_conf in apps_conf.get('apps', {}).iteritems():
        if app_conf.get('type') not in APP_TYPES or \
                'juggler-events' in app_conf.get('ignore-features', []):
            logging.info('app %s: unknown app type or juggler-events feature ignored, skipping' % (app,))
            continue

        app_conf['juggler-events-dir'] = get_events_dir(app_conf)

        logging.info('app %s: get juggler-events svn path' % (app,))
        svn_path, svn_rev = get_app_svn_path(app, app_conf)
        if not svn_rev:
            logging.warning('app %s: cannot get svn revision, skipping' % (app,))
            continue

        env = {'SVN_SSH': 'ssh -i ' + args.svn_key}
        events = get_app_events(svn_path, svn_rev, args.svn_user, args.svn_binary, env, target)

        conductor_groups = app_conf.get(conductor_groups_section_name, [])

        stages = app_conf.get('yadeploy-stages', {}).get(environ, [])
        if not isinstance(stages, list):
            stages = [{'stage': stages}]
        yadeploy_stages = [item['stage'] for item in stages]

        conf.append({
            'app-name': app,
            'namespace': namespace,
            'conductor_groups': conductor_groups,
            'yadeploy-stages': yadeploy_stages,
            'juggler-events-dir': app_conf.get('juggler-events-dir'),
            'juggler-events': events,
        })

    if args.output:
        with open(args.output, 'w') as f:
            json.dump(conf, f, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '))
    else:
        print(jdumps(conf))

if __name__ == '__main__':
    main()
