#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Читает конфиг со списком приложений и предоставляемых juggler-raw-events
Заводит над ними агрегаты по определенным правилам
"""

from __future__ import print_function

import argparse
import sys
import traceback

sys.path.extend(['share/direct-apps-juggler', '/usr/local/share/direct-apps-juggler'])
from dt_apps_juggler.common import *

from dt_apps_juggler.juggler_sdk_helpers import hash_merge, apply_checks
from juggler_sdk import Check, Child, NotificationOptions

# используется для отправки событий о глобальном статусе накатывания проверок
# на это имя завязана juggler_mark - при его изменении старые проверки нужно будет удалить руками
SCRIPT_NAME = 'direct-apps-juggler'

APP_SCHEMA = {
    'type': 'object',
    'additionalProperties': True,
    'required': ['app-name', 'juggler-events-dir', 'conductor_groups'],
    'properties': {
        'app-name': { 'type': 'string', 'minLength': 1 },
        'juggler-events-dir': { 'type': 'string', 'minLength': 1 },
        'conductor_groups': { 'type': 'array'},
        'yadeploy-stages': {'type': 'array'},
        'juggler-events': { 'type': 'array' }, # может быть пустым, может отсутствовать - это легальный способ удалить все проверки
    }
}

EVENT_SCHEMA = {
    'type': 'object',
    'additionalProperties': True,
    'required': ['service'],
    'properties': {
        'service': { 'type': 'string', 'minLength': 1 },
        'ok_if': { 'enum': ['any_ok', 'all_ok'] },
        'tags': { 'type': 'array', 'minItems': 1, 'items': { 'type': 'string', 'minLength': 1 } },
        'ttl': { 'type': 'integer' },
        'sharded': { 'type': 'string' },
    }
}

DEFAULT_CHECK = {
    'description': SCRIPT_NAME + ' auto check',
    'aggregator': 'logic_or',
    'ttl': 60*15,
    'tags': [SCRIPT_NAME],
    'notifications': [
        NotificationOptions(template_name='push', template_kwargs=dict(push_url='http://juggler-history.da.yandex.ru/save_notifications'), description=u'сервис для хранения истории по проверкам'),
        ],
}

YADEPLOY_PROJECTS = {
    'direct.prod': 'direct-production',
    'direct.test': 'direct-np',
    'direct.dev': 'direct-np',
}

ENV_FILE = '/etc/yandex/environment.type'

def get_shards_num(dbname):
    dbconfig = json.loads(read_file('/etc/yandex-direct/db-config.json'))
    if dbname == 'ppc':
        return len(dbconfig['db_config']['CHILDS']['ppc']['CHILDS'].keys())

    return 0


def make_app_checks(app_conf, namespace):
    checks = []
    app = app_conf['app-name']
    provided_events = app_conf['juggler-events']
    cgroups  = app_conf.get('conductor_groups', [])

    yd_stages = app_conf.get('yadeploy-stages')
    #yd_prj = YADEPLOY_PROJECTS.get(namespace)
    yd_prj = 'direct-production' if namespace == 'direct.prod' else 'direct-np'
    default_check = DEFAULT_CHECK
    if namespace != 'direct.prod':
        default_check['notifications'] = []

    for event in provided_events:
        children = []
        children.extend([Child(host=x, group_type='CGROUP', service=event['service']) for x in cgroups])
        children.extend([Child(host='%s@stage=%s' % (yd_prj, x), group_type='DEPLOY', service=event['service']) for x in yd_stages])
        if len(children) == 0:
            continue
        params = {
            'host': namespace + '_' + app,
            'namespace': namespace,
            'children': children,
            'tags': list(set([app] + event.get('tags', []) + default_check.get('tags', []))),
        }
        aggr = event.pop('ok_if', None)
        if aggr == 'any_ok':
            params['aggregator'] = 'logic_and'
            params['aggregator_kwargs'] = {'downtimes_mode': 'ignore'}
        elif aggr == 'all_ok':
            params['aggregator'] = 'logic_or'
        # иначе будет то, что в DEFAULT_CHECK

        dbname = event.pop('sharded', 'none')
        shards_num = get_shards_num(dbname)
        if shards_num > 0:
            aggr_params = copy.deepcopy(params)
            aggr_params['children'] = []
            aggr_params['aggregator'] = 'logic_or' # общая проверка пока всегда с logic_or

            params['tags'] = [SCRIPT_NAME + '.aggregation_child']
            for shard in range(1, shards_num + 1):
                params['service'] = '%s.%s_%d' % (event['service'], dbname, shard)
                cchildren = []
                cchildren.extend([Child(host=x, group_type='CGROUP', service=params['service']) for x in cgroups])
                cchildren.extend([Child(host='%s@stage=%s' % (yd_prj, x), group_type='DEPLOY', service=params['service']) for x in yd_stages])
                if len(cchildren) == 0:
                    continue
                params['children'] = cchildren
                checks.append(Check(**hash_merge(default_check, event, params)))

                aggr_params['children'].append(
                    Child(host=aggr_params['host'], group_type='HOST', service=params['service'])
                )

            checks.append(Check(**hash_merge(default_check, event, aggr_params)))
        else:
            checks.append(Check(**hash_merge(default_check, event, params)))

    return checks


def ya_env_to_namespace(ya_env):
    if ya_env == 'production':
        return 'direct.prod'
    elif ya_env == 'testing' or ya_env == 'prestable':
        return 'direct.test'
    elif ya_env == 'development':
        return 'direct.dev'
    else:
        logging.info('unknown yandex-environment %s, cannot find relevant juggler namespace' % (ya_env,))

    return None


def get_ya_env():
    with open(ENV_FILE, 'r') as f:
        return f.readline().rstrip()


def prepare_config(apps_conf, namespace):
    """
    принимает список вида
    [ { 'app-name': 'java-web', 'juggler-events': [ { 'service': ..., } ], 'conductor_groups': [ ... ] } ]
    проверяет конфиг на повторные вхождения приложений и событий,
    убирает лишние ключи и все, что может помешать заведению проверок
    возвращает очищенный конфиг в исходном формате
    """
    conf = {}
    err = False
    for app_conf in apps_conf:
        logging.debug('processing app_conf: ' + jdumps(app_conf))

        # если не подошла схема app-config - падаем
        check_schema_or_die(app_conf, APP_SCHEMA)

        conf_namespace = app_conf.get('namespace')
        if conf_namespace != None and conf_namespace != namespace:
            die(SCRIPT_NAME, 'config namespace and target namespace differs: %s -> %s' % (conf_namespace, namespace))

        if app_conf['app-name'] in conf: # app-name тут уже точно есть, проверили в схеме
            logging.warn('duplicate app %s in %s, ignoring' % (app_conf['app-name'], app_conf['juggler-events-dir']))
            err = True
            continue

        app = app_conf['app-name']
        conf[app] = {k: app_conf.get(k) for k in ('app-name', 'juggler-events-dir', 'conductor_groups', 'yadeploy-stages')}
        conf[app]['juggler-events'] = []

        processed = set()
        for evt in app_conf.get('juggler-events', []): # пустой список событий - это ок, удалим все =)
            logging.debug('processing event: ' + jdumps(evt))

            check_schema_or_die(evt, EVENT_SCHEMA)
            if evt['service'] in processed:
                logging.warn('duplicate service %s in %s:%s, ignoring' % (evt.get('service'), app, app_conf['juggler-events-dir']))
                err = True
                continue

            event, event_err = filter_schema_keys(evt, EVENT_SCHEMA)
            # незначительные ошибки при подготовке конфига события (лишние ключи и тд) - не падаем, но зажигаем мониторинг
            if event_err:
                err = True

            processed.add(event['service'])
            conf[app]['juggler-events'].append(event)

    return conf.values(), err


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument('--apply', default=False, action='store_true', help='применить все изменения в juggler')
    parser.add_argument('-f', '--force', default=False, action='store_true', help='действительно применять все изменения в juggler при работе не с продовой машинки(ppcback*)')
    parser.add_argument('-n', '--namespace', default='direct.dev', help='для указания namespace')
#    parser.add_argument('-e', '--ya-env', default='development', help='окружение yandex-environment (используется для выбора namespace)')
    parser.add_argument('-t', '--token', default='/etc/direct-tokens/juggler_api', help='токен для доступа в juggler-api')
    parser.add_argument('-c', '--config', required=True,
        help='конфиг с описанием предоставляемых приложениями проверок (см. пример в share/direct-apps-juggler/examples/direct-apps-juggler.conf.json)')
    parser.add_argument('-m', '--max-config-age', type=int, default=3600, help='допустимый возраст конфига (секунд)')
    args = parser.parse_args()

    return args


def main():
    args = parse_args()
    init_logger()
    logging.debug('running with args: ' + jdumps(args))

#    namespace = ya_env_to_namespace(args.ya_env) - все проверки заводим с ppback-ов
    namespace = args.namespace
    ya_env = get_ya_env()
    if not namespace:
        die(SCRIPT_NAME, 'bad namespace', not args.apply)

    if not check_file_age(args.config, args.max_config_age):
        die(SCRIPT_NAME, 'config file is too old or does not exist', not args.apply)

    if ya_env != 'production' and not args.force:
        die(SCRIPT_NAME, 'use --force in non-production environment', not args.apply)

    try:
        apps_conf, conf_err = prepare_config(json.loads(read_file(args.config)), namespace)
        logging.debug('running with config: ' + jdumps(apps_conf))

        checks = []
        logging.info('making checks ...')
        for app_conf in apps_conf:
            logging.debug('making checks for app ' + app_conf['app-name'])
            checks.extend(make_app_checks(app_conf, namespace))

        logging.info('applying juggler checks ...')
        apply_checks(checks,
                     token=read_file(args.token).strip(),
                     mark=namespace + '_' + SCRIPT_NAME,
                     dry_run=not args.apply)
    except Exception as e:
        die(SCRIPT_NAME, '\n%s\ncannot apply checks: %s %s' % (traceback.format_exc(), type(e), e), not args.apply)

    report_status_to_juggler(SCRIPT_NAME, conf_err, not args.apply)
    if args.apply:
        logging.info('completed successfully')
    else:
        logging.info('no checks configured, run with --apply')


if __name__ == '__main__':
    main()
