#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Читает yaml-конфиги для различных direct-apps-juggler скриптов из svn,
склеивает их в общий json конфиг для конкретного скрипта
"""

from __future__ import print_function

import argparse
import json
import logging
import os
import sys
import shutil
import yaml
from subprocess import Popen, check_output, PIPE, STDOUT
from tempfile import mkdtemp


def die(msg):
    logging.info(msg + ' - exiting')
    sys.exit(1)


def read_file(fname):
    with open(fname, 'r') as f:
        return f.read()


def jdumps(jdict, pretty=False):
    if pretty:
        return '\n' + json.dumps(jdict, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '))
    return json.dumps(jdict, sort_keys=True, ensure_ascii=False)


def make_svn_url(svn_path, svn_rev, svn_user):
    url = 'svn+ssh://'
    if svn_user:
        url += svn_user + '@'
    url += svn_path
    if svn_rev:
        url += '@' + svn_rev
    return url


def get_configs(svn_path, svn_rev, svn_user, svn_binary, env):
    logging.info('svn: get configs from path %s, rev %s' % (svn_path, svn_rev))

    # скачиваем и парсим конфиги проверок
    cwd = mkdtemp()
    path = cwd + '/configs'
 
    svn_url = make_svn_url(svn_path, svn_rev, svn_user)
    cmd = [svn_binary, 'export', svn_url, path]
    logging.info('svn: run cmd ' + ' '.join(cmd) + ' with env ' + str(env))
    p = Popen(cmd, stdout=PIPE, stderr=PIPE, cwd=cwd, env=env)
    out, err = p.communicate()
    logging.info('svn: cmd finished with retcode: %s\nstdout:\n%s\nstderr:\n%s' % (p.returncode, out, err))

    if p.returncode != 0:
        shutil.rmtree(cwd)
        die('bad svn return code')

    configs = []
    for item in os.listdir(path):
        fpath = path + '/' + item
        logging.info('processing file ' + fpath)
        if not os.path.isfile(fpath) or not fpath.endswith('.yaml'):
            logging.info('skipping file ' + fpath)
            continue

        conf = yaml.load(read_file(fpath))
        logging.info('read conf ' + jdumps(conf))
        if conf:
            configs.extend(conf)

    shutil.rmtree(cwd)

    return configs


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument('-o', '--output', help='писать результирующий конфиг с проверками в файл (иначе - в stdout)')
    parser.add_argument('-p', '--svn-path', required=True, help='')
    parser.add_argument('-u', '--svn-user', help='пользователь, с которым ходить в svn')
    parser.add_argument('-k', '--svn-key', help='путь до ключа, с которым ходить в svn')
    parser.add_argument('-s', '--svn-binary', default='/usr/bin/svn', help='путь к бинарнику svn')
    args = parser.parse_args()

    return args


def main():
    args = parse_args()

    logging.basicConfig(level=logging.DEBUG, stream=sys.stderr,
                        format='[%(asctime)s]\t%(message)s')
    logging.info('running with args: ' + jdumps(vars(args)))

    env = None
    if args.svn_key:
        env = {'SVN_SSH': 'ssh -i ' + args.svn_key}
    conf = get_configs(args.svn_path, 'HEAD', args.svn_user, args.svn_binary, env)

    if args.output:
        with open(args.output, 'w') as f:
            json.dump(conf, f, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '))
    else:
        print(jdumps(conf, True))

if __name__ == '__main__':
    main()
