#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Проверка успешности выполнения тасков, созданных sandbox scheduler'ами
OK - когда есть success таска с now - finished <= ok_interval,
даже если за этот период были failed таски.

Manual таски не учитываются, только запущенные шедулером.
"""

import argparse
import sys
import traceback
from datetime import datetime as dt

sys.path.extend(['share/direct-apps-juggler', '/usr/local/share/direct-apps-juggler'])
from dt_apps_juggler.common import *

from dt_apps_juggler.juggler_sdk_helpers import hash_merge, apply_checks
from juggler_sdk import Check, Child, NotificationOptions

from sandbox.common import rest
from sandbox.common.proxy import OAuth

# используется для отправки событий о глобальном статусе накатывания проверок
# на это имя завязана juggler_mark - при его изменении старые проверки нужно будет удалить руками
SCRIPT_NAME = 'dt-sb-sched-mon'

SCHED_SCHEMA = {
    'type': 'object',
    'additionalProperties': True,
    'required': ['env', 'name', 'ok_interval', 'sb_sched_id'],
    'properties': {
        'env': { 'enum': ['prod', 'test', 'dev'] },
        'name': { 'type': 'string', 'minLength': 1 },
        'ok_interval': { 'type': 'integer' },
        'sb_sched_id': { 'type': 'integer' },
        'tags': { 'type': 'array', 'minItems': 1, 'items': { 'type': 'string', 'minLength': 1 } },
    }
}

DEFAULT_CHECK = {
    'description': SCRIPT_NAME + ' auto check',
    'aggregator': 'logic_or',
    'ttl': 5*60*15, # 5 мин в cron.d
    'tags': [SCRIPT_NAME],
    'notifications': [
        NotificationOptions(template_name='push', template_kwargs=dict(push_url='http://juggler-history.da.yandex.ru/save_notifications'), description=u'сервис для хранения истории по проверкам'),
    ],
}



def make_juggler_checks(args):
    token = read_file(args.juggler_token).strip()

    checks = []
    for conf in args.config:
        namespace = 'direct.' + conf['env']
        service = sched_to_juggler_service(conf)
        params = {
            'host': namespace + '_sandbox',
            'namespace': namespace,
            'service': service,
            'children': [Child(host='direct_backs', group_type='CGROUP', service=service)],
            'tags': list(set(conf.get('tags', []) + [conf['name']] + ['sb_sched_' + str(conf['sb_sched_id'])] + DEFAULT_CHECK.get('tags', []))),
        }
        params['aggregator'] = 'logic_and'

        checks.append(Check(**hash_merge(DEFAULT_CHECK, params)))

    logging.info('applying juggler checks ...')
    apply_checks(checks, token=token, mark=SCRIPT_NAME,
                 dry_run=not args.apply)


def check_schedulers(args):
    token = read_file(args.sandbox_token).strip()

    events = []
    for conf in args.config:
        logging.info('check scheduler ' + jdumps(conf))
        events.append(check_scheduler(conf, token))

    logging.info('juggler queue event: ' + jdumps(events))
    if not args.apply:
        return

    res = dj.queue_events(events)
    logging.info('juggler queue event response: ' + jdumps(res))


def check_scheduler(conf, token):
    event = {
        'status': 'CRIT',
        'service': sched_to_juggler_service(conf),
    }
    try:
        client = rest.Client(auth=OAuth(token))

        #sched = client.scheduler.read(id=conf['sb_sched_id'], limit=1)['items'][0]
        last_task = client.task.read(
            #type=sched['task']['type'],
            scheduler=[conf['sb_sched_id']],
            order='-created',
            status='SUCCESS',
            limit=1,
        )['items'][0]
        logging.info('last success task for sched_id %d: %s' % (conf['sb_sched_id'], jdumps(last_task)))

        last_task_finished = dt.strptime(last_task['execution']['finished'][:19], '%Y-%m-%dT%H:%M:%S')
        sec_since_last_ok = (dt.utcnow().replace(microsecond=0) - last_task_finished).total_seconds()

        event['description'] = 'last success %ds ago (%s), ok_interval %ds' % (sec_since_last_ok, last_task_finished, conf['ok_interval'])
        if sec_since_last_ok <= conf['ok_interval']:
            event['status'] = 'OK'
    except Exception as e:
        event['description'] = '%s %s' % (type(e), e)

    logging.info('scheduler checked: ' + jdumps(event))
    return event


def sched_to_juggler_service(conf):
    return conf['name'] + '.' + 'sb_sched_' + str(conf['sb_sched_id'])


def prepare_config(raw_conf):
    conf = {}
    err = False

    for item in raw_conf:
        logging.debug('processing conf item: ' + jdumps(item))
        # если не подошла схема - падаем, зажигаем общий мониторинг
        check_schema_or_die(item, SCHED_SCHEMA)

        name = sched_to_juggler_service(item)
        # если такой уже есть - добавляем только первый, зажигаем общий мониторинг
        if name in conf:
            logging.warning('duplicate item {}, ignoring'.format(name))
            err = True
            continue

        conf[name], item_err = filter_schema_keys(item, SCHED_SCHEMA)
        if item_err:
            logging.warning('odd keys in alert {name}, ignoring'.format(**item))
            err = True

    return conf.values(), err


ACTIONS = {
    'check_schedulers': check_schedulers,
    'make_juggler_checks': make_juggler_checks,
}


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument('--apply', default=False, action='store_true', help='применять все изменения в juggler (без этого никаких пишущих действий не производится)')
    parser.add_argument('--action', choices=ACTIONS.keys(), help='что делаем (проверяем работу зашедуленных заданий, заводим агрегаты в juggler или репортим OK в working-мониторинг скрипта)')
    parser.add_argument('-c', '--config', required=True, help='путь до конфига со списком проверяемых шедулеров')
    parser.add_argument('-s', '--sandbox-token', help='путь до sandbox-token')
    parser.add_argument('-j', '--juggler-token', default='/etc/direct-tokens/juggler_api', help='токен для доступа в juggler-api')
    parser.add_argument('-m', '--max-config-age', type=int, default=3600, help='допустимый возраст конфига (секунд)')
    args = parser.parse_args()

    return args


def main():
    args = parse_args()
    init_logger()
    logging.debug('running with args: ' + jdumps(args))

    if not check_file_age(args.config, args.max_config_age):
        die(SCRIPT_NAME, 'config file is too old or does not exist', not args.apply)

    if not args.action in ACTIONS:
        die(SCRIPT_NAME, 'unknown action, see help', not args.apply)

    try:
        args.config, conf_err = prepare_config(json.loads(read_file(args.config)))
        logging.info('running with config: ' + jdumps(args.config))

        ACTIONS[args.action](args)
    except Exception as e:
        die(SCRIPT_NAME, '\n%s\ncannot make solomon alerts: %s %s' % (traceback.format_exc(), type(e), e), not args.apply)

    report_status_to_juggler(SCRIPT_NAME, conf_err, not args.apply)
    if args.apply:
        logging.info('completed successfully')
    else:
        logging.info('no alerts configured, run with --apply')


if __name__ == '__main__':
    main()
