#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Генератор solomon-alert'ов. Конфиг генерируется отдельным скриптом.

В solomon могут быть алерты с одинаковым id в разных проектах
У алертов с разными id могут быть одинаковые name, даже в пределах одного проекта

В конфигах пока пытаемся поддерживать id == name
"""

from __future__ import print_function

import argparse
import requests
import sys
import traceback

sys.path.extend(['share/direct-apps-juggler', '/usr/local/share/direct-apps-juggler'])
from dt_apps_juggler.common import *

from dt_apps_juggler.juggler_sdk_helpers import hash_merge, apply_checks
from juggler_sdk import Check, Child, NotificationOptions

# используется для отправки событий о глобальном статусе накатывания проверок
# для выборки и удаления solomon-alert'ов, созданных через этот скрипт
# на это имя завязана juggler_mark - при его изменении старые проверки нужно будет удалить руками
SCRIPT_NAME = 'dt-solomon-alert'

# используется для выбора проекта и для поиска/удаления алертов, которых уже нет в конфиге
ENV_TO_SOLOMON_PRJ = {
    'prod': 'direct',
    'test': 'direct-test',
    'dev': 'direct-test',
    'junk': 'direct-junk',
}

SELECTORS_FUNC = {
    'any': 'AT_LEAST_ONE',
    'all': 'AT_ALL_TIMES',
    'last': 'LAST_NON_NAN',
    'avg': 'AVG',
    'min': 'MIN',
    'max': 'MAX',
    'sum': 'SUM',
    'count': 'COUNT',
}

THRESHOLD_FUNC = {
    'lt': 'LT',
    'le': 'LTE',
    'eq': 'EQ',
    'ne': 'NE',
    'gt': 'GT',
    'ge': 'GTE',
}

ALERT_SCHEMA = {
    'type': 'object',
    'additionalProperties': True,
    'required': ['env', 'name', 'selectors', 'threshold'],
    'properties': {
        'env': { 'enum': ['prod', 'test', 'dev'] },
        'name': { 'type': 'string', 'minLength': 1 },
        'description': { 'type': 'string', 'minLength': 0 },
        'selectors': { 'type': 'string', 'minLength': 1 },
        'selectors_function': { 'enum': SELECTORS_FUNC.keys() },
        'threshold_function': { 'enum': THRESHOLD_FUNC.keys() },
        'threshold': { 'type': 'number' },
        'period': { 'type': 'integer' },
        'delay': { 'type': 'integer' },
        'tags': { 'type': 'array', 'minItems': 1, 'items': { 'type': 'string', 'minLength': 1 } },
        'annotations': { 'type': 'object' },
        'notificationChannels': { 'type': 'array', 'minItems': 1, 'items': { 'type': 'string' } },
        'groupByLabels': { 'type': 'array', 'minItems': 1, 'items': { 'type': 'string' } },
    }
}

DEFAULT_ALERT = {
    'selectors_function': 'avg',
    'threshold_function': 'gt',
    'period': 300, # 5min
    'delay': 0,
    'description': "",
    'annotations': {},
    'notificationChannels': ['juggler-default'],
}

DEFAULT_CHECK = {
    'description': SCRIPT_NAME + ' auto check',
    'aggregator': 'logic_or',
    'tags': [SCRIPT_NAME],
    'notifications': [
        NotificationOptions(template_name='push', template_kwargs=dict(push_url='http://juggler-history.da.yandex.ru/save_notifications'), description=u'сервис для хранения истории по проверкам'),
    ],
}

class DtSolomonApi(object):
    def __init__(self, token, dry_run=False, url_prefix='http://solomon.yandex.net/api/v2'):
        self.token = token
        self.dry_run = dry_run
        self.url_prefix = url_prefix


    def call(self, path, params=None, json=None, method=requests.get):
        url = self.url_prefix + '/' + path
        auth = {'Authorization': 'OAuth ' + self.token}

        logging.debug('going to call {} on {} with params: {} and json_data: {}'.format(method.__name__.upper(), path, params, json))
        out = {}
        if self.dry_run and method is not requests.get:
            return out

        try:
            resp = method(url, params=params, json=json, headers=auth, timeout=(2, 10))
            resp.raise_for_status()
        except requests.HTTPError as e:
            logging.error('http error: ' + resp.text)
            raise

        # некоторые методы не возвращают ничего, но если возвращают - там должен быть json
        if resp.text:
            out = resp.json()
        return out


    def insert_alert(self, alert):
        logging.debug('insert new alert {id}'.format(**alert))
        return self.call('projects/{projectId}/alerts'.format(**alert), json=alert, method=requests.post)


    def update_alert(self, alert):
        logging.debug('update alert {id}'.format(**alert))
        return self.call('projects/{projectId}/alerts/{id}'.format(**alert), json=alert, method=requests.put)


    def delete_alert(self, alert):
        logging.debug('delete alert {id}'.format(**alert))
        return self.call('projects/{projectId}/alerts/{id}'.format(**alert), method=requests.delete)


    def get_alert(self, alert):
        logging.debug('get alert {id}'.format(**alert))
        return self.call('projects/{projectId}/alerts/{id}'.format(**alert), method=requests.get)


    def insert_or_update_alert(self, alert):
        logging.debug('insert or update solomon alert {id}'.format(**alert))
        try:
            cur_alert = self.get_alert(alert)
        except:
            return self.insert_alert(alert)

        alert = copy.deepcopy(alert)
        # копируем из существующего алерта следующую версию и общие поля
        for k in ['createdBy', 'createdAt', 'updatedBy', 'updatedAt', 'version']:
            alert[k] = cur_alert.get(k)

        if cur_alert == alert: # да, в глубину сравнение тоже идет
            logging.info('existing alert {id} same as new, skipping'.format(**alert))
            return alert

        logging.debug("current alert '%s' differs from new: '%s'" % (jdumps(cur_alert), jdumps(alert)))
        return self.update_alert(alert)


    def get_all_alerts(self, project):
        params = {'pageSize': 10}
        alerts = []
        logging.debug('get all alerts for {}, page 0'.format(project))
        page_alerts = self.call('projects/{}/alerts'.format(project), params=params)

        while page_alerts.get('items', []):
            logging.debug('got {} alerts for {}'.format(len(page_alerts['items']), project))
            for page_alert in page_alerts['items']:
                alert = self.get_alert(page_alert)
                alerts.append(alert)

            if 'nextPageToken' not in page_alerts:
                break

            params['pageToken'] = page_alerts['nextPageToken']
            logging.debug('get all alerts for {}, page {}'.format(project, params['pageToken']))
            page_alerts = self.call('projects/{}/alerts'.format(project), params=params)

        return alerts


def is_my_alert(alert):
    return alert.get('annotations', {}).get('createdWith') == SCRIPT_NAME


def solomon_init_alert(alert_id, name, description, project, selectors, threshold, aggr, predicate, period, delay, annotations, notificationChannels, groupByLabels):
    # так не работает - соломон умеет конвертить много всякого, кавычки и тд
    #if isinstance(selectors, str) or isinstance(selectors, unicode):
    #    try:
    #        selectors = json.loads(selectors)
    #    except:
    #        selectors = json.loads('{%s}' % selectors)
    #
    #    logging.debug('parsed selectors: %s %s' % (type(selectors), selectors))

    alert = {
        'id': alert_id,
        'projectId': project,
        'name': name,
        'description': description,
        'state': 'ACTIVE',
        # 'groupByLabels': [], # не надо добавлять сюда пустые ключи, solomon удалит их при insert, потом текущее состояние будет каждый раз отличаться от конфига
        'notificationChannels': notificationChannels,
        'type': {
            'threshold': {
                'selectors': selectors,
                'timeAggregation': aggr,
                'predicate': predicate,
                'threshold': threshold,
            },
        },
        'annotations': annotations,
        'periodMillis': period,
        'delaySeconds': delay
    }
    alert['annotations']['createdWith'] = SCRIPT_NAME
    if groupByLabels != None:
        alert['groupByLabels'] = groupByLabels
    return json.loads(json.dumps(alert)) # хотим на выходе все ключи и значения в unicode =(


def dt_alert_conf_to_solomon(alert_conf):
    alert_params = {
        'alert_id': alert_conf['name'],
        'name': alert_conf['name'],
        'description': alert_conf['description'],
        'project': ENV_TO_SOLOMON_PRJ[alert_conf['env']],
        'selectors': alert_conf['selectors'],
        'threshold': alert_conf['threshold'],
        'aggr': SELECTORS_FUNC[alert_conf['selectors_function']],
        'predicate': THRESHOLD_FUNC[alert_conf['threshold_function']],
        'period': alert_conf['period'] * 1000, # ms
        'delay': alert_conf['delay'],
        'annotations': alert_conf['annotations'],
        'notificationChannels': alert_conf['notificationChannels'],
        'groupByLabels': alert_conf.get('groupByLabels', None),
    }

    return solomon_init_alert(**alert_params)


def make_solomon_alerts(args):
    processed = {x: set() for x in set(ENV_TO_SOLOMON_PRJ.values())}
    solomon = DtSolomonApi(args.solomon_token, not args.apply)
    for alert_conf in args.config:
        alert = dt_alert_conf_to_solomon(alert_conf)
        solomon.insert_or_update_alert(alert)

        assert alert['projectId'] in processed
        processed[alert['projectId']].add(alert['id'])

    logging.debug('processed alerts: ' + str(processed))

    for project in set(ENV_TO_SOLOMON_PRJ.values()):
        logging.debug('find alerts to remove in project ' + project)
        my_alerts = filter(is_my_alert, solomon.get_all_alerts(project))
        to_remove = [x for x in my_alerts if x['id'] not in processed[project]]

        logging.debug('going to remove: ' + jdumps([x['id'] for x in to_remove]))
        for alert in to_remove:
            solomon.delete_alert(alert)


def make_juggler_aggrs(args):
    checks = []
    for alert_conf in args.config:
        alert = dt_alert_conf_to_solomon(alert_conf)

        if 'juggler-default' not in alert['notificationChannels']:
            logging.info('skipping juggler aggr for ' + alert['id'])
            continue

        namespace = 'direct.' + alert_conf['env']
        raw_service = alert['id']
        raw_host = alert['projectId'] + '.solomon-alert'

        params = {
            'host': namespace + '_solomon-alert',
            'namespace': namespace,
            'service': raw_service,
            'children': [Child(host=raw_host, group_type='HOST', service=raw_service)],
            'tags': list(set(
                alert_conf.get('tags', []) +
                DEFAULT_CHECK.get('tags', [])
            )),
        }
        checks.append(Check(**hash_merge(DEFAULT_CHECK, params)))

    logging.info('applying juggler checks ...')
    apply_checks(checks, token=args.juggler_token, mark=SCRIPT_NAME, dry_run=not args.apply)


def prepare_config(raw_conf):
    conf = {}
    err = False

    for item in raw_conf:
        item = hash_merge(DEFAULT_ALERT, item)
        logging.debug('processing conf item: ' + jdumps(item))
        # если не подошла схема - падаем, зажигаем общий мониторинг
        check_schema_or_die(item, ALERT_SCHEMA)

        name = '{env}-{name}'.format(**item)
        # если такой алерт уже есть - добавляем только первый, зажигаем общий мониторинг
        if name in conf:
            logging.warning('duplicate alert {}, ignoring'.format(name))
            err = True
            continue

        conf[name], item_err = filter_schema_keys(item, ALERT_SCHEMA)
        if item_err:
            logging.warning('odd keys in alert {name}, ignoring'.format(**item))
            err = True

    return conf.values(), err


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter, description=__doc__)
    parser.add_argument('--apply', default=False, action='store_true', help='применять все изменения в juggler (без этого никаких пишущих действий не производится)')
    parser.add_argument('-c', '--config', required=True, help='путь до конфига со списком проверяемых шедулеров')
    parser.add_argument('-l', '--solomon-token', required=True, help='путь до sandbox-token')
    parser.add_argument('-j', '--juggler-token', required=True, default='/etc/direct-tokens/juggler_api', help='токен для доступа в juggler-api')
    parser.add_argument('-m', '--max-config-age', type=int, default=3600, help='допустимый возраст конфига (секунд)')
    args = parser.parse_args()

    return args


def main():
    args = parse_args()
    init_logger()
    logging.debug('running with args: ' + jdumps(args))

    if not check_file_age(args.config, args.max_config_age):
        die(SCRIPT_NAME, 'config file is too old or does not exist', not args.apply)

    try:
        args.config, conf_err = prepare_config(json.loads(read_file(args.config)))
        logging.info('running with config: ' + jdumps(args.config))

        args.solomon_token = read_file(args.solomon_token).strip()
        args.juggler_token = read_file(args.juggler_token).strip()

        make_solomon_alerts(args)
        make_juggler_aggrs(args)
    except Exception as e:
        die(SCRIPT_NAME, '\n%s\ncannot make solomon alerts: %s %s' % (traceback.format_exc(), type(e), e), not args.apply)

    report_status_to_juggler(SCRIPT_NAME, conf_err, not args.apply)
    if args.apply:
        logging.info('completed successfully')
    else:
        logging.info('no alerts configured, run with --apply')


if __name__ == '__main__':
    main()
