# -*- coding: utf-8 -*-
from direct_juggler import juggler as dj
import copy
import jsonschema
import json
import logging
import sys
import os
import time


def jdumps(jdict, pretty=False):
    def serialize(obj):
        try:
            s = obj.__dict__
        except:
            s = obj
        return s

    if pretty:
        return '\n' + json.dumps(jdict, sort_keys=True, ensure_ascii=False, indent=4, separators=(',', ': '), default=serialize)
    return json.dumps(jdict, sort_keys=True, ensure_ascii=False, default=serialize)


def read_file(fname):
    with open(fname, 'r') as f:
        return f.read()


def report_status_to_juggler(service, conf_err=False, dry_run=False):
    evt = {
        'service': service,
        'status': 'WARN' if conf_err else 'OK',
        'description': 'some config issues, see /var/log/direct-apps-juggler/*-{}.log'.format(service) if conf_err else 'OK'
    }
    logging.debug('juggler queue event: ' + jdumps(evt))
    if dry_run:
        return

    res = dj.queue_events([evt])
    logging.debug('juggler queue event response: ' + jdumps(res))


def die(service, msg='', dry_run=False):
    logging.critical(msg + ' - exiting')
    evt = {
        'service': service,
        'status': 'CRIT',
        'description': 'script failed, see /var/log/direct-apps-juggler/*-{}.log'.format(service)
    }
    logging.debug('juggler queue event: ' + jdumps(evt))
    if dry_run:
        sys.exit(1)

    res = dj.queue_events([evt])
    logging.debug('juggler queue event response: ' + jdumps(res))
    sys.exit(1)


def check_schema_or_die(data, schema):
    try:
        jsonschema.validate(data, schema)
    except jsonschema.exceptions.ValidationError as e:
        logging.critical('bad event schema in path .%s with value %s - %s' % ('.'.join([str(x) for x in e.path]), e.instance, e.message))
        raise


def filter_schema_keys(data, schema):
    conf = {}
    err = False
    # просто удаляем лишние ключи, в глубину пока ничего нет
    for k, v in data.items():
        if k not in schema['properties'].keys():
            if k != "tags_np":
                logging.warning('possible bad or mistyped key: ' + str(k))
                err = True
            continue
        else:
            conf[k] = copy.deepcopy(v)

    return conf, err


def init_logger():
    logging.basicConfig(level=logging.DEBUG, stream=sys.stdout,
                        format='[%(asctime)s]\t%(filename)40s:%(funcName)-40s\t%(levelname)-10s\t%(message)s')


def check_file_age(filename, max_age):
    return os.path.isfile(filename) and (time.time() - os.stat(filename).st_mtime) <= max_age


