from __future__ import absolute_import

import sys
import httplib
import inspect
import datetime as dt
import itertools as it
import collections
import distutils.util

import aniso8601

from .. import rest
from .. import utils
from ..types import user as ctu

httplib.responses[451] = "Unavailable For Legal Reasons"
httplib.LEGAL_REASONS = 451
httplib.responses[423] = "Locked"


class Scope(utils.Enum):
    QUERY = "query"
    BODY = "body"
    PATH = "path"


class ArrayFormat(utils.Enum):
    CSV = "csv"
    MULTI = "multi"


class IntegerFormat(utils.Enum):
    I16 = "int16"
    I32 = "int32"
    I64 = "int64"


class DataType(object):
    """
    Base data type
    @DynamicAttrs
    """

    # noinspection PyPep8Naming
    class __metaclass__(type):
        __type__ = None
        __re__ = r"[^/]+"
        __param_name__ = None

        def __call__(cls, *args, **kws):
            frame = inspect.currentframe().f_back
            if "__names__" not in frame.f_locals:
                frame.f_locals["__names__"] = frame.f_code.co_names
            func = cls.__new__
            code = func.__code__
            names = code.co_varnames[1:code.co_argcount + 1]
            defaults_dict = dict(zip(names, func.__defaults__))
            defaults_dict.update(dict(zip(names[:len(args)], args)))
            defaults_dict.update(kws)
            required = defaults_dict.get("required")
            description = defaults_dict.get("description")
            assert isinstance(description, (basestring, type(None))), "`description` must be a string"
            if description is None and cls.__doc__:
                defaults_dict["description"] = cls.__doc__.strip()
            defaults = tuple(_[1] for _ in sorted(defaults_dict.iteritems(), key=lambda _: names.index(_[0])))
            return type(
                "__metaclass__",
                (type(cls),),
                dict(
                    __nonzero__=lambda _: required,
                    __getattr__=lambda _, attr: DataType.__getattr__(attr, defaults_dict)
                )
            )(
                cls.__name__,
                (cls,),
                dict(
                    # __doc__=description,
                    __new__=type(func)(func.__code__, func.__globals__, func.__name__, defaults, func.__closure__)
                )
            )

        def __iter__(cls):
            func = cls.__new__
            code = func.__code__
            names = code.co_varnames[1:code.co_argcount + 1]
            if cls.__param_name__:
                yield ("name", cls.__param_name__)
            if cls.__type__:
                yield ("type", cls.__type__)
            for k, v in it.izip(names, func.__defaults__):
                if v is None:
                    continue
                if k == "scope":
                    k = "in"
                _ = k.split("_")
                k = "".join(it.chain(_[:1], it.imap(str.capitalize, _[1:])))
                yield (k, v)

        def __getattr__(cls, attr, attrs=None):
            if not attrs or attr not in attrs:
                raise AttributeError
            return attrs[attr]

    __fixed_param_fields__ = {"in", "required", "name", "description"}
    __schema_fields__ = {
        "type", "title", "format", "description", "default", "minimum", "maximum", "required", "enum", "items"
    }

    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None):
        pass

    def __call__(self, *args, **kws):
        pass

    @classmethod
    def decode(cls, value):
        if cls and value is None:
            raise ValueError("`{}` is required".format(cls.__param_name__))
        return cls.default if value is None else value

    encode = decode


class String(DataType):
    __type__ = "string"

    @classmethod
    def decode(cls, value):
        value = super(String, cls).decode(value)
        if value is cls.default:
            return value
        return value if isinstance(value, basestring) else str(value)

    encode = decode


class DateTime(DataType):
    """ UTC date in ISO 8601 format """
    __type__ = "string"

    # noinspection PyShadowingBuiltins
    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None, format="date-time"):
        pass

    @classmethod
    def decode(cls, value):
        value = super(DateTime, cls).decode(value)
        if value is cls.default:
            return value
        value = value.replace(" ", "T")  # in case there's no proper separator
        return aniso8601.parse_datetime(value).replace(tzinfo=None)

    @classmethod
    def encode(cls, value):
        value = super(DateTime, cls).encode(value)
        if value is None:
            return
        if not isinstance(value, dt.datetime):
            raise ValueError("`{}` must be a datetime object".format(cls.__param_name__))
        return value.isoformat() + "Z"


class DateTimeRange(DataType):
    """ A range of UTC dates in ISO 8601 format separated by `..` to be used in list filter """
    __type__ = "string"

    @classmethod
    def decode(cls, value):
        value = super(DateTimeRange, cls).decode(value)
        if value is cls.default:
            return value
        date_range = str(value).split("..")
        if len(date_range) != 2:
            raise ValueError(
                "`{}` must be a range of UTC dates in ISO 8601 format separated by `..`".format(cls.__param_name__)
            )
        return map(lambda _: aniso8601.parse_datetime(_).replace(tzinfo=None), date_range)

    @classmethod
    def encode(cls, value):
        value = super(DateTimeRange, cls).encode(value)
        if value is None:
            return
        if len(value) != 2:
            raise ValueError(
                "`{}` must be a pair of two datetime objects".format(cls.__param_name__)
            )
        return "..".join(map(lambda _: _.isoformat() + "Z", value))


class Enum(String):
    # noinspection PyPep8Naming
    class __metaclass__(type(String)):
        def __iter__(cls):
            # noinspection PyUnresolvedReferences
            return it.imap(lambda (k, v): ("enum", list(v)) if k == "values" else (k, v), type(String).__iter__(cls))

    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None, values=()):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Enum, cls).decode(value)
        if value is not None and value not in cls.values:
            raise ValueError(
                "`{}` must be one of the {}, passed {}".format(cls.__param_name__, list(cls.values), value)
            )
        return value

    encode = decode


class Integer(DataType):
    __type__ = "integer"
    __re__ = r"\d+"

    # noinspection PyShadowingBuiltins
    def __new__(
        cls, description=None, required=False, scope=Scope.QUERY, default=None,
        format=IntegerFormat.I32, minimum=0, maximum=None
    ):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Integer, cls).decode(value)
        if value is None:
            return
        value = int(value)
        if cls.minimum is not None and value < cls.minimum or cls.maximum is not None and value > cls.maximum:
            raise ValueError("`{}` must be from {} to {}".format(cls.__param_name__, cls.minimum, cls.maximum))
        return value

    encode = decode


class Array(DataType):
    __type__ = "array"

    # noinspection PyPep8Naming
    class __metaclass__(type(DataType)):
        def __call__(cls, *args, **kws):
            frame = inspect.currentframe().f_back
            if "__names__" not in frame.f_locals:
                frame.f_locals["__names__"] = frame.f_code.co_names
            cls_ = type(DataType).__call__(cls, *args, **kws)
            assert inspect.isclass(cls_.items) and issubclass(
                cls_.items, DataType
            ), "Item's type must be specified for Array: {!r}".format(cls_.items)
            if args and args[0] is cls_.items:
                cls_ = type(DataType).__call__(cls, args[0](), *args[1:], **kws)
            return cls_

        def __iter__(cls):
            # noinspection PyUnresolvedReferences
            for k, v in type(DataType).__iter__(cls):
                if inspect.isclass(v):
                    v = (
                        {"$ref": v.ref}
                        if issubclass(v, Schema) else
                        dict(
                            it.ifilter(
                                lambda _:
                                    _[0] not in DataType.__fixed_param_fields__ and _[0] in DataType.__schema_fields__,
                                v
                            )
                        )
                    )
                yield (k, v)

    # noinspection PyShadowingBuiltins
    def __new__(
        cls, items=None, description=None, required=False, scope=Scope.QUERY, default=None,
        collection_format=ArrayFormat.CSV
    ):
        pass

    def __iter__(self):
        return iter(self.__items__)

    def __getstate__(self):
        return self.__items__

    @classmethod
    def decode(cls, value):
        value = super(Array, cls).decode(value)
        if value is cls.default:
            return value
        if cls.collection_format == ArrayFormat.CSV and cls.scope != Scope.BODY:
            if not isinstance(value, basestring):
                raise ValueError("Expected string in CSV format")
            value = it.ifilter(None, value.split(","))
        return map(cls.items.decode, utils.chain(value))

    @classmethod
    def encode(cls, value):
        value = super(Array, cls).encode(value)
        if value is cls.default:
            return value
        return map(cls.items.encode, utils.chain(value))


class Object(DataType):
    __type__ = "object"

    @classmethod
    def decode(cls, value):
        value = super(Object, cls).decode(value)
        if value is cls.default:
            return value
        if not isinstance(value, dict):
            raise ValueError("`{}` must be a dict: {!r}".format(cls.__param_name__, value))
        return value

    encode = decode


class Id(Integer):
    """ Object identifier """
    # noinspection PyShadowingBuiltins
    def __new__(
        cls, description=None, required=True, scope=Scope.QUERY, default=None,
        format=IntegerFormat.I64, minimum=1, maximum=None
    ):
        pass


class Number(DataType):
    __type__ = "number"

    # noinspection PyShadowingBuiltins
    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Number, cls).decode(value)
        if value is None:
            return
        if isinstance(value, (int, long, float)):
            return value
        try:
            value = int(value)
        except (ValueError, TypeError):
            value = float(value)
        return value

    encode = decode


class Schema(DataType):
    __fields__ = {}
    __ref__ = None

    # noinspection PyPep8Naming
    class __metaclass__(type(DataType)):
        """ @DynamicAttrs """
        __api__ = None
        __registry__ = None
        __used__ = None

        def __new__(mcs, name, bases, namespace):
            namespace["__ref__"] = "#/definitions/{}".format(name)
            cls = type(DataType).__new__(mcs, name, bases, namespace)
            if bases != (DataType,) and "__new__" not in namespace:
                assert mcs.__registry__ is not None, "Do not use common.api.Schema directly"
                assert name not in mcs.__registry__, "schema duplication: {}".format(name)
                fields = {k: v for base in bases if issubclass(base, Schema) for k, v in base.__fields__.iteritems()}
                for k, v in namespace.iteritems():
                    if inspect.isclass(v) and issubclass(v, DataType):
                        fields[k] = v
                        v.__param_name__ = k
                cls.__fields__ = fields
                mcs.__registry__[name] = cls
            return cls

        def __iter__(cls):
            if cls is type(cls).__api__.__schema__:
                for k, v in type(cls).__registry__.iteritems():
                    yield (k, dict(v))
                return
            # noinspection PyUnresolvedReferences
            for k, v in type(DataType).__iter__(cls):
                if k in DataType.__fixed_param_fields__:
                    continue
                yield (k, v)
            required = []
            properties = {}
            for k, v in cls.__fields__.items():
                k = k.rstrip("_")
                if v:
                    required.append(k)
                if issubclass(v, Schema):
                    properties[k] = {"$ref": v.ref}
                    description = dict(v).get("description")
                    if description is not None:
                        properties[k]["description"] = description
                else:
                    v = dict(
                        it.ifilter(
                            lambda _:
                                _[0] not in DataType.__fixed_param_fields__ and _[0] in DataType.__schema_fields__,
                            v
                        )
                    )
                    properties[k] = v
            yield ("properties", properties)
            if required:
                yield ("required", required)

        @property
        def ref(cls):
            type(cls).__used__.add(cls.__name__)
            return cls.__ref__

    def __iter__(self):
        for name, field in self.__fields__.iteritems():
            value = getattr(self, name, field.default)
            if value is None and field.required:
                raise ValueError("{}.{} is required and has no default value".format(type(self).__name__, name))
            yield name.rstrip("_"), value

    def __getstate__(self):
        return dict(self)

    def __setattr__(self, name, value):
        try:
            field = self.__fields__[name]
        except KeyError:
            raise AttributeError
        object.__setattr__(self, name, value if value is None else field.encode(value))

    @classmethod
    def decode(cls, value):
        data = super(Schema, cls).decode(value)
        if data is None:
            return
        if not isinstance(data, dict):
            raise ValueError("`{}` must be a dict (`{}`)".format(cls.__param_name__, cls.__name__))
        obj = object.__new__(cls)
        for name, field in obj.__fields__.iteritems():
            value = data.pop(name.rstrip("_"), field.default)
            if value is None and field.required:
                raise ValueError("{}.{} is required and has no default value".format(cls.__name__, name))
            setattr(obj, name, value if value is None else field.encode(value))
        if data:
            raise ValueError("Extra fields for {} given: {}".format(cls.__name__, data))
        return obj

    @classmethod
    def encode(cls, *args, **kws):
        if args:
            if len(args) > 1:
                raise ValueError("Only one positional argument supported and it must be a dict")
            obj = args[0]
            if isinstance(obj, DataType):
                return obj
            kws = dict(obj)
        obj = object.__new__(cls)
        for name, field in obj.__fields__.iteritems():
            value = kws.pop(name, field.default)
            object.__setattr__(obj, name, value if value is None else field.encode(value))
        if kws:
            raise ValueError("Extra fields for {} given: {}".format(cls.__name__, kws.keys()))
        return obj

    @classmethod
    def create(cls, *args, **kws):
        return cls.encode(*args, **kws)


class Boolean(DataType):
    __type__ = "boolean"

    @classmethod
    def decode(cls, value):
        value = super(Boolean, cls).decode(value)
        if value is None or isinstance(value, bool):
            return value
        return bool(distutils.util.strtobool(str(value)))

    @classmethod
    def encode(cls, value):
        value = super(Boolean, cls).encode(value)
        if value is None:
            return
        return bool(value)


class Response(object):
    # noinspection PyPep8Naming
    class __metaclass__(type):
        __content__ = rest.Client.JSON
        __schema__ = None
        __http_code__ = None
        __headers__ = None

        def __new__(mcs, name, bases, namespace):
            if bases != (object,):
                http_name = utils.ident(name)
                http_code = getattr(httplib, http_name, None)
                assert http_code and http_code in httplib.responses,\
                    "name {!r} does not correspond to any of known HTTP responses".format(name)
                schemas = []
                headers = {}
                for k, v in namespace.iteritems():
                    if not inspect.isclass(v):
                        continue
                    if issubclass(v, DataType):
                        schemas.append(v)
                    elif issubclass(v, Header):
                        headers["-".join(map(lambda _: _.capitalize(), k.split("_")))] = v
                assert len(schemas) < 2, "Cannot define more than one schema for response"
                namespace.update(
                    __schema__=(schemas[0] if schemas else None),
                    __http_code__=http_code,
                    __headers__=headers
                )
            return type.__new__(mcs, name, bases, namespace)

        def __iter__(cls):
            yield ("description", (cls.__doc__ or "").strip())
            if cls.__schema__ is not None:
                if issubclass(cls.__schema__, Schema):
                    schema = {"$ref": cls.__schema__.ref}
                else:
                    schema = dict(
                        it.ifilter(
                            lambda _:
                                _[0] not in DataType.__fixed_param_fields__ and _[0] in DataType.__schema_fields__,
                            cls.__schema__
                        )
                    )
                yield ("schema", schema)
            if cls.__headers__:
                yield ("headers", {k: dict(v) for k, v in cls.__headers__.iteritems()})


class Request(object):
    # noinspection PyPep8Naming
    class __metaclass__(type):
        __content__ = None
        __security__ = ctu.Restriction.AUTHENTICATED
        __allow_ro__ = False
        __parameters__ = ()
        __responses__ = None
        __operation_id__ = None

        def __new__(mcs, name, bases, namespace):
            if bases != (object,):
                parameters = []
                responses = []
                parameters.extend(it.chain.from_iterable(_.__parameters__ for _ in bases))
                own_parameters = []
                for key, value in namespace.iteritems():
                    if not inspect.isclass(value):
                        continue
                    if issubclass(value, DataType):
                        value.__param_name__ = key
                        if issubclass(value, Array):
                            value.items.__param_name__ = "{}[]".format(key)
                        own_parameters.append(value)
                    elif issubclass(value, Response):
                        responses.append(value)
                parameters_names = namespace.get("__names__", ())
                own_parameters.sort(key=lambda _: parameters_names.index(_.__param_name__))
                parameters.extend(own_parameters)
                namespace.update(__parameters__=parameters, __responses__=responses)
            return type.__new__(mcs, name, bases, namespace)

        def __iter__(cls):
            if cls.__doc__:
                doc = cls.__doc__.strip().split("\n")
                summary = doc[0]
                description = " ".join(it.ifilter(None, it.imap(str.strip, doc[1:])))
                if summary:
                    yield ("summary", summary)
                if description:
                    yield ("description", description)
            if cls.__operation_id__:
                yield ("operationId", cls.__operation_id__)
            yield ("tags", [cls.__module__.split(".")[-1]])
            parameters = []
            for param in cls.__parameters__:
                if issubclass(param, Schema):
                    ref = param.ref
                    # noinspection PyUnresolvedReferences
                    param = dict(type(DataType).__iter__(param))
                    param["schema"] = {"$ref": ref}
                else:
                    if param.scope == Scope.BODY:
                        schema = {}
                        param = dict(param)
                        for k, v in param.items():
                            if k not in DataType.__fixed_param_fields__:
                                v = param.pop(k)
                                if k in DataType.__schema_fields__:
                                    schema[k] = v
                        param = {k: v for k, v in param.iteritems() if k in DataType.__fixed_param_fields__}
                        if schema:
                            param["schema"] = schema
                    else:
                        param = dict(param)
                parameters.append(param)
            yield ("parameters", parameters)
            responses = {}
            produces = []
            for response in cls.__responses__:
                responses[str(response.__http_code__)] = dict(response)
                if response.__content__ != type(response).__content__:
                    produces.append(response.__content__.type)
            yield ("responses", responses)
            if cls.__content__ != type(cls).__content__:
                yield ("consumes", [cls.__content__.type])
            if produces:
                yield ("produces", produces)
            if cls.__security__ != ctu.Restriction.ANY:
                yield (
                    "security",
                    [{
                        "oauth": map(str.lower, utils.chain(cls.__security__))
                    }]
                )


class Get(Request):
    __security__ = ctu.Restriction.ANY


class Post(Request):
    # noinspection PyPep8Naming
    class __metaclass__(type(Request)):
        __content__ = rest.Client.JSON


class Put(Request):
    # noinspection PyPep8Naming
    class __metaclass__(type(Request)):
        __content__ = rest.Client.JSON


class Delete(Request):
    pass


class Header(object):
    # noinspection PyPep8Naming
    class __metaclass__(type):
        __item__ = None

        def __call__(cls, item):
            if isinstance(item, basestring):
                item = String(item)
            return type(cls.__name__, (cls,), dict(__item__=item))

        def __iter__(cls):
            return it.ifilter(lambda _: _[0] not in DataType.__fixed_param_fields__, cls.__item__)

    def __new__(cls, item):
        pass


class Api(object):
    __header__ = None
    __paths__ = None
    __schema__ = None
    __modules__ = None

    # noinspection PyPep8Naming
    class __metaclass__(type):
        def __new__(mcs, name, bases, namespace):
            if bases == (object,):
                return type.__new__(mcs, name, bases, namespace)
            mcs = type(mcs.__name__, (mcs,), dict(dict=utils.singleton_property(mcs.dict.fget)))
            cls = type.__new__(mcs, name, bases, namespace)
            cls.__header__ = {k: v for k, v in namespace.iteritems() if not k.startswith("_")}
            cls.__paths__ = {}
            cls.__modules__ = set()
            return cls

        @property
        class Path(object):
            api = None
            path = None
            requests = None

            # noinspection PyPep8Naming
            class __metaclass__(type):
                def __new__(mcs, name, bases, namespace):
                    assert bases == (object,) or bases[0].path, "path must be defined"
                    if bases == (object,):
                        return type.__new__(mcs, name, bases, namespace)
                    cls = bases[0]
                    operation_prefix = utils.ident(name).lower()
                    requests = []
                    for value in namespace.itervalues():
                        if not inspect.isclass(value) or not issubclass(value, Request):
                            continue
                        value.__operation_id__ = "{}_{}".format(operation_prefix, value.__name__.lower())
                        requests.append(value)
                        setattr(cls, value.__name__, value)
                    cls.requests = requests
                    cls.api.__modules__.add(namespace["__module__"])
                    return cls

                def __call__(cls, api_or_path):
                    # noinspection PyUnresolvedReferences
                    api = cls.api
                    if api:
                        path = api_or_path
                        cls.path = path
                        paths = api.__paths__
                        assert path not in paths, "path duplication: {}".format(path)
                        paths[path] = cls
                        return cls
                    else:
                        return type.__new__(type(cls), cls.__name__, cls.__bases__, dict(api=api_or_path))

                def __iter__(cls):
                    for request in cls.requests:
                        yield (request.__name__.lower(), dict(request))

            def __new__(cls, path):
                pass

        @property
        class Schema(Schema):
            # noinspection PyPep8Naming
            class __metaclass__(type(Schema)):
                def __call__(cls, api):
                    if api.__schema__:
                        return api.__schema__
                    mcs = type(Schema)
                    api.__schema__ = type.__new__(
                        type(
                            mcs.__name__,
                            (mcs,),
                            dict(
                                __api__=api,
                                __registry__={},
                                __used__=set()
                            )
                        ),
                        cls.__name__, cls.__bases__, dict(api=api)
                    )
                    return api.__schema__

            def __new__(cls):
                pass

        @utils.singleton_property
        def dict(cls):
            result = {}
            result.update(cls.__header__)
            # noinspection PyArgumentList
            result.update(
                paths=collections.OrderedDict(
                    sorted(((k, dict(v)) for k, v in cls.__paths__.iteritems()), key=lambda _: _[0])
                )
            )
            # XXX: need to account for sub schemas
            list(cls.Schema)
            result.update(definitions={k: v for k, v in cls.Schema if k in cls.Schema.__used__})
            result.update(
                tags=map(
                    lambda _: dict(name=_.split(".")[-1], description=(sys.modules[_].__doc__ or "").strip()),
                    cls.__modules__
                )
            )
            return result

    # just for auto completion
    Path = None
    Schema = Schema
    dict = None
