"""
The head of this file is a copy of base part of ``library.config`` module, which is supposed to be dropped
after the interface will be committed and released with skynet itself.

Also, skynet should provide `classproperty` and `Singleton` classes together with that release, so we have
to replace them too.
"""

from __future__ import absolute_import, print_function

import os
import re
import time
import socket
import getpass
import logging
import platform
import multiprocessing as mp

import py
import yaml
import msgpack

try:
    from yaml import CLoader as YamlLoader
except ImportError:
    from yaml import Loader as YamlLoader

from . import utils
from . import patterns
from . import import_hook
from .types import misc as ctm


class RegistryBase(object):
    """
    The class encapsulates all the logic related to cacheable registry file parsing and content representation.
    """
    __metaclass__ = utils.SingletonMeta

    CACHE_RECHECK_TIMEOUT = 10  # Cache registry file for max 10 seconds.

    RE_PLACEHOLDER = re.compile(r'\${([a-z0-9_.-]+)}', re.IGNORECASE)
    RE_KEY_ESC = re.compile(r'(^[^a-z_]|[^a-z_0-9]|^\s*$)', re.IGNORECASE)

    class Config(dict):
        """
        The common parent class for any configuration slotted class' instances, which will
        be produced by @c query() function, commonly designed to distinguish them from plain dictionaries.
        """
        __slots__ = []

    class StorageOutdated(Exception):
        """
        Special exception class to signal the registry cannot be loaded because it is out-of-date.
        """

    class Cache(patterns.Abstract):
        """
        The class represents local data cache storage with appropriate metadata.
        """
        __slots__ = ['atime', 'mtime', 'data']
        __defs__ = [0, 0, None]

    def __init__(self):
        self.logger = logging.getLogger(__name__)
        self._cache = {}
        super(RegistryBase, self).__init__()

    def load(self, path, base=None):
        """
        Internal caching function, which will fetch the registry raw content.
        :return: Cache for the registry path given.
        """

        now = time.time()

        if import_hook.inside_the_binary():
            # module path in binary is relative to arcadia root, so strip PWD to normalize path
            resource_path = path.relto(py.path.local())
            if resource_path not in self._cache:
                from library.python import resource
                resource_data = resource.find(resource_path)
                if resource_data is not None:
                    self.logger.debug("Loading registry data from arcadia resource %r", resource_path)
                    self._cache[resource_path] = self.Cache(
                        now, now, self._merge_recursive(base, yaml.load(resource_data, Loader=YamlLoader))
                    )
            if resource_path in self._cache:
                return self._cache[resource_path]

        cache = self._cache.get(path, None)
        if cache and cache.atime + self.CACHE_RECHECK_TIMEOUT > now:
            return cache

        mtime = path.stat().mtime
        if cache and mtime <= cache.mtime:
            return cache

        self.logger.debug("Loading registry data from %r", path.strpath)
        data = self._merge_recursive(base, yaml.load(path.open("rb").read(), Loader=YamlLoader))
        self._cache[path] = self.Cache(now, mtime, data)
        return self._cache[path]

    @classmethod
    def _dict2slotted(cls, dct, fix_keys=True):
        """
        Internal function for dictionary to slotted class instance transformation.
        :param dct:     Dictionary to be processed.
        :param fix_keys: Flags the dictionary keys should be transliterated to be safe Python identifier.
        :return:        Slotted class instance.
        """
        keys = [str(k) if not fix_keys else cls.RE_KEY_ESC.sub(r'_', str(k)) for k in dct.keys()]

        try:
            class _Config(cls.Config):
                __slots__ = keys

                def __init__(self):
                    super(_Config, self).__init__(dct)

                def copy(self):
                    return cls._dict2slotted(dct, fix_keys=fix_keys)
        except TypeError:
            raise TypeError('Unable to set {!r} as class slots: not an identifier detected.'.format(keys))

        c = _Config()
        for key, value in dct.items():
            setattr(
                c,
                str(key) if not fix_keys else cls.RE_KEY_ESC.sub(r'_', str(key)),
                cls._dict2slotted(value, fix_keys) if isinstance(value, dict) else value
            )
        return c

    @staticmethod
    def _dict_query(dct, path):
        for k in path.split('.') if path else []:
            dct = dct[k]
        return dct

    @classmethod
    def _merge_recursive(cls, base, other):
        deepcopy = lambda x: msgpack.loads(msgpack.dumps(x))
        if not isinstance(other, dict):
            return other
        if not isinstance(base, dict):
            return deepcopy(other)
        result = deepcopy(base)
        for k, v in other.iteritems():
            if isinstance(result.get(k), dict):
                result[k] = cls._merge_recursive(result[k], v)
            else:
                result[k] = deepcopy(v)
        return result

    @classmethod
    def _evaluate(cls, data):
        # Recursively evaluate placeholders inplace
        queue = [data]
        while queue:
            next_queue = []
            for item in queue:
                if isinstance(item, dict):
                    for k in item:
                        if isinstance(item[k], tuple):
                            item[k] = list(item[k])
                    items = item.items()
                elif isinstance(item, list):
                    items = enumerate(item)

                for k, v in items:
                    if isinstance(v, (dict, list)):
                        next_queue.append(v)
                    elif isinstance(v, basestring):
                        changed = False
                        while True:
                            match = cls.RE_PLACEHOLDER.search(v)
                            if not match:
                                break
                            v = ''.join([
                                v[:match.start()],
                                str(cls._dict_query(data, match.group(1))),
                                v[match.end():]
                            ])
                            changed = True
                        if changed:
                            item[k] = v

                queue = next_queue

    @classmethod
    def query(
        cls,
        data,
        subsection=None,
        base=None,
        overrides={},
        as_dict=False,
        fix_keys=True,
        evaluate=True,
    ):
        """
        Perform given data query.

        :param subsection:   Sub-section of the queried section. `None` by default.
                             In case of `None` the whole section will be returned.
                             Can fetch nested dictionary be separating their keys with commas.
        :param base:         Dictionary of values to be considered as base result, its values will be
                             overridden by the ones from the fetched config.
        :param overrides:    Plain dictionary of values to be placed into result additionally
                             __relatively__ to the sub-section specified above, overriding
                             any existing data. Key can be comma-separated to represent inner dictionaries.
        :param as_dict:      Return configuration as plain dictionary instead of slotted class instance.
        :param fix_keys:     In case of slotted class instance requested, perform automatic transliteration
                             of unsafe key characters.
        :param evaluate:     Perform evaluation of string values' placeholders in form of ``${path}``.
                             Placeholders evaluation are __relative__ to the sub-section specified above.
        :return:             Dictionary with the configuration data for the queried section and sub-section.

        :raise OSError:      In case of no local registry file exists or not readable.
        :raise KeyError:     In case of no such section or sub-section exists.
        """

        fmt = (
            lambda x: cls._dict2slotted(x, fix_keys=fix_keys),
            lambda x: x,
        )[int(as_dict)]

        data = cls._dict_query(data, subsection)
        data = cls._merge_recursive(base, data) if base else data

        # Perform overrides merge
        for k, v in overrides.iteritems():
            path = k.split('.')
            curr, prev = data, None
            for p in path[:-1]:
                prev, curr = curr, curr.get(p, None)
                if not isinstance(curr, dict):
                    curr = prev[p] = {}
            curr[path[-1]] = v

        if evaluate:
            cls._evaluate(data)
        return fmt(data)


class Registry(RegistryBase):
    """
    Sandbox-specific registry loader. Loads first `etc/.settings.yaml` file from the current directory and
    customizes it by merging `etc/settings.yaml` file's content over it.
    """
    CONFIG_ENV_VAR = "SANDBOX_CONFIG"

    def __init__(self):
        self.__config = None
        super(Registry, self).__init__()

    def __getattr__(self, item):
        value = getattr(self.root(), item)
        setattr(self, item, value)
        return value

    @property
    def _config_dir(self):
        return py.path.local(__file__).dirpath(os.pardir, "etc")

    @property
    def _base(self):
        """ Path to base (skeleton) configuration file. """
        return self._config_dir.join(".settings.yaml")

    @property
    def _custom(self):
        """ Path to custom (custom values for skeleton) configuration file. """
        return (
            py.path.local(os.environ[self.CONFIG_ENV_VAR])
            if self.CONFIG_ENV_VAR in os.environ else
            (
                None  # CONFIG_ENV_VAR should be defined during binary execution
                if import_hook.inside_the_binary() else
                self._config_dir.join("settings.yaml")
            )
        )

    def root(self):
        if self.__config:
            return self.__config

        saved_getattr, self.__getattr__ = self.__getattr__, getattr
        try:
            base = None
            for _base in utils.chain(self._base):
                base = self.load(_base, base=base).data
            this = base.setdefault("this", {})
            this.setdefault("fqdn", socket.getfqdn())
            this.setdefault("id", this["fqdn"].split(".")[0])
            system = this.setdefault("system", {"user": getpass.getuser()})
            sysname = system.setdefault("name", platform.system().lower())
            this.setdefault("cpu", {"cores": mp.cpu_count()})
            if sysname == "linux":
                system.setdefault("family", ctm.OSFamily.LINUX)
            elif sysname == "freebsd":
                system.setdefault("family", ctm.OSFamily.FREEBSD)
            elif sysname == "darwin":
                system.setdefault("family", ctm.OSFamily.OSX)
            elif sysname.startswith("cygwin"):
                system.setdefault("family", ctm.OSFamily.CYGWIN)
            elif sysname.startswith("windows"):
                system.setdefault("family", ctm.OSFamily.WIN_NT)
            else:
                raise NotImplementedError("Unsupported system '{}'".format(sysname))

            custom = {}
            custom_config_path = self._custom
            if os.path.exists(str(custom_config_path)):
                custom = self.load(custom_config_path).data
            if "dc" not in custom.get("this", {}) and "dc" not in this:
                this["dc"] = utils.HostInfo.dc(this["fqdn"])[:3]
            self.__config = self.query(custom if custom else {}, base=base)
        finally:
            self.__getattr__ = saved_getattr
        return self.__config

    def reload(self, *attrs):
        """ Remove cached config and attributes """
        self.__config = None
        for attr in attrs:
            try:
                delattr(self, attr)
            except AttributeError:
                pass


def ensure_local_settings_defined():
    if import_hook.inside_the_binary():
        if os.environ.get(Registry.CONFIG_ENV_VAR) is None:
            raise RuntimeError("env '{}' should be defined".format(Registry.CONFIG_ENV_VAR))
