"""
Helper tools for console interactions module.
"""
from __future__ import absolute_import, print_function

import os
import abc
import sys
import time
import random
import struct
import inspect
import getpass
import hashlib
import logging
import binascii
import progressbar
import requests.exceptions

from .types import misc as ctm
from .types import user as ctu

from . import rest
from . import proxy
from . import utils
from . import upload


class UpgradeStep(object):
    """
    Single upgrade step object. Will be instantiated and executed automatically by calling `run` method.
    """
    __metaclass__ = abc.ABCMeta

    def pre(self):
        """
        Optional sub-step of the upgrade script. Can be executed while the service is operating in normal mode.
        It is a good place to pre-create some calculated document fields, build indexes, etc.
        Will be automatically executed before the `main` part.
        """
        pass

    @abc.abstractmethod
    def main(self):
        """
        The main part of the upgrade step. It is required to run this part exclusively, while the service
        will be automatically switched into READONLY mode.
        """
        pass

    def post(self):
        """
        Optional sub-step of the upgrade script. Can be executed when the service switched back to normal operation.
        It is a good place to remove some unused document attributes, drop indexes, etc.
        Will be automatically executed after the `main` part.
        """
        pass


class AnsiColorizer(object):
    """
    A colorizer is an object that loosely wraps around a stream, allowing
    callers to write text to the stream in a particular color.
    """
    COLORS = {"black": 30, "red": 31, "green": 32, "yellow": 33, "blue": 34, "magenta": 35, "cyan": 36, "white": 37}

    def __init__(self, bold=True):
        self.bold = bold
        try:
            import termios
            termios.tcgetattr(sys.stdout.fileno())
        except:
            self.colorize = lambda text, _: text

        for color in self.COLORS:
            setattr(self, color, lambda text, _color=color: self.colorize(text, _color))

    def colorize(self, text, color):
        return "\x1b[{}{}m{}\x1b[0m".format(self.COLORS[color], ";1" if self.bold else "", text)


class ProgressBar(object):
    """ Generally used progress bar implementation. """

    def __init__(self, label, maxval, widgets=None, fh=sys.stderr):
        self.pb = progressbar.ProgressBar(
            widgets=widgets or [
                "{}: ".format(label),
                progressbar.Bar(), " ", progressbar.Percentage(),
                " | ", progressbar.FormatLabel("%(value)d/%(max)d"),
                " | ", progressbar.Timer(),
                " | ", progressbar.ETA(),
            ],
            maxval=maxval or 1,
            fd=fh
        ).start()

    @utils.not_each_time(.3)
    def update(self, val):
        self.pb.update(min(val, self.pb.maxval))

    @property
    def current(self):
        return self.pb.currval

    def add(self, val):
        cv = self.pb.currval = min(self.pb.currval + val, self.pb.maxval)
        self.update(cv)

    def finish(self):
        self.pb.finish()


class LongOperation(object):
    """ Helper context manager which prints some long operation information message. """

    def __init__(self, msg, fh=sys.stderr):
        self.fh = fh
        self.has_newline = False
        self.cz = AnsiColorizer()
        self.started = time.time()
        print("{} ... ".format(self.cz.black(msg)), end="", file=self.fh)
        sys.stdout.flush()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, *_):
        status = self.cz.red("Failed") if exc_type else self.cz.black("Done")
        print(
            "{}. [{}]".format(status, self.cz.white(utils.td2str(time.time() - self.started))),
            file=self.fh
        )

    def intermediate(self, msg, cr=False):
        """ Outputs some intermediate message. """
        if not self.has_newline:
            print("", file=self.fh)
            self.has_newline = True
        print("\t{}".format(msg), file=self.fh, end="\r" if cr else "\n")


class Operation(object):
    """ Console operation collector. """
    def __init__(self, fh):
        self.fh = fh
        self._long = None
        self._pbar = None

    def finish(self, maxval=None):
        if self._long:
            self._long.__exit__(None)
            self._long = None
        if self._pbar:
            if maxval:
                self._pbar.pb.maxval = maxval
            self._pbar.finish()
            self._pbar = None

    @property
    def long(self):
        return self._long

    @long.setter
    def long(self, label):
        self._long = LongOperation(label, self.fh)
        self._long.__enter__()

    @property
    def pbar(self):
        return self._pbar

    @pbar.setter
    def pbar(self, args):
        self._pbar = ProgressBar(*args, fh=self.fh)


class Token(object):
    """ Token manipulation routines collection. """

    def __init__(self, url, uname, interactive, fh):
        self.interactive = interactive
        self.uname = uname
        self.url = url
        self.fh = fh

    def ask(self):
        cz = AnsiColorizer()
        while True:
            print(
                "You have to provide it using one of the following methods:\n"
                "  1) Using web browser - just open URL {url}/oauth and paste it here;\n"
                "  2) By typing your password below;\n"
                "  0) Exit the program.".format(url=cz.blue(self.url)),
            )

            answer = ""
            while answer not in ("1", "2", "0"):
                answer = raw_input(cz.green("Please select one of the methods above: (1/2): ")).strip()
            if answer == "1":
                while len(answer) < 32:
                    answer = raw_input(cz.green("Please enter token provided by URL above: ")).strip()
                return answer
            elif answer == "2":
                srv = upload.HTTPHandle.rest_proxy(self.url)
                while True:
                    passwd = getpass.getpass(
                        cz.green("Please enter your domain password (empty string to cancel): ")
                    ).strip()
                    if not passwd:
                        break
                    try:
                        return (srv >> srv.PLAINTEXT).authenticate.oauth.token(
                            {"username": self.uname, "password": passwd}
                        )
                    except requests.exceptions.RequestException as ex:
                        print(cz.red("Server respond {}".format(ex)))
            else:
                raise KeyboardInterrupt()

    def check(self, token):
        op = Operation(self.fh)
        op.long = "Validating OAuth token"
        old_log_level, logging.getLogger().level = logging.getLogger().level, logging.FATAL
        try:
            srv = upload.HTTPHandle.rest_proxy(self.url, token, 10)
            if not (srv >> srv.PLAINTEXT).service.time.current[:]:
                raise Exception("Server did not return correct response.")
        except rest.Client.HTTPError as ex:
            op.long.intermediate(AnsiColorizer().red("Cannot validate token provided: {}".format(ex)))
            return False
        finally:
            logging.getLogger().level = old_log_level
            op.finish()
        return True

    def __try_ssh_key(self, tag, puk):
        cz = AnsiColorizer()
        cnonce = struct.pack("L", int(time.time()) << 32 | random.getrandbits(32))
        fp = binascii.hexlify(puk.get_fingerprint())
        h = hashlib.sha1()
        h.update(self.uname)
        h.update(puk.get_fingerprint())
        h.update(cnonce)
        argspec = inspect.getargspec(puk.sign_ssh_data)
        sig = str(puk.sign_ssh_data(h.digest()) if len(argspec[0]) < 3 else puk.sign_ssh_data(None, h.digest()))

        print(cz.black("Trying to authenticate by digest '{}' with cnonce '{}' signed by {} ('{}')".format(
            h.hexdigest(), binascii.hexlify(cnonce), tag, fp
        )), file=self.fh)

        srv = upload.HTTPHandle.rest_proxy(self.url)
        try:
            return (srv << srv.BINARY >> srv.PLAINTEXT).authenticate["ssh-key"][self.uname][fp](
                sig, cnonce=binascii.hexlify(cnonce)
            ).strip()
        except Exception as ex:
            logging.exception("url=%s", self.url)
            print(cz.yellow("Error getting token by SSH key: {}".format(ex)), file=sys.stderr)
        return None

    def get_token_from_ssh(self, key_file):
        cz = AnsiColorizer()
        try:
            import paramiko
        except ImportError:
            print(
                cz.yellow("This script requires PyCrypto module to be installed to authenticate by SSH keys."),
                file=sys.stderr
            )
            return None

        def load_key(f, passwd=None):
            try:
                return paramiko.RSAKey.from_private_key_file(f, passwd)
            except (IOError, paramiko.SSHException) as ex:
                if "key file is encrypted" in ex.args[0]:
                    return load_key(f, getpass.getpass(cz.green("Please enter your key file password: ")).strip())
                if ex.args[0] == "not a valid RSA private key file":
                    raise ValueError("Only RSA keys are supported yet.")
                raise ValueError(ex.args[0])

        if key_file:
            try:
                key = load_key(key_file)
            except ValueError as ex:
                print(cz.red("Unable to load key '{}': {}".format(key_file, str(ex))), file=sys.stderr)
                sys.exit(2)
            keys = [(key_file, key)] if key else []
        else:
            keys = [("agent key", _) for _ in paramiko.Agent().get_keys()]

        if not keys:
            keys = []
            for key_file in ("id_rsa", "id_dsa"):
                key_file = os.path.expanduser(os.path.join("~", ".ssh", key_file))
                if os.path.exists(key_file):
                    try:
                        keys.append((key_file, load_key(key_file)))
                    except ValueError as ex:
                        print(cz.yellow("Unable to load key '{}': {}".format(key_file, str(ex))), file=sys.stderr)
            if not keys:
                print(
                    cz.yellow(
                        "No private key files found and no running instance of SSH agent with loaded key(s) detected."
                    ),
                    file=self.fh
                )

        for tag, key in keys:
            token = self.__try_ssh_key(tag, key)
            if token:
                return token

    def __call__(self, key_file=None):
        if self.uname == ctu.ANONYMOUS_LOGIN:
            return None

        cz = AnsiColorizer()
        token = self.get_token_from_ssh(key_file)
        if token:
            return token

        cfile = os.path.expanduser(ctm.Upload.TOKEN_CACHE_FILENAME_TMPL.format(self.uname))
        if os.path.isfile(cfile):
            with open(cfile, "r") as fh:
                token = fh.read().strip()
                if not self.check(token):
                    print(cz.red("OAuth token cache seems as invalid. You have to provide valid one."), file=self.fh)
                else:
                    return proxy.OAuth(token)
        else:
            print(
                cz.yellow("There's no cached OAuth token for user '{username}'.\n".format(username=self.uname)),
                file=self.fh
            )

        while self.interactive:
            token = self.ask()
            if not self.check(token):
                print(cz.red("OAuth token provided seems as invalid. Please try again."), file=self.fh)
            else:
                break

        if not token:
            print(cz.red("No valid OAuth token provided."), file=sys.stderr)
            sys.exit(2)

        answer = ""
        while self.interactive and answer not in ("y", "n"):
            answer = raw_input(
                cz.green("I can store the token in UNENCRYPTED form at '{}'. Should I (y/n)? ".format(cfile))
            ).strip().lower() or "y"
        if answer == "y":
            dirname = os.path.dirname(cfile)
            if not os.path.isdir(dirname):
                os.makedirs(dirname, mode=0700)
            with open(cfile, "w") as fh:
                os.fchmod(fh.fileno(), 0600)
                fh.write(token)
        return proxy.OAuth(token)
