from __future__ import absolute_import

import os
import sys
import fcntl
import signal


class Daemon(object):
    def __init__(self,
                 pidfile_name,
                 stdin='/dev/null',
                 stdout='/dev/null',
                 stderr='/dev/null',
                 finalize_fn=None):

        self.stdin = stdin
        self.stdout = stdout
        self.stderr = stderr
        self.pidfile_name = pidfile_name
        dirname = os.path.dirname(pidfile_name)
        if not os.path.isdir(dirname):
            os.makedirs(dirname)
        open(pidfile_name, 'a').close()
        self.pidfile = open(pidfile_name, 'r+')

        def cleanup_handler(signum, frame):
            if finalize_fn:
                try:
                    finalize_fn()
                except Exception as e:
                    sys.stderr.write("Error in finalize_fn: %s\n" % e)
            try:
                fcntl.lockf(self.pidfile, fcntl.LOCK_UN)
            except IOError as e:
                sys.stderr.write("Can't unlock pidfile: %s\n" % e)
                sys.exit(1)
            self.pidfile.close()
            with open(self.pidfile_name, 'r+') as f:
                f.write(' ' * 16)
            sys.exit(0)

        signal.signal(signal.SIGTERM, cleanup_handler)

    def daemonize(self, ch_to="/"):
        try:
            fcntl.lockf(self.pidfile, fcntl.LOCK_EX | fcntl.LOCK_NB)
            fcntl.lockf(self.pidfile, fcntl.LOCK_UN)
        except IOError as e:
            sys.stderr.write(
                "Pidfile %s locked. Process already running?\n" % self.pidfile_name)
            sys.exit(1)

        try:
            pid = os.fork()
            if pid > 0:
                # exit first parent
                os._exit(0)
        except OSError as e:
            sys.stderr.write("fork #1 failed: %d (%s)\n" % (e.errno,
                             e.strerror))
            sys.exit(1)

        # decouple from parent environment
        if ch_to:
            os.chdir(ch_to)
        os.setsid()
        os.umask(0)

        # do second fork
        try:
            pid = os.fork()
            if pid > 0:
                # exit from second parent
                os._exit(0)
        except OSError as e:
            sys.stderr.write("fork #2 failed: %d (%s)\n" % (e.errno,
                             e.strerror))
            sys.exit(1)

        # redirect standard file descriptors
        sys.stdout.flush()
        sys.stderr.flush()
        si = open(self.stdin, 'r')
        so = open(self.stdout, 'a+')
        se = open(self.stderr, 'a+', 0)
        os.dup2(si.fileno(), sys.stdin.fileno())
        os.dup2(so.fileno(), sys.stdout.fileno())
        os.dup2(se.fileno(), sys.stderr.fileno())

        fcntl.lockf(self.pidfile, fcntl.LOCK_EX | fcntl.LOCK_NB)

        data = str(os.getpid())
        data += ' ' * (16 - len(data))
        self.pidfile.write(data)
        self.pidfile.flush()
        os.fsync(self.pidfile.fileno())
