from __future__ import absolute_import

import os
import sys
import importlib


class PackageImporter(object):
    """ Import hook that creates virtual package by ignoring given package name in the importing module name. """

    def __init__(self, pkg):
        self.pkg = pkg

    def __eq__(self, other):
        return isinstance(other, self.__class__) and self.pkg == other.pkg

    def __ne__(self, other):
        return not self.__eq__(other)

    def realname(self, fullname):
        return "__init__" if fullname == self.pkg else fullname[len(self.pkg) + 1:]

    def find_module(self, fullname, path=None):
        if fullname.split(".", 1)[0] == self.pkg:
            try:
                importlib.import_module(self.realname(fullname))
                return self
            except ImportError:
                pass

    def load_module(self, fullname):
        realname = self.realname(fullname)
        mod = importlib.import_module(realname)
        if realname == "__init__":
            mod.__path__ = [os.path.abspath(os.path.dirname(mod.__file__))]
        sys.modules[fullname] = mod
        return mod

    def is_package(self, module):
        fname = sys.modules[module].__file__
        return fname.endswith("__init__.py") or fname.endswith("__init__.pyc")


def inside_the_binary():
    return getattr(sys, "frozen", False)


def setup_sandbox_namespace():
    if not inside_the_binary():  # use hook only for non-binary run
        importer = PackageImporter("sandbox")
        sys.meta_path = [importer] + [mp for mp in sys.meta_path if mp != importer]
