""" RPC errors classification. """


class RPCError(Exception):
    """ Any RPC-related error or generic library error. """
    def __str__(self):
        strfunc = super(Exception, self).__str__
        return {
            0: lambda self: strfunc(),
            1: lambda self: strfunc(),
            2: lambda self: '{%s} %s' % (self.args[1], self.args[0]),
            3: lambda self: '{%s:%d} %s' % (self.args[1], self.args[2], self.args[0]),
        }[len(self.args)](self)


class ProtocolError(RPCError):
    """ Protocol-level internal error (i.e., client or server sent unacceptable data). """


class Timeout(RPCError):
    """ Any operation timeout error. """


class HandshakeError(RPCError):
    """ Initial server handshake problem. """


class HandshakeTimeout(HandshakeError, Timeout):
    """ Initial server handshake timeout. """


class CallError(RPCError):
    """ Any problem related to remote call processing. """


class Reconnect(CallError):
    """ The server currently stopping and cannot serve new requests. Client should reconnect. """


class CallTimeout(CallError, Timeout):
    """ Remote call processing timed out. """


class ServerError(RPCError):
    """ Some server-side exception. """
    def __init__(self, *args, **kwargs):
        if kwargs:
            self.sid = kwargs.get("sid", None)
            self.jid = kwargs.get("jid", None)
            self.module = kwargs.get("module", None)
            self.cls = kwargs.get("cls", None)
            self.tb = kwargs.get("tb", None)
        super(ServerError, self).__init__(*args)

    def __str__(self):
        return (
            "{{{}:{}}}: {}.{}{!r}".format(self.sid, self.jid, self.module, self.cls, self.args)
            if hasattr(self, "sid") else
            super(ServerError, self).__str__()
        )


class SilentException(ServerError):
    """ Instances of this exception and its successors are not logged on server side. """
