# vim: foldmethod=marker

from __future__ import absolute_import, print_function, division

import sys
import time
import errno
import socket
import struct
import threading
import collections

import msgpack
try:
    import gevent
    import gevent.lock
except ImportError:
    gevent = None  # NOQA

from .. import utils

# linux specific
_SO_PEERCRED = 17
_struct3i = struct.Struct('3i')

# darwin specific
_LOCAL_PEERCRED = 0x001
_NGROUPS = 16
_structxucred = struct.Struct('2ih{0}i'.format(_NGROUPS))


PeerId = collections.namedtuple("PeerId", "pid uid gid")


class EOF(Exception):
    pass


class Socket(object):
    class ScopedTimeout(object):
        def __init__(self, sock, timeout):
            self.__socket = None
            if timeout is not None:
                self.__socket = sock
                self.__timeout = sock.timeout
                sock.timeout = timeout

        def __enter__(self):
            pass

        def __exit__(self, *_):
            if self.__socket:
                self.__socket.timeout = self.__timeout

    class _NoLock(object):
        def __enter__(self):
            return self

        def __exit__(self, *_):
            pass

    def __init__(self, sock, gevent_mode=False):
        self.sock = sock
        self.__gevent_mode = gevent_mode
        self.__closed = False

        if gevent_mode:
            self.__lock = gevent.lock.Semaphore(1)
        elif gevent_mode is not None:
            self.__lock = threading.Lock()
        else:
            self.__lock = self._NoLock()

        sockname = sock.getsockname()
        if sockname == '':
            self.__addr = ('', '')  # socketpair?
            self.__binded = True
        elif sockname[1] == 0:
            self.__addr = None
            self.__binded = False
        else:
            self.__addr = sock.getsockname()
            self.__binded = True

        try:
            self.__peer = sock.getpeername()
            self.__connected = True
        except socket.error as ex:
            if ex.errno == errno.ENOTCONN:
                self.__peer = None
                self.__connected = False
            else:
                raise

    # Properties and shortcuts {{{
    @property
    def addr(self):
        assert self.__binded or self.__connected, 'Bind first'
        if self.__addr is None:
            self.__addr = self.sock.getsockname()
        return self.__addr

    @property
    def peer(self):
        assert self.__connected, 'Connect first'
        if self.__peer is None:
            self.__peer = self.sock.getpeername()

        return self.__peer

    @property
    @utils.singleton
    def peerid(self):
        if sys.platform.startswith("linux"):
            res = self.sock.getsockopt(socket.SOL_SOCKET, _SO_PEERCRED, _struct3i.size)
            return PeerId(*_struct3i.unpack(res))
        elif sys.platform.startswith("darwin"):
            res = self.sock.getsockopt(socket.SOL_IP, _LOCAL_PEERCRED, _structxucred.size)
            if len(res) < _structxucred.size:
                return PeerId(None, None, None)
            res = _structxucred.unpack(res)
            uid = res[1]
            groups_len = res[2]
            guids = res[3:3 + groups_len]
            return PeerId(None, uid, guids)
        else:
            return PeerId(None, None, None)

    @property
    def nodelay(self):
        return self.sock.getsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY)

    @nodelay.setter
    def nodelay(self, value):
        self.sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, value)

    @property
    def send_buffer(self):
        return self.sock.getsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF)

    @send_buffer.setter
    def send_buffer(self, value):
        self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF, value)

    @property
    def receive_buffer(self):
        return self.sock.getsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF)

    @receive_buffer.setter
    def receive_buffer(self, value):
        self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF, value)

    @property
    def timeout(self):
        return self.sock.gettimeout()

    @timeout.setter
    def timeout(self, value):
        self.sock.settimeout(value)

    # Properties and shortcuts }}}

    def close(self, shutdown=socket.SHUT_RDWR):
        with self.__lock:
            if self.__closed:
                return

            if shutdown:
                try:
                    self.sock.shutdown(shutdown)
                except:
                    pass

            self.sock.close()
            self.__closed = True

    def recv(self, readby=8192, timeout=None):
        if timeout is not None and timeout < 0:
            return None

        with Socket.ScopedTimeout(self, timeout):
            try:
                return self.sock.recv(readby)
            except socket.error as ex:
                if str(ex) == 'timed out':
                    return None
                raise

    def send(self, data, timeout=None):
        if timeout is not None and timeout < 0:
            return None

        with Socket.ScopedTimeout(self, timeout):
            try:
                return self.sock.send(data)
            except socket.error as ex:
                if ex.errno == errno.EPIPE:
                    raise EOF()
                if str(ex) == 'timed out':
                    return None
                raise

    def read(self, count, readby=0x2000, timeout=None, raiseEof=True):
        with self.__lock:
            buff = []
            left = count

            if timeout:
                deadline = time.time() + timeout

            while left:
                try:
                    data = self.recv(min(readby, left), timeout=deadline - time.time() if timeout else None)
                except socket.error as ex:
                    if ex.errno == errno.EINTR:
                        continue
                    raise

                if data is None:
                    # Timed out
                    return None

                if not len(data):
                    if raiseEof:
                        raise EOF()
                    else:
                        break

                left -= len(data)
                buff.append(data)

            return ''.join(buff)

    def write(self, data, timeout=None):
        with self.__lock:
            length = len(data)
            totalSent = 0

            if timeout:
                deadline = time.time() + timeout

            while totalSent < length:
                try:
                    sent = self.send(data[totalSent:], timeout=deadline - time.time() if timeout else None)
                except socket.error as ex:
                    if ex.errno == errno.EINTR:
                        continue
                    raise
                if not sent:
                    return None
                totalSent += sent

            return totalSent

    def write_msgpack(self, obj, timeout=None):
        return self.write(msgpack.dumps(obj, use_bin_type=True), timeout)

    def read_msgpack(self, readby=0x2000):
        unpacker = msgpack.Unpacker(use_list=True, encoding="utf-8")

        while True:
            try:
                buf = self.recv(readby)
            except socket.error as ex:
                if ex.errno == errno.EINTR:
                    continue
                if ex.errno in (errno.EBADF, ):
                    return
                raise

            if not buf:
                # No more data to process here, connection closed or timed out
                return

            unpacker.feed(buf)
            for data in unpacker:
                stop = yield data
                if stop:
                    yield 0
                    return

    def bind(self, host, port):
        with self.__lock:
            assert not self.__binded
            self.sock.bind((host, port))
            self.__binded = True

    def connect(self, host, port, timeout=None):
        with self.__lock:
            assert not self.__connected

            with Socket.ScopedTimeout(self, timeout):
                if port is None:
                    self.sock.connect(host)
                else:
                    self.sock.connect((host, port))
                self.__connected = True
