# coding: utf-8

from __future__ import print_function
from __future__ import absolute_import

import os
import sys
import time
import uuid
import json
import errno
import Queue
import signal
import datetime
import cStringIO
import traceback
import threading as th

import gevent
import gevent.monkey
import colorlog

import logging
import logging.handlers

import requests

from . import utils
from . import config
from . import statistics
from .types import statistics as ctss


class ExceptionSignalsSenderHandler(logging.Handler):
    def __init__(self, log_file_or_handler=None):
        super(ExceptionSignalsSenderHandler, self).__init__(logging.ERROR)

        if isinstance(log_file_or_handler, logging.Handler):
            if isinstance(log_file_or_handler, logging.FileHandler):
                self.filename = log_file_or_handler.baseFilename
            else:
                self.filename = sys.argv[0]
        else:
            self.filename = log_file_or_handler

    def emit(self, record):
        if not record.exc_info:
            return
        try:
            if statistics.Signaler.instance is not None:
                statistics.Signaler().push(dict(
                    type=ctss.SignalType.EXCEPTION_STATISTICS,
                    timestamp=datetime.datetime.utcnow(),
                    exc_type=record.exc_info[0].__name__,
                    client_tags=config.Registry().client.tags,
                    client_id=config.Registry().this.id,
                    file=self.filename,
                ))
        except:
            self.handleError(record)


class GZipTimedRotatingFileHandler(logging.handlers.TimedRotatingFileHandler):
    def getFilesToDelete(self):
        # copy paste from logging.handlers but remove re match
        dirname, basename = os.path.split(self.baseFilename)
        filenames = os.listdir(dirname)
        result = []
        prefix = basename + "."
        plen = len(prefix)
        for filename in filenames:
            if filename[:plen] == prefix:
                # kill them all!
                result.append(os.path.join(dirname, filename))
        result.sort()
        if len(result) < self.backupCount:
            result = []
        else:
            result = result[:len(result) - self.backupCount]
        return result

    def doRollover(self):
        if self.stream:
            self.stream.close()
            self.stream = None
        dirname, basename = os.path.split(self.baseFilename)
        with utils.FLock(os.path.join(dirname, "rollover.lock")):
            # The block below is copy'n'pasted from the base `doRollover()` method
            # just to determine destination file name.
            currenttime = int(time.time())

            t = self.rolloverAt - self.interval
            if self.utc:
                timetuple = time.gmtime(t)
            else:
                timetuple = time.localtime(t)
                dstnow = time.localtime(currenttime)[-1]
                if dstnow != timetuple[-1]:
                    timetuple = time.localtime(t + 3600 * (1 if dstnow else -1))

            dfn = self.baseFilename + "." + time.strftime(self.suffix, timetuple)
            # EOF copy'n'paste

            if os.path.exists(dfn):
                # This actually means the log file was already rotated by another process.
                self.rolloverAt = self.computeRollover(currenttime)
                while self.rolloverAt <= currenttime:
                    self.rolloverAt += self.interval
                self.stream = self._open()
                return

            super(GZipTimedRotatingFileHandler, self).doRollover()

        # list unzipped files
        from .utils import gzip_file
        filenames = os.listdir(dirname)
        prefix = basename + '.'
        plen = len(prefix)
        suffix = '.gz'
        slen = len(suffix)
        result = []
        for filename in filenames:
            if filename[:plen] == prefix and filename[-slen:] != suffix:
                result.append(filename)
        result.sort()

        # keep last one
        reslut = result[:-1]
        for filename in reslut:
            filename = os.path.join(dirname, filename)
            if os.path.exists(filename):
                try:
                    t = th.Thread(target=gzip_file, args=(filename, filename + suffix), kwargs=dict(remove_source=True))
                    t.daemon = True
                    t.start()
                except:
                    pass


class SnowdenReporter(object):
    __metaclass__ = utils.SingletonMeta

    def __init__(self, logger=None):
        settings = config.Registry()
        self.logger = (logger or logging.getLogger()).getChild("snowden")
        self.conf = settings.common.snowden
        self.__base = {
            "value": None,
            "uid": None,
            "timestamp": None,
            "hostname": settings.this.fqdn,
            "platform_name": settings.this.system.family,
            "user": settings.this.system.user,
            "namespace": "sandbox",
        }

        if self.conf.enabled:
            self._queue = Queue.Queue()
            self._worker = th.Thread(target=self.main)
            self._worker.daemon = True
            self._worker.start()
        else:
            self._queue = None

    @classmethod
    def format_exc(cls, ei):
        """ Formats given exception information into a string. """
        sio = cStringIO.StringIO()
        traceback.print_exception(ei[0], ei[1], ei[2], None, sio)
        s = sio.getvalue()
        sio.close()
        if s[-1:] == "\n":
            s = s[:-1]
        return s

    def _process(self, msgs):
        data = []
        self.__base["timestamp"] = int(time.time())
        for kind, msg, namespace, gsid in msgs:
            r = self.__base.copy()
            r.update({"key": kind, "value": msg, "uid": uuid.uuid4().hex, "session_id": gsid})
            if namespace:
                r["namespace"] = namespace
            data.append(r)

        try:
            data = json.dumps({"reports": data}, ensure_ascii=False, encoding="utf-8")
        except Exception as ex:
            self.logger.error("Unable to create a request to Snowden: %s", ex)
            return

        ex, i = None, None
        for i in xrange(self.conf.attempts):
            try:
                self.logger.debug('Send data to Snowden %r', data)
                requests.post(self.conf.api, data=data, timeout=self.conf.timeout, verify=False).raise_for_status()
                break
            except requests.HTTPError as ex:
                self.logger.error("Bad response from Snowden: %s\nData was: %s", ex, data)
                break
            except Exception as ex:
                if i < self.conf.attempts - 1:
                    time.sleep(0.1 * (i + 1))
        else:
            self.logger.error("Unable to send a request to Snowden: %s", ex)

    def exeption(self, ei=None, namespace=None, gsid=None, async=True):
        """
        Reports given message as an exception to Snowden.

        :param ei:         Exception information to be sent.
                            In case of `None`, current stack's exception information will be sent.
        :param namespace:   Namespace to be used instead of default.
        :param gsid:        Global session identifier to be send if any.
        :param async:       Flags the message should be sent asynchronously.
        """
        if not ei and not async:
            ei = sys.exc_info()
        return self(
            "exception",
            {
                "type": ei[0].__name__,
                "traceback": self.format_exc(ei),
            },
            namespace, gsid, async
        )

    def __call__(self, kind, msg, namespace=None, gsid=None, async=True):
        """
        Reports given message of given type to Snowden.

        :param kind:        Type of the message to be send.
        :param msg:         Message to be send.
        :param namespace:   Namespace to be used instead of default.
        :param gsid:        Global session identifier to be send if any.
        :param async:       Flags the message should be sent asynchronously.
        """
        if not self._queue:
            return self

        if gsid and isinstance(gsid, basestring):
            gsid = gsid.split()
        if async:
            self._queue.put((kind, msg, namespace, gsid))
        else:
            self._process([(kind, msg, namespace, gsid)])
        return self

    def main(self):
        """ Asynchronous worker's main loop. """
        stop = False
        print("Snowden sender thread started.", file=sys.stderr)
        while not stop:
            msgs = []
            while True:
                try:
                    msg = self._queue.get(not msgs)
                    if not msg:
                        stop = True
                    else:
                        msgs.append(msg)
                except Queue.Empty:
                    break

            if not msgs:
                break
            self._process(msgs)

        print("Snowden sender thread stopped.", file=sys.stderr)


class ExceptionSenderHandler(logging.Handler):
    def __init__(self):
        super(ExceptionSenderHandler, self).__init__(logging.ERROR)
        self.sender = SnowdenReporter()

    def emit(self, record):
        if not record.exc_info:
            return
        try:
            msg = self.format(record)
            self.sender(msg)
        except:
            self.handleError(record)


class _SighupHandler(object):
    __metaclass__ = utils.SingletonMeta

    def __init__(self, do_setup):
        # noinspection PyProtectedMember
        if not isinstance(th.current_thread(), th._MainThread):
            return
        signal.signal(
            signal.SIGHUP,
            (
                (
                    (lambda *_: gevent.spawn(self.__on_signal))
                    if gevent.monkey.is_module_patched("threading") else
                    (lambda *_: th.Thread(target=self.__on_signal).start())
                )
                if do_setup else
                signal.SIG_IGN
            )
        )

    @staticmethod
    def __on_signal():
        for logger in utils.chain(logging.root, logging.Logger.manager.loggerDict.values()):
            if not isinstance(logger, logging.Logger):
                continue
            for handler in logger.handlers:
                if isinstance(handler, logging.FileHandler):
                    logger.info("Rotating log %s", handler.baseFilename)
                    handler.acquire()
                    try:
                        handler.stream.close()
                        handler.stream = open(handler.baseFilename, handler.mode)
                    finally:
                        handler.release()


def setup_log(log_file_or_handler, log_level, log_name=None, report_exceptions=False):
    """
    setup application log: rotation, formatting
    if log_name is set - setup specific log
    """
    is_handler = isinstance(log_file_or_handler, logging.Handler)
    _SighupHandler(is_handler and not isinstance(log_file_or_handler, logging.handlers.WatchedFileHandler))
    logger_handler = log_file_or_handler if is_handler else GZipTimedRotatingFileHandler(
        log_file_or_handler,
        when='midnight',
        backupCount=14,
    )
    logger_handler.setFormatter(
        logging.Formatter('%(asctime)s %(levelname)-6s (%(name)s) %(message)s'))
    logger = logging.getLogger(log_name)
    logger.propagate = False  # don't send to root
    logger.addHandler(logger_handler)
    if report_exceptions:
        logger.addHandler(ExceptionSenderHandler())
    logger.addHandler(ExceptionSignalsSenderHandler(log_file_or_handler))
    logger.setLevel(getattr(logging, log_level))
    return logger


def _get_log_root(log_name=None, log_file=None, log_level=None, report_exceptions=False):
    """
    local use only!
    use singleton versions instead
    get specific log
    """
    if not log_file:
        return
    if not log_name:
        log_name = os.path.basename(log_file)
    else:
        log_name = '{0}.log'.format(log_name)
    log_file = os.path.join(os.path.dirname(log_file), log_name)
    log_name = log_name.split('.')[0]
    return setup_log(log_file, log_level, log_name, report_exceptions=report_exceptions)


@utils.singleton
def get_server_log(log_name=None, log_level=None):
    """
    log_name - specific log name without ".log"
    """
    settings = config.Registry()
    log_file = os.path.join(settings.server.log.root, settings.server.log.name)
    if log_level is None:
        log_level = settings.server.log.level
    return _get_log_root(log_name, log_file, log_level)


@utils.singleton
def get_core_log(log_name=None, log_level=None):
    """
        Получить логер для задач ядра

        :param log_name: название логера. Имя файла будет {log_name}.log
        :param log_level: уровень логирования
        :return: Объект логера
    """
    settings = config.Registry()
    core_log_dir = os.path.join(settings.server.log.root, 'core')
    if not os.path.exists(core_log_dir):
        try:
            os.makedirs(core_log_dir)
        except OSError as exc:
            if exc.errno == errno.EEXIST and os.path.isdir(core_log_dir):
                pass
            else:
                raise
    core_log_file = os.path.join(core_log_dir, '{0}.log'.format(log_name))
    if log_level is None:
        log_level = settings.server.services.log.level
    return _get_log_root(log_name, core_log_file, log_level)


@utils.singleton
def get_client_log(log_name=None, log_level=None, log_root=None):
    """
    log_name - specific log name without ".log"
    """
    settings = config.Registry()
    log_file = os.path.join(log_root or settings.client.log.root, settings.client.log.name)
    if log_level is None:
        log_level = settings.client.log.level
    return _get_log_root(log_name, log_file, log_level)


class LogLazyLoader(object):
    """ Lazy loader for global logger objects. """

    def __init__(self, func, args):
        self.constructor = lambda: func(*args)
        self.logger = None

    def __getattr__(self, name):
        if not self.logger:
            self.logger = self.constructor()
        return getattr(self.logger, name)


class MessageAdapter(logging.LoggerAdapter):
    def __init__(self, logger, fmt='%(message)s', data=None):
        super(MessageAdapter, self).__init__(logger, None)
        if data is None:
            data = {}
        else:
            assert 'message' not in data
        self.logger = logger
        self.fmt = fmt
        self.data = data

    @property
    def handlers(self):
        return self.logger.handlers

    def process(self, msg, kwargs):
        self.data['message'] = msg
        msg = self.fmt % self.data
        return msg, kwargs

    def getChild(self, suffix, propagate=None):
        data = self.data.copy()
        data.pop('message', None)
        logger = self.logger.getChild(suffix)
        logger.handlers = self.logger.handlers
        if propagate is not None:
            logger.propagate = propagate
        return type(self)(logger, self.fmt, data)


class TimeDeltaMeasurer(logging.Filter):
    """ Calculates timedelta in seconds between two consecutive records in a log. """

    def __init__(self, name=""):
        super(TimeDeltaMeasurer, self).__init__(name)
        self.now = None

    def filter(self, record):
        now = time.time()
        record.delta = now - self.now if self.now is not None else 0
        self.now = now
        return True


class VaultFilter(logging.Filter):
    """ Changes all occurrences of vault data to it's id """

    def __init__(self):
        super(VaultFilter, self).__init__(name="vault_filter")
        self._vault_data = {}

    def filter_message(self, message):
        for vid, value in self._vault_data.iteritems():
            message = message.replace(value, "<vault record #{}>".format(vid))
        return message

    @staticmethod
    def filter_from_logger(logger):
        for handler in logger.handlers:
            if hasattr(handler, "vault_filter"):
                return handler.vault_filter

    def add_record(self, vid, data):
        self._vault_data[vid] = data

    def update_records(self, vault_filter):
        if vault_filter is not None:
            self._vault_data.update(**vault_filter._vault_data)

    def filter(self, record):
        try:
            record.msg = self.filter_message(record.getMessage())
        except:
            return super(VaultFilter, self).filter(record)
        record.args = ()
        if record.exc_info:
            if not record.exc_text:
                record.exc_text = logging.Formatter().formatException(record.exc_info)
            record.exc_text = self.filter_message(record.exc_text)
        return True


def script_log(verbosity, basedir, fname):
    """ Configures logging for standalone scripts such as `upload.sfx.py` """

    if not os.path.exists(basedir):
        os.makedirs(basedir)
    verbosity = verbosity or 0
    logfile = os.path.join(basedir, fname)
    for i in xrange(9, -1, -1) if os.path.exists(logfile) else []:
        src = ".".join((logfile, str(i))) if i else logfile
        if not os.path.exists(src):
            pass
        elif i == 9:
            os.unlink(src)
        else:
            os.rename(src, ".".join((logfile, str(i + 1))))

    if verbosity > 2:
        __import__("httplib").HTTPConnection.debuglevel = 1

    for no, alias in {
        logging.DEBUG: "DBG",
        logging.INFO: "INF",
        logging.WARNING: "WRN",
        logging.ERROR: "ERR",
        logging.CRITICAL: "CRI",
        logging.NOTSET: "---",
    }.iteritems():
            logging.addLevelName(no, alias)

    handler = logging.FileHandler(logfile)
    handler.setFormatter(logging.Formatter("%(asctime)s %(levelname)-6s (%(name)s) %(message)s"))
    handler.setLevel(logging.DEBUG)
    logger = logging.getLogger()
    logger.addHandler(handler)
    logger.setLevel(logging.DEBUG)

    if verbosity:
        handler = logging.StreamHandler(sys.stderr)
        handler.setFormatter(colorlog.ColoredFormatter(
            "%(cyan)s%(asctime)s%(log_color)s%(bold)s%(levelname)4s%(reset)s "
            "%(bold)s%(cyan)s[%(name)s]%(reset)s %(message)s",
            log_colors={"DBG": "black", "INF": "green", "WRN": "yellow", "ERR": "red", "CRI": "red"}
        ))
        levels = [logging.ERROR, logging.WARNING, logging.INFO, logging.DEBUG]
        handler.setLevel(levels[min(verbosity + 1 if verbosity else 0, len(levels) - 1)])
        logger.addHandler(handler)
    return logfile
