""" OS-specific wrappers and helpers. """

from __future__ import absolute_import

import os
import re
import sys

import pwd
import grp
import getpass
import collections
import itertools as it
import subprocess as sp

from . import utils
from . import config

# noinspection PyUnresolvedReferences
# noinspection PyPackageRequirements
import kernel.util.sys.user


class User(collections.namedtuple("User", ("login", "group", "uid", "gid", "home"))):
    """ User login, group name and home directory location type. """

    SERVICE_USER_ENV = "SANDBOX_USER"
    Users = collections.namedtuple("Users", ("service", "unprivileged"))

    # noinspection PyMethodParameters
    @utils.singleton_classproperty
    def has_root(cls):
        """ Determines the current process is run under root privileges or not. """
        return os.getuid() == 0

    @utils.singleton_classproperty
    def can_root(self):
        """ Determines whether root privileges is available for the current process or not. """
        with open(os.devnull, "w") as devnull:
            return not sp.call(["sudo", "-n", "/bin/true"], stderr=devnull)

    def __new__(cls, login=None, group=None):
        if not login:
            return super(User, cls).__new__(cls, None, None, None, None, None)
        else:
            pw, gid = pwd.getpwnam(login), None
            try:
                if group:
                    gid = grp.getgrnam(group).gr_gid
                else:
                    group = grp.getgrgid(pw.pw_gid).gr_name
            except KeyError:
                pass
            return super(User, cls).__new__(
                cls,
                login,
                group or login,
                pw.pw_uid,
                gid or pw.pw_gid,
                os.path.realpath(os.path.expanduser("~" + login))
            )

    # noinspection PyMethodParameters
    @utils.classproperty
    def service_users(cls):
        """ Returns a pair of service and unprivileged users. """
        return cls.get_service_users(config.Registry())

    @classmethod
    def get_service_users(cls, settings):
        if cls.has_root and os.environ.get(cls.SERVICE_USER_ENV):
            # Local root mode
            return cls.Users(*(User(os.environ[cls.SERVICE_USER_ENV]),) * 2)
        elif settings.client.sandbox_user:
            # Production mode
            return cls.Users(
                User("zomb-sandbox", "sandbox"),
                User(settings.client.sandbox_user)
            )
        else:
            # Local unprivileged mode
            return cls.Users(*(User(getpass.getuser()),) * 2)

    Privileges = kernel.util.sys.user.UserPrivileges


if not User.has_root:  # Fills in and remember availability of root privileges at module import time.
    User.Privileges = type("FakeUserPrivileges", (object,), {
        "__init__": lambda *_, **__: None,
        "__enter__": lambda *_: None,
        "__exit__": lambda *_: None
    })


class CGroup(object):
    """
    Representation of linux control groups (cgroups)

    Usage examples:

    .. code-block:: python

        ## cgroups of the current process
        cgroup = CGroup()

        ## cgroups of the specific process
        cgroup = CGroup(pid=<process id>)

        ## cgroups with path relative to cgroups of the current process
        cgroup = CGroup("custom/path")

        ## cgroups with path relative to cgroups of the specific process
        cgroup = CGroup("custom/path", pid=<process id>)

        ## cgroups with specific absolute path
        cgroup = CGroup("/custom/path")

        ## OS does not support cgroups
        assert CGroup() is None

        ## specification cgroup to subsystems
        cgroup.cpu
        cgroup.cpuacct
        cgroup.devices
        cgroup.freezer
        assert cgroup.freezer.name == "freezer"
        cgroup.memory
        cgroup["memory"]
        ...
        for subsystem in cgroup:  # iterate over all subsystems of the cgroup
            ...

        ## changing path of the cgroup
        cgroup = CGroup("/custom/path")
        assert cgroup.name == "/custom/path"
        cgroup2 = cgroup >> "subpath"
        assert cgroup2.name == "/custom/path/subpath"
        assert (cgroup << 1).name == "/custom"
        assert (cgroup2 << 2).name == "/custom"
        assert (cgroups.memory >> "subpath").cgroup.name == "/custom/path/subpath"
        assert (cgroups2.cpu << 2).cgroup.name == "/custom"

        ## create cgroup for all subsystems, does nothing if already exists
        cgroup = CGroup().create()
        assert cgroup.exists

        ## create cgroup for specific subsystem, does nothing if already exists
        cgroup = CGroup().freezer.create()
        assert cgroup.freezer.exists

        ## delete cgroup for all subsystems, does nothing if not exists
        cgroup = CGroup().delete()
        assert not cgroup.exists

        ## delete cgroup for specific subsystem, does nothing if not exists
        cgroup = CGroup().freezer.delete()
        assert not cgroup.freezer.exists

        ## check the occurrence of the process in the cgroup
        cgroup = CGroup()
        assert os.getpid() in cgroup  # process there is at least in one of subsystem

        ## add process to all subsystems of the cgroup
        cgroup = CGroup("custom/path")
        cgroup += <pid>
        assert <pid> in cgroup
        assert all(<pid> in subsystem for subsystem in cgroup)

        ## add process to specific subsystem of the cgroup
        freezer = CGroup("custom/path").freezer
        freezer += <pid>
        assert <pid> in freezer
        # or
        freezer.tasks = <pid>
        assert <pid> in freezer.tasks

        ## changing cgroup's limits
        cgroup = CGroup()
        cgroup.memory["limit_in_bytes"] = "11G"
        assert cgroup.memory["limit_in_bytes"] == "11G"

        ## common usage example
        import subprocess as sp
        from sandbox import common
        cg = common.cgroup.CGroup("my_group")
        cg.memory["low_limit_in_bytes"] = 16911433728
        cg.memory["limit_in_bytes"] = 21421150208
        cg.cpu["smart"] = 1
        sp.Popen(<cmd>, preexec_fn=cg.set_current)
    """
    ROOT = "/sys/fs/cgroup"
    EXCLUDE_SUBSYSTEMS = ["cpuset"]

    @property
    class Subsystem(object):
        __name = None

        # *_ added just for dummy PyCharm
        def __init__(self, cgroup, *_):
            self.__cgroup = cgroup

        def __call__(self, name, pid=None, owner=None):
            assert name
            path = os.path.join(CGroup.ROOT, name)
            self.__name = real_name = name
            if os.path.exists(path) and os.path.islink(path):
                real_name = os.readlink(path)
            self.__owner = owner
            if pid is not None:
                name_substr = ":{}:".format(real_name)
                filename = "/proc/{}/cgroup".format(pid)
                if os.path.exists(filename):
                    with open(filename) as f:
                        cgroup_name = next(iter(filter(
                            lambda _: name_substr in _, f.readlines()
                        )), "").split(":", 3)[-1].strip()
                    return type(self.__cgroup)(
                        os.path.join(cgroup_name, self.__cgroup.name.lstrip("/")),
                        owner=owner
                    )[self.__name] if cgroup_name else None

            return self

        def __repr__(self):
            return "<{}.{}({}: {})>".format(
                type(self.cgroup).__name__, type(self).__name__, self.__name, self.__cgroup.name
            )

        @property
        def name(self):
            return self.__name

        @property
        def cgroup(self):
            return self.__cgroup

        @property
        def path(self):
            return os.path.join(CGroup.ROOT, self.__name, self.__cgroup.name.lstrip("/"))

        @property
        def exists(self):
            return os.path.exists(self.path)

        @property
        def tasks(self):
            with open(os.path.join(self.path, "tasks")) as f:
                return map(int, f.readlines())

        @tasks.setter
        def tasks(self, pid):
            self.create()
            with open(os.path.join(self.path, "tasks"), "wb") as f:
                f.write("{}\n".format(pid))

        def create(self):
            if not self.exists:
                os.makedirs(self.path, mode=0o755)
                if self.__owner:
                    os.chown(self.path, pwd.getpwnam(self.__owner).pw_uid, -1)
            return self

        def delete(self):
            if self.exists:
                map(lambda _: _.delete(), self)
                os.rmdir(self.path)
            return self

        def set_current(self):
            """ Place current process to the subsystem of cgroup """
            if self.exists:
                self.tasks = os.getpid()

        def __resource_path(self, resource):
            return os.path.join(self.path, ".".join((self.__name, resource)))

        def __getitem__(self, resource):
            with open(self.__resource_path(resource)) as f:
                return map(str.strip, f.readlines())

        def __setitem__(self, resource, value):
            self.create()
            with open(self.__resource_path(resource), "wb") as f:
                f.write("{}\n".format(value))

        def __iter__(self):
            path = self.path
            for _ in os.listdir(path):
                if os.path.isdir(os.path.join(path, _)):
                    yield type(self.__cgroup)(os.path.join(self.__cgroup.name, _), owner=self.__owner)[self.__name]

        def __rshift__(self, name):
            return (self.__cgroup >> name)[self.__name]

        def __lshift__(self, level):
            return (self.__cgroup << level)[self.__name]

        def __contains__(self, pid):
            return (
                self
                if pid in self.tasks else
                next(it.ifilter(None, it.imap(lambda _: _.__contains__(pid), self)), None)
            )

        def __iadd__(self, pid):
            self.tasks = pid

    def __new__(cls, *args, **kws):
        if cls.mounted:
            return super(CGroup, cls).__new__(cls)
        assert not config.Registry().client.cgroup_available, "cgroups available but not mounted"

    def __init__(self, name=None, pid=None, owner=None):
        self.__name = (name or "").rstrip("/") if name != "/" else name
        self.__pid = pid if pid is not None or self.__name.startswith("/") else os.getpid()
        self.__owner = owner

    def __repr__(self):
        return "<{}({})>".format(type(self).__name__, self.__name)

    def __rshift__(self, name):
        return type(self)(os.path.join(self.__name, name) if self.__name else name)

    def __lshift__(self, level):
        return type(self)(reduce(lambda p, _: os.path.dirname(p), xrange(level), self.__name), owner=self.__owner)

    def __getitem__(self, subsystem):
        return self.Subsystem(subsystem, self.__pid, self.__owner)

    def __iter__(self):
        for subsys_name in os.listdir(self.ROOT):
            if subsys_name in self.EXCLUDE_SUBSYSTEMS:
                continue
            subsys_path = os.path.join(self.ROOT, subsys_name)
            if os.path.isdir(subsys_path) and os.path.exists(os.path.join(subsys_path, "tasks")):
                subsys = self.Subsystem(subsys_name, self.__pid, self.__owner)
                if subsys is not None:
                    yield subsys

    def __contains__(self, pid):
        return any(it.imap(lambda subsys: pid in subsys, self))

    def __iadd__(self, pid):
        map(lambda subsys: subsys.__iadd__(pid), self)

    @utils.classproperty
    def mounted(self):
        return os.path.isdir(self.ROOT) and bool(os.listdir(self.ROOT))

    @property
    def name(self):
        return self.__name

    @property
    def exists(self):
        return any(it.imap(lambda subsys: subsys.exists, self))

    def create(self):
        map(lambda subsys: subsys.create(), self)
        return self

    def delete(self):
        map(lambda subsys: subsys.delete(), self)
        return self

    def set_current(self):
        """ Place current process to all subsystems of the cgroup """
        for subsys in self:
            subsys.set_current()

    @property
    def cpu(self):
        return self.Subsystem("cpu", self.__pid, self.__owner)

    @property
    def cpuacct(self):
        return self.Subsystem("cpuacct", self.__pid, self.__owner)

    @property
    def devices(self):
        return self.Subsystem("devices", self.__pid, self.__owner)

    @property
    def freezer(self):
        return self.Subsystem("freezer", self.__pid, self.__owner)

    @property
    def memory(self):
        return self.Subsystem("memory", self.__pid, self.__owner)


class Namespace(object):
    """ Representation of Linux namespaces """

    NS_ID_REGEXP = re.compile(r"([a-z]+):\[(\d+)\]")
    ParsedNamespace = collections.namedtuple("ParsedNamespace", "type id")

    class Type(utils.Enum):
        utils.Enum.lower_case()

        IPC = None
        MNT = None
        NET = None
        PID = None
        USER = None
        UTS = None

    def __init__(self, namespace):
        self.__namespace = namespace
        self.__parsed_namespace = self.ParsedNamespace(*self.NS_ID_REGEXP.match(namespace).groups())

    def __repr__(self):
        return "<{}: {}>".format(type(self).__name__, self.__namespace)

    @property
    def parsed(self):
        return self.__parsed_namespace

    @classmethod
    def from_pid(cls, pid, ns_type):
        try:
            if ns_type not in cls.Type:
                # noinspection PyTypeChecker
                raise ValueError("ns_type must be one of: {}".format(list(cls.Type)))
            ns_path = "/proc/{}/ns/{}".format(pid, ns_type)
            return Namespace(os.readlink(ns_path))
        except (OSError, IOError):
            pass

    @property
    def pids(self):
        pids = []
        root_path = "/proc"
        for pid in os.listdir(root_path):
            if not pid.isdigit():
                continue
            path = os.path.join(root_path, pid, "ns", self.__parsed_namespace.type)
            try:
                if os.readlink(path) == self.__namespace:
                    pids.append(int(pid))
            except (OSError, IOError):
                pass
        return pids


def real_pid(pid, namespace):
    """ Returns pid on host for pid from namespace """
    for ns_pid in namespace.pids:
        try:
            with open("/proc/{}/status".format(ns_pid)) as f:
                for line in f.readlines():
                    rows = line.split()
                    if rows[0] == "NSpid:":
                        if pid == int(rows[-1]):
                            return int(rows[1])
                        break
        except (OSError, IOError):
            pass


def path_env(value, prepend=True, key="PATH"):
    """ Mostly used to start 3rd-party binaries subprocessess. Additionally cut-off skynet binaries path. """
    env = filter(lambda x: not x.startswith("/skynet/python/bin"), os.environ.get(key, "").split(":"))
    (lambda x: env.insert(0, x) if prepend else env.append)(value)
    return ':'.join(filter(None, env))


def system_log_path(prefix="/"):
    """ Returns the path to system log, depending on platform """
    if sys.platform.startswith("linux"):
        valid_paths = ["var/log/messages", "var/log/syslog"]
        for tail in valid_paths:
            path = os.path.join(prefix, tail)
            if os.path.exists(path):
                return path
    elif sys.platform == "darwin":
        return "/var/log/system.log"
