# coding: utf-8

from __future__ import absolute_import

import logging
from .types import misc as ctm
from .types import client as ctc

PLATFORM_ALIASES = {
    "osx_10.9_mavericks": (
        "Darwin-13.4.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.10_yosemite": (
        "Darwin-14.0.0-x86_64-i386-64bit".lower(),
        "Darwin-14.3.0-x86_64-i386-64bit".lower(),
        "Darwin-14.4.0-x86_64-i386-64bit".lower(),
        "Darwin-14.5.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.11_el_capitan": (
        "Darwin-15.3.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.12_sierra": (
        "Darwin-16.5.0-x86_64-i386-64bit".lower(),
        "Darwin-16.7.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.13_high_sierra": (
        "Darwin-17.4.0-x86_64-i386-64bit".lower(),
    ),
    "linux_ubuntu_18.04_bionic": (
        "Linux-4.4.7-1.ng-x86_64-with-Ubuntu-18.04-bionic".lower(),
    ),
    "linux_ubuntu_16.04_xenial": (
        "Linux-4.4.7-1.ng-x86_64-with-Ubuntu-16.04-xenial".lower(),
    ),
    "linux_ubuntu_14.04_trusty": (
        "linux-3.13.0-35-generic-x86_64-with-ubuntu-14.04-trusty".lower(),
    ),
    "linux_ubuntu_12.04_precise": (
        "Linux.3.2.0-38-generic".lower(),
        "Linux.3.2.0-48-generic".lower(),
        "Linux.3.2.0-51-generic".lower(),
        "Linux.3.2.0-43-generic".lower(),
        "Linux-3.2.0-51-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-52-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-48-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-53-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-43-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
    ),
    "linux_ubuntu_10.04_lucid": (
        "Linux.2.6.32-42-server".lower(),
        "Linux.2.6.32-38-server".lower(),
        "Linux.2.6.32-33-server".lower(),
        "Linux-2.6.32-38-server-x86_64-with-Ubuntu-10.04-lucid".lower(),
        "Linux-3.10.9-10yapatch10-x86_64-with-Ubuntu-10.04-lucid".lower(),
        "Linux-2.6.32-42-server-x86_64-with-Ubuntu-10.04-lucid".lower(),
        "Linux-3.2.0-27-server-x86_64-with-Ubuntu-10.04-lucid".lower(),
    ),
    "linux_rhel_santiago": (
        "Linux-3.10.48-8-x86_64-with-redhat-6.5-Santiago".lower(),
        "Linux-3.10.53-14-x86_64-with-redhat-6.5-Santiago".lower(),
    ),
    "freebsd9": (
        "FreeBSD.9.0-STABLE".lower(),
        "FreeBSD-9.0-STABLE-amd64-64bit-ELF".lower(),
        "FreeBSD-9.1-RELEASE-amd64-64bit-ELF".lower(),
    ),
    "freebsd8": (
        "FreeBSD.8.2-STABLE".lower(),
        "FreeBSD-8.2-STABLE-amd64-64bit-ELF".lower(),
    ),
    "linux_ubuntu_14.04_trusty_aarch64": (
        "Linux-3.13.0-48-generic-aarch64-with-Ubuntu-14.04-trusty".lower(),
    )
}

# Map from platform alias to pair of binary platform key and version
# Binary from any platform can run on platforms with matching key and greater or equal versions
PLATFORM_ALIAS_BINARY = {
    "freebsd8": ("freebsd8", 0),  # Didn't check whether freebsd 8 and freebsd 9 are binary compatible.
    "freebsd9": ("freebsd9", 0),  # Just mark them incompatible.

    "linux_rhel_santiago": ("rhel", 0),

    "linux_ubuntu_10.04_lucid": ("ubuntu", 10),
    "linux_ubuntu_12.04_precise": ("ubuntu", 12),
    "linux_ubuntu_14.04_trusty": ("ubuntu", 14),
    "linux_ubuntu_16.04_xenial": ("ubuntu", 16),
    "linux_ubuntu_18.04_bionic": ("ubuntu", 18),

    "osx_10.9_mavericks": ("osx", 1009),
    "osx_10.10_yosemite": ("osx", 1010),
    "osx_10.11_el_capitan": ("osx", 1011),
    "osx_10.12_sierra": ("osx", 1012),
    "osx_10.13_high_sierra": ("osx", 1013),
}

PLATFORM_TO_TAG = {
    "linux_ubuntu_10.04_lucid": ctc.Tag.LINUX_LUCID,
    "linux_ubuntu_12.04_precise": ctc.Tag.LINUX_PRECISE,
    "linux_ubuntu_14.04_trusty": ctc.Tag.LINUX_TRUSTY,
    "linux_ubuntu_16.04_xenial": ctc.Tag.LINUX_XENIAL,
    "linux_ubuntu_18.04_bionic": ctc.Tag.LINUX_BIONIC,
    "linux": ctc.Tag.Group.LINUX,

    "osx_10.9_mavericks": ctc.Tag.OSX_MAVERICKS,
    "osx_10.10_yosemite": ctc.Tag.OSX_YOSEMITE,
    "osx_10.11_el_capitan": ctc.Tag.OSX_EL_CAPITAN,
    "osx_10.12_sierra": ctc.Tag.OSX_SIERRA,
    "osx_10.13_high_sierra": ctc.Tag.OSX_HIGH_SIERRA,
    "osx": ctc.Tag.Group.OSX,
}

TAG_TO_PLATFORM = {str(t): p for p, t in PLATFORM_TO_TAG.iteritems()}

PLATFORM_TAGS = TAG_TO_PLATFORM.viewkeys()

DEFAULT_PLATFORM = "linux_ubuntu_12.04_precise"

LXC_PLATFORMS = tuple(sorted(
    (platform for platform, tag in PLATFORM_TO_TAG.iteritems() if tag in ctc.Tag.Group.LINUX),
    key=lambda p: None if p == DEFAULT_PLATFORM else p
))


def get_platform_alias(platform):
    """
        Получить алиас совместимых платформ для указанной платформы

        :param platform: название платформы
        :return: название алиаса в виде строки; название платформы, если алиаса найти не удалось
        :rtype: str
    """
    if not platform:
        return ""
    platform = str(platform).lower()
    for platform_alias, platforms_list in PLATFORM_ALIASES.items():
        if platform in platforms_list:
            return platform_alias

    # Catch marginal platforms
    if "aarch64" in platform:
        if "trusty" in platform:
            return "linux_ubuntu_14.04_trusty_aarch64"
    else:
        if ("lucid" in platform) or ("squeeze" in platform):
            return "linux_ubuntu_10.04_lucid"
        elif ("precise" in platform) or ("wheezy" in platform):
            return "linux_ubuntu_12.04_precise"
        elif "trusty" in platform:
            return "linux_ubuntu_14.04_trusty"
        elif "xenial" in platform:
            return "linux_ubuntu_16.04_xenial"
        elif "bionic" in platform:
            return "linux_ubuntu_18.04_bionic"
        elif "rhel" in platform or "redhat" in platform:
            if "santiago" in platform:
                return "linux_rhel_santiago"
            else:
                return "linux_rhel"
        elif "cygwin_nt-" in platform:
            return "cygwin_" + platform[10:13]
    return platform


def compare_platforms(platform, another_platform):
    """
        Сравнить платформы с учётом алиасов совместимости.

        :param platform: значение архитектуры для проверки
        :param another_platform: с какой платформой сравнивать.
        :return: True, если платформы совместимы; False в противном случае
        :rtype: bool
    """
    if platform == another_platform:
        return True
    return get_platform_alias(platform) == get_platform_alias(another_platform)


def is_binary_compatible(binary_platform, host_platform):
    """
        Сравнить платформу исполняемого файла и хоста, на котором предполагается исполнение
        :param binary_platform: платформа исполняемого файла
        :param host_platform: платформа хоста
        :return: True, если исполняемый файл может быть запущен на указанной платформе
        :rtype: bool
    """
    if compare_platforms(host_platform, binary_platform):
        return True
    try:
        host_key_version = PLATFORM_ALIAS_BINARY.get(get_platform_alias(host_platform))
        binary_key_version = PLATFORM_ALIAS_BINARY.get(get_platform_alias(binary_platform))

        if host_key_version and binary_key_version:
            return host_key_version[0] == binary_key_version[0] and host_key_version[1] >= binary_key_version[1]
    except Exception:
        logging.getLogger(__name__).exception('Failed to compare platforms')
        return False


def get_arch_from_platform(platform):
    """
    Returns OS family name for given platform string.

    :param platform: string to be examined
    :return: one of "freebsd", "linux", "osx", "cygwin", "win_nt", "any"
    """
    if not platform:
        return ctm.OSFamily.ANY
    if platform in ctm.OSFamily:
        return platform

    if "aarch64" in platform:
        return ctm.OSFamily.LINUX_ARM

    alias = get_platform_alias(platform)
    if "freebsd" in alias:
        return ctm.OSFamily.FREEBSD
    elif "linux" in alias:
        return ctm.OSFamily.LINUX
    elif "osx" in alias or "darwin" in alias:
        return ctm.OSFamily.OSX
    elif "cygwin" in alias:
        return ctm.OSFamily.CYGWIN
    elif "win_nt" in alias:
        return ctm.OSFamily.WIN_NT
    raise ValueError("Unknown platform {!r}".format(platform))
