from __future__ import absolute_import

import os
import re
import sys
import inspect
import logging
import importlib
import collections

from . import utils
from . import config
from . import import_hook
from .types import misc as ctm


logger = logging.getLogger(__name__)

TaskTypeLocation = collections.namedtuple("TaskType", ("package", "cls", "revision"))

_MODULE_NAME_PATTERN = re.compile("[a-zA-Z][_a-zA-Z0-9]*")
_OWNERS_PATTERN = re.compile("OWNER\((?P<owners>[\-\ \n_:a-zA-Z0-9]+)\)")
_PREFIX_PATTERN = re.compile("^(rb|g):")


def load_task_classes(module, force_load=False):
    from sandbox import sdk2

    if isinstance(module, basestring):
        module = importlib.import_module(module)
    if not module:
        return
    if not import_hook.inside_the_binary() or force_load:
        from sandbox.sandboxsdk import task as sdk1_task
        cls = getattr(module, "__Task__", None)
        if cls and cls.type:
            yield cls
        for _, sym in inspect.getmembers(module, inspect.isclass):
            if sym is not cls and issubclass(sym, sdk1_task.SandboxTask) and sym.type:
                yield sym

    for _, sym in inspect.getmembers(module, inspect.isclass):
        if issubclass(sym, sdk2.Task) and sym.name in sdk2.Task:
            yield sym


def _get_package_revision(package):
    dir_path = None
    if package and hasattr(package, "__file__"):
        dir_path = os.path.dirname(package.__file__)
    elif isinstance(package, basestring):
        dir_path = package
    if dir_path:
        rf = os.path.join(dir_path, ".revision")
        if os.path.exists(rf):
            try:
                with open(rf) as f:
                    return int(f.read().strip())
            except ValueError:
                pass
    return None


@utils.singleton
def tasks_dir():
    from sandbox import projects
    return os.path.realpath(os.path.join(os.path.dirname(inspect.getfile(projects)), os.pardir))


def _import_modules_inside_binary():
    from library.python.sfx import extract
    init_suffix = ".__init__"
    modules = []
    for src_path, module_name in extract.iter_py_fs():
        if module_name.startswith("projects.") or module_name.startswith("sandbox.projects."):
            if module_name.endswith(init_suffix):
                module_name = module_name[:-len(init_suffix)]
            modules.append(importlib.import_module(module_name))
    return modules


def _import_modules_on_fs(raise_exceptions):
    projects_dir = tasks_dir()

    if config.Registry().common.installation == ctm.Installation.TEST:
        # TODO: This actually a dirtiest hack to speedup tests and should be relocated to `conftest.py` somehow.
        im = importlib.import_module
        modules = [
            im("sandbox.sdk2.tests"),
            im("sandbox.projects.sandbox.reshare_resource"),
            im("sandbox.projects.sandbox.restore_resource"),
            im("sandbox.projects.sandbox.test_task"),
            im("sandbox.projects.sandbox.test_task_2"),
            im("sandbox.projects.sandbox.test_task_21"),
            im("sandbox.projects.sandbox.unit_test_task"),
            im("sandbox.projects.sandbox.unit_test_task_2"),
            im("sandbox.projects.sandbox.backup_resource_2"),
        ]
    else:
        modules = [_import_module(pkg, raise_exceptions=raise_exceptions) for pkg in get_packages(projects_dir)]

    if config.Registry().common.installation == ctm.Installation.LOCAL:
        import sandbox.sandboxsdk.nodejs.code_generation as nodejs_code
        modules.extend(nodejs_code.generate_nodejs_task_modules(projects_dir, raise_exceptions))
    return modules


def load_project_types(raise_exceptions=False, reuse=False, force_load=False):
    """
    Save names of task types into projects.TYPES,
    task modules loaded into package `projects`, needed to correct working of reload

    If reuse=True, reuse existing projects.TYPES if it exists
    """
    logger.debug("Load project types")

    try:
        from sandbox import projects
    except ImportError:
        if raise_exceptions:
            raise
        logger.warning("Can not import projects")
        return

    if reuse and getattr(projects, "TYPES", None):
        return projects.TYPES

    if import_hook.inside_the_binary():
        modules = _import_modules_inside_binary()
    else:
        modules = _import_modules_on_fs(raise_exceptions)

    try:
        from projects.tests.sdk1_tasks import is_new_sdk1_task
    except ImportError:
        def is_new_sdk1_task(_):
            return False

    types = projects.TYPES = {}
    projects.BLACK_LIST = []
    for m in filter(None, modules):
        revision = _get_package_revision(m)
        for cls in load_task_classes(m, force_load=force_load):
            if cls and cls.type and (cls.__module__ == m.__name__ or getattr(m, "__Task__", None) == cls):
                if is_new_sdk1_task(cls):
                    if cls.type not in projects.BLACK_LIST:
                        projects.BLACK_LIST.append(cls.type)
                else:
                    types[cls.type] = TaskTypeLocation(m.__name__, cls, revision)

    logger.info("Projects code r%s loaded. Modules inspected: %d", getattr(projects, "__revision__", 0), len(modules))
    if projects.BLACK_LIST:
        logger.warning("Blacklisted task types: %s", " ".join(projects.BLACK_LIST))
    return types


def _is_package(root_dir, modules):
    if not all(map(_MODULE_NAME_PATTERN.match, modules)):
        return False
    for ind in xrange(len(modules)):
        init_file_path = os.path.join(root_dir, *(modules[:ind + 1] + ["__init__.py"]))
        if not os.path.isfile(init_file_path):
            return False
    return True


def get_packages(root_dir):
    packages = []
    for dir_path, _, _ in os.walk(root_dir):
        if dir_path == root_dir:
            continue
        modules = os.path.relpath(dir_path, root_dir).split(os.sep)
        if modules and modules[0] == "projects" and _is_package(root_dir, modules):
            packages.append(".".join(modules))
    return packages


def _import_module(name, raise_exceptions=False):
    """
    Safely import a module.

    :return:    Module object in case the module has been loaded successfully and `None` otherwise.
    """
    try:
        return importlib.import_module(name)
    except Exception as e:
        if raise_exceptions:
            raise
        else:
            logger.error("Module {!r} import FAILURE".format(name))
            logger.exception(e)
            return None


def current_projects_revision():
    """
    Get task code version from projects

    :return: version
    :rtype: str
    """
    try:
        from sandbox import projects
        return str(projects.__revision__)
    except Exception as error:
        logger.error("Can't load projects version. Error: {0}".format(error))
        return "cannot get tasks version"


def task_type_relative_path(typename):
    from sandbox import projects
    if typename not in projects.TYPES:
        return None
    fullpath = sys.modules[projects.TYPES[typename].cls.__module__].__file__
    if fullpath.endswith('.pyc'):
        fullpath = fullpath[:-1]
    return os.path.relpath(fullpath, os.path.dirname(os.path.dirname(projects.__file__)))


def task_type_owners(typename):
    import yasandbox.controller.user as user_controller
    task_path = task_type_relative_path(typename)
    if not task_path:
        return []
    yamakelist = os.path.join(tasks_dir(), os.path.dirname(task_path), "ya.make")
    if not os.path.exists(yamakelist):
        return []
    with open(yamakelist, "r") as fo:
        result = re.search(_OWNERS_PATTERN, fo.read())
        owners = result.group("owners").split() if result else []

        logins = set()
        for owner in owners:
            owner, is_group = _PREFIX_PATTERN.subn("", owner)
            logins.update(user_controller.Group.rb_group_content(owner) if is_group else (owner,))
        return list(logins)
