import requests
import logging
import time

from . import config
from . import utils

log = logging.getLogger(__name__)

'''
Constants in block bellow are using as default parameters in calls to staff api.
 CALL_TIMEOUT - timeout of http request, determines how many seconds wait for response.
 CALL_RETRIES - number of retries to make api http call.
 CALL_DELAY - delay between two sequential http calls; seconds to wait before next attempt.
'''
CALL_TIMEOUT = 30
CALL_RETRIES = 2
CALL_DELAY = 1


def _http_call(url, method, params, headers=None, timeouts=None, retry_delay=None):
    request_url = '{}/{}'.format(url, method)
    session = requests.Session()
    request = session.prepare_request(requests.Request('GET', request_url, headers, params=params))
    timeouts = timeouts or [CALL_TIMEOUT for _ in xrange(CALL_RETRIES)]
    retry_delay = retry_delay or CALL_DELAY
    for timeout in timeouts[:-1]:
        try:
            resp = session.send(request, timeout=timeout)
            if resp.ok:
                return resp
            log.warning(
                'staff _http_call(*%s, **%s, **%s) failed with status %s', method, headers, params, resp.status_code)
        except Exception as exc:
            log.exception('staff _http_call(*%s, **%s, **%s) failed. %s', method, headers, params, exc)
        time.sleep(retry_delay)
    return session.send(request, timeout=timeouts[-1])


def _staff_persons_call(login, fields, timeouts=None, retry_delay=None):
    settings = config.Registry()
    url = settings.server.auth.staff.url
    staff_token = utils.read_settings_value_from_file(settings.server.auth.oauth.token)
    method = 'persons'
    params = {'login': login}
    if fields:
        params['_fields'] = ','.join(fields)
    headers = {'Authorization': 'OAuth {}'.format(staff_token)}
    answ = _http_call(url, method, params, headers, timeouts, retry_delay).json()
    res = answ.get('result')
    log.info('_staff_persons_call(%s) result: %s', login, res)
    if res:
        return res[0]


def validate_login(login, timeouts=None, retry_delay=None):
    ret = _staff_persons_call(login, ('login', 'official.is_dismissed', 'official.is_robot'), timeouts, retry_delay)
    if ret:
        return ret['official']['is_dismissed'], ret['official']['is_robot']
    return None


def keys(login, timeouts=None, retry_delay=None):
    return _staff_persons_call(login, ('keys',), timeouts, retry_delay).get('keys', []) or []
