import logging
import requests

from . import utils


class Method(utils.Enum):
    SEND_MESSAGE = "sendMessage"
    SEND_STICKER = "sendSticker"
    GET_UPDATES = "getUpdates"
    GET_ME = "getMe"
    DELETE_MESSAGE = "deleteMessage"


class ParseMode(utils.Enum):
    MARKDOWN = "Markdown"
    HTML = "HTML"


class LoggerLevelChanger(object):
    def __init__(self, logger, level):
        self.__logger = logger
        self.__level = level
        self.__previous_level = logger.level

    def __enter__(self):
        self.__logger.setLevel(self.__level)

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.__logger.setLevel(self.__previous_level)


class TelegramError(Exception):
    """ An exception just to wrap internal failures with """


class TelegramBot(object):

    BASE_URL = "https://api.telegram.org/bot"
    REQUEST_TIMEOUT = 30  # 30s should be enough, eh?

    def __init__(self, bot_token, logger=logging):
        self.__token = bot_token
        self.__logger = logger

    def __method(self, http_method, api_method, *args, **kwargs):
        try:
            method = getattr(requests, http_method)
            if method is None:
                raise TelegramError("Invalid HTTP method requested")

            kwargs.update(timeout=self.REQUEST_TIMEOUT)
            with LoggerLevelChanger(logging.getLogger(requests.__name__), logging.INFO):
                self.__logger.debug("%s %s: %s, %s", http_method.upper(), api_method, args, kwargs)
                response = method("{}{}/{}".format(self.BASE_URL, self.__token, api_method), *args, **kwargs)
                self.__logger.debug("Response: %s", response.text)
            return response.json()

        except Exception as exc:
            raise TelegramError(exc)

    def _get(self, api_method, *args, **kwargs):
        return self.__method("get", api_method, *args, **kwargs)

    def _post(self, api_method, *args, **kwargs):
        return self.__method("post", api_method, *args, **kwargs)

    def send_sticker(self, chat_id, sticker, **kwargs):
        params = {
            "chat_id": chat_id,
            "sticker": sticker,
        }
        params.update(kwargs)
        return self._get(Method.SEND_STICKER, params=params)

    def send_message(self, chat_id, text, parse_mode="", **kwargs):
        params = {
            "chat_id": chat_id,
            "text": text,
            "parse_mode": parse_mode,
        }
        params.update(kwargs)
        return self._post(Method.SEND_MESSAGE, data=params)

    def delete_message(self, chat_id, message_id):
        params = {
            "chat_id": chat_id,
            "message_id": message_id,
        }
        return self._post(Method.DELETE_MESSAGE, params=params)

    def get_updates(self, offset=None, **kwargs):
        params = {"offset": offset} if offset else {}
        params.update(kwargs)
        return self._get(Method.GET_UPDATES, params=params)

    def get_me(self):
        return self._get(Method.GET_ME)
