"""
This file's content is mostly coming from a patch, which is suggested by Sebastian Noack
in a scope of Python's bug #8800: http://bugs.python.org/issue8800
"""

from __future__ import absolute_import

import os
import sys
import time
import thread
import logging
import threading as th
import traceback as tb

from . import patterns

# Maximum allowed timeout, which will not raise `OverflowError` exception.
TIMEOUT_MAX = 2 << 30


class Condition(th._Condition):
    def wait_for(self, predicate, timeout=None):
        """Wait until a condition evaluates to True.

        predicate should be a callable which result will be interpreted as a
        boolean value.  A timeout may be provided giving the maximum time to
        wait.

        """
        endtime = None
        waittime = timeout
        result = predicate()
        while not result:
            if waittime is not None:
                if endtime is None:
                    endtime = time.time() + waittime
                else:
                    waittime = endtime - time.time()
                    if waittime <= 0:
                        break
            self.wait(waittime)
            result = predicate()
        return result


# noinspection PyAbstractClass
class _RWLockBase(patterns.RWLock):
    """
    A Reader/Writer lock, that can be acquired either in reader or writer mode.
    In reader mode, the lock may be held by any number of threads.
    In writer mode, only one thread can hold the lock.
    A RWLock is reentrant in a limited fashion:  A thread holding the lock
    can always get another read lock.  And a thread holding an write lock
    can get another lock (read or write.)
    But in general, a thread holding a read lock cannot recursively acquire an
    write lock.
    Any acquire_read() or acquire_write must be matched with a release().
    """

    # Proxy classes that do either read or write locking
    class _ReaderLock(object):
        def __init__(self, lock):
            self.lock = lock
            super(_RWLockBase._ReaderLock, self).__init__()

        @staticmethod
        def _timeout(blocking, timeout):
            # A few sanity checks to satisfy the unittests.
            if timeout < 0 and timeout != -1:
                raise ValueError("invalid timeout")
            if timeout > TIMEOUT_MAX:
                raise OverflowError
            if blocking:
                return timeout if timeout >= 0 else None
            if timeout > 0:
                raise ValueError("can't specify a timeout when non-blocking")
            return 0

        def acquire(self, blocking=True, timeout=-1):
            return self.lock.acquire_read(self._timeout(blocking, timeout))

        def release(self):
            self.lock.release()

        def __enter__(self):
            self.acquire()

        def __exit__(self, exc, val, tb):
            self.release()

        def _is_owned(self):
            raise TypeError("a reader lock cannot be used with a Condition")

    class _WriterLock(_ReaderLock):
        def acquire(self, blocking=True, timeout=-1):
            return self.lock.acquire_write(self._timeout(blocking, timeout))

        def _is_owned(self):
            return self.lock._is_owned()

        def _release_save(self):
            return self.lock._release_save()

        def _acquire_restore(self, arg):
            return self.lock._acquire_restore(arg)

    @property
    def reader(self):
        """ Return a proxy object that acquires and releases the lock in read mode. """
        return self._ReaderLock(self)

    @property
    def writer(self):
        """ Return a proxy object that acquires and releases the lock in write mode. """
        return self._WriterLock(self)


class RWLock(_RWLockBase):
    """ FIFO reader-writer recursive lock. """

    def __init__(self):
        self.cond = Condition()
        self.state = 0    # positive is shared count, negative exclusive count
        self.owning = []  # threads will be few, so a list is not inefficient
        super(RWLock, self).__init__()

    def acquire_read(self, timeout=None):
        """ Acquire the lock in shared mode. """
        with self.cond:
            return self.cond.wait_for(self._acquire_read, timeout)

    def acquire_write(self, timeout=None):
        """ Acquire the lock in exclusive mode. """
        with self.cond:
            return self.cond.wait_for(self._acquire_write, timeout)

    def release(self):
        """ Release the lock. """
        with self.cond:
            me = thread.get_ident()
            try:
                self.owning.remove(me)
            except ValueError:
                raise RuntimeError("cannot release an un-acquired lock")
            if self.state > 0:
                self.state -= 1
            else:
                self.state += 1
            if self.state == 0:
                self.cond.notify_all()

    def _acquire_write(self):
        # we can only take the write lock if no one is there, or we already hold the lock
        me = thread.get_ident()
        if self.state == 0 or (self.state < 0 and me in self.owning):
            self.state -= 1
            self.owning.append(me)
            return True
        if self.state > 0 and me in self.owning:
            raise RuntimeError("cannot upgrade RWLock from read to write")
        return False

    def _acquire_read(self):
        if self.state < 0:
            # lock is in write mode.  See if it is ours and we can recurse
            return self._acquire_write()
        self.state += 1
        self.owning.append(thread.get_ident())
        return True

    # Interface for condition variable.  Must hold an exclusive lock since the
    # condition variable's state may be protected by the lock
    def _is_owned(self):
        return self.state < 0 and thread.get_ident() in self.owning

    def _release_save(self):
        # In a exclusively locked state, get the recursion level and free the lock
        with self.cond:
            if thread.get_ident() not in self.owning:
                raise RuntimeError("cannot release an un-acquired lock")
            r = self.owning
            self.owning = []
            self.state = 0
            self.cond.notify_all()
            return r

    def _acquire_restore(self, x):
        # Reclaim the exclusive lock at the old recursion level
        self.acquire_write()
        with self.cond:
            self.owning = x
            self.state = -len(x)


class FairRWLock(RWLock):
    """ A specialization giving writer priority. """

    def __init__(self):
        super(FairRWLock, self).__init__()
        self.waiting = 0

    def acquire_write(self, timeout=None):
        """ Acquire the lock in exclusive mode. """
        with self.cond:
            self.waiting += 1
            try:
                return self.cond.wait_for(self._acquire_write, timeout)
            finally:
                self.waiting -= 1

    def _acquire_read(self):
        if self.state < 0:
            # lock is in write mode.  See if it is ours and we can recurse.
            return self._acquire_write()

        # Implement "exclusive bias" giving exclusive lock priority.
        me = thread.get_ident()
        if not self.waiting:
            ok = True  # no exclusive acquires waiting.
        else:
            # Recursion must have the highest priority
            ok = me in self.owning

        if ok:
            self.state += 1
            self.owning.append(me)
        return ok


class KamikadzeThread(th.Thread):
    def __init__(self, ttl, logger=None, name=None):
        super(KamikadzeThread, self).__init__(name=name or "Kamikadze thread")
        self.logger = logger.getChild("kamikadze") if logger else logging
        self.deadline = time.time() + ttl
        self.stopping = th.Event()

    @property
    def ttl(self):
        return self.deadline - time.time()

    @ttl.setter
    def ttl(self, value):
        self.logger.debug("Setting TTL to %ds.", value)
        self.deadline = time.time() + value

    def stop(self):
        self.stopping.set()
        return self

    def run(self):
        now = time.time()
        self.logger.info("Thread started.")
        while self.deadline > now:
            if self.stopping.wait(self.deadline - now):
                self.logger.info("Thread stopped.")
                return
            now = time.time()
        self.logger.warn("TTL expired. Killing self.")
        for tid, frame in sys._current_frames().iteritems():
            self.logger.info("Thread 0x%x\nFrame:\n%s\n---\n", tid, "".join(tb.format_stack(frame)))
        os._exit(42)
