import re
import operator as op
import itertools as it

from .. import utils

CUSTOM_TAG_PREFIX = "CUSTOM_"
CUSTOM_TAG_PREFIX_RE = re.compile(r"{}[A-Z\d_]+".format(CUSTOM_TAG_PREFIX))

SERVICE_TOKEN = "0" * 32


class ReloadCommand(utils.Enum):
    """ Client reload commands in order of importance (processing by client) """
    utils.Enum.preserve_order()
    utils.Enum.lower_case()

    SHUTDOWN = None
    RESET = None
    REBOOT = None
    RESTART = "reload"
    CLEANUP = None


class RemovableResources(utils.Enum):
    """ A kind of client's cache removable resources """
    utils.Enum.lower_case()

    DELETED = None
    EXTRA = None
    REPLICATED = None


class ServiceActivity(utils.Enum):
    utils.Enum.lower_case()

    NONE = None
    CLEANUP = None
    RELOAD = None
    MAINTAIN = None


class ClientAvailability(utils.Enum):
    utils.Enum.lower_case()

    ALIVE = None
    DEAD = None
    UNKNOWN = None


class Tag(utils.Enum):
    """
    Client tags

    @DynamicAttrs
    """

    # noinspection PyPep8Naming
    class __metaclass__(utils.Enum.__metaclass__):
        def __new__(mcs, name, bases, namespace):
            cls = utils.Enum.__metaclass__.__new__(mcs, name, bases, namespace)

            group_defaults = set()
            for group in cls.Group:
                if not group.primary:
                    continue
                positive_default = set()
                negative_default = set()
                for item_name in group:
                    item = getattr(cls, item_name)
                    if item.default is True:
                        positive_default.add(item_name)
                    elif item.default is False:
                        negative_default.add(item_name)
                if positive_default or negative_default:
                    frozenset.__setattr__(group, "positive_default", frozenset(positive_default))
                    frozenset.__setattr__(group, "negative_default", frozenset(negative_default))
                    group_defaults.add(group)
            type.__setattr__(cls.Group, "defaults", frozenset(group_defaults))

            return cls

        def __getattr__(cls, name):
            if not name.startswith(CUSTOM_TAG_PREFIX) and name != CUSTOM_TAG_PREFIX or not name.isupper():
                raise AttributeError("Tag with name {!r} is not defined".format(name))
            return type(name, (cls.Item,), {})

    class Query(object):
        NOT = "~"
        ALLOWED_SYMBOLS = re.compile(r"[A-Z\d_()|&~\s]+")
        __predicates_cache = {}

        # noinspection PyPep8Naming
        class __metaclass__(type):
            def __call__(cls, dnf):
                if isinstance(dnf, basestring):
                    # noinspection PyUnresolvedReferences
                    return cls.parse(dnf)
                return type.__call__(cls, dnf)

        def __init__(self, dnf):
            self.__dnf = tuple(sorted(set(it.imap(lambda _: tuple(sorted(set(_))), dnf))))
            self.__repr = str(dnf)
            self.__prio = 0

        def __repr__(self):
            return self.__repr

        def __eq__(self, other):
            return self.__dnf == other.__dnf

        def __ne__(self, other):
            return self.__dnf != other.__dnf

        def __or__(self, item):
            query = type(self)(set(self) | set(item.query))
            query.__prio = -2
            query.__repr = " | ".join((self.__repr, item.query.__repr))
            return query

        def __and__(self, item):
            query = type(self)(it.imap(it.chain.from_iterable, it.product(list(self), list(item.query))))
            query.__prio = -1
            query.__repr = " & ".join(
                map(lambda _: "({})".format(_.__repr) if _.__prio < query.__prio else _.__repr, (self, item.query))
            )
            return query

        def __invert__(self):
            query = type(self)(it.product(*map(lambda _: it.imap(op.inv, _) if isinstance(_, tuple) else ~_, self)))
            query.__repr = "~{}".format("({})".format(self.__repr) if self.__prio < query.__prio else self.__repr)
            return query

        def __iter__(self):
            return iter(self.__dnf)

        def __getstate__(self):
            return repr(self)

        def __setstate__(self, state):
            self.__dnf = self.parse(state).__dnf

        def __nonzero__(self):
            return bool(self.__dnf)

        @property
        def query(self):
            return self

        def check(self):
            if not self.__dnf:
                raise ValueError("Query expression is identically false")

            singles = set(it.chain.from_iterable(it.ifilter(lambda _: len(_) == 1, self.__dnf)))
            for item in self.__dnf:
                if set(it.imap(op.inv, item)) <= singles:
                    raise ValueError("Query expression is identically true")

        @classmethod
        def parse(cls, query):
            if not cls.ALLOWED_SYMBOLS.match(query):
                raise ValueError(
                    "Query contains illegal symbols, must matching regular expression '{}'".format(
                        cls.ALLOWED_SYMBOLS.pattern.replace("\\\\", "\\")
                    )
                )
            env = {
                k: v
                for k, v in it.chain(
                    Tag.__dict__.iteritems(),
                    Tag.Group.__dict__.iteritems(),
                    ((t, getattr(Tag, t)) for t in re.findall(CUSTOM_TAG_PREFIX_RE, query))
                )
                if k.isupper()
            }
            return eval(query, env, {}).query

        @classmethod
        def predicates(cls, tags, ignore_defaults=False):
            cached = cls.__predicates_cache.get(tags)
            if cached:
                return cached
            if tags:
                result = map(
                    lambda i: reduce(
                        lambda r, t: r[str(t).startswith(cls.NOT)].add(str(t).replace(cls.NOT, "")) or r,
                        i,
                        (set(), set())
                    ),
                    cls(tags)
                )
            else:
                result = [(set(), set())]
            for p, n in result:
                n.add(str(Tag.Group.SERVICE))
                if ignore_defaults or any(it.ifilter(lambda t: t in Tag.Group.CUSTOM, p)):
                    continue
                for group in Tag.Group.defaults:
                    if set(group) & p:
                        continue
                    p.update(group.positive_default)
                    n.update(group.negative_default)
            cls.__predicates_cache[tags] = result
            return result

    class Item(utils.Enum.Item):
        default = None

        # noinspection PyPep8Naming
        class __metaclass__(utils.Enum.Item.__metaclass__):
            def __call__(cls, doc=None, default=None):
                return type(cls)(cls.__name__, (cls,), {"__doc__": doc, "default": default})

            @property
            def query(cls):
                return Tag.Query(cls)

            def __iter__(cls):
                yield cls

            def __or__(cls, item):
                return cls.query | item.query

            def __and__(cls, item):
                return cls.query & item.query

            def __invert__(cls):
                value = str(cls)
                return type(
                    value[len(Tag.Query.NOT):] if value.startswith(Tag.Query.NOT) else Tag.Query.NOT + value,
                    (cls,),
                    {}
                )

        # noinspection PyUnusedLocal
        def __init__(self, doc=None, default=None):
            pass  # for auto completion in IDE

    class Group(utils.GroupEnum):
        """ Groups of tags """
        defaults = frozenset()

        class Item(utils.GroupEnum.Item):
            positive_default = frozenset()
            negative_default = frozenset()

            @property
            def query(self):
                return Tag.Query(type(str(self), (Tag.Item,), {}))

            def __or__(self, item):
                return self.query | item.query

            def __and__(self, item):
                return self.query & item.query

            def __invert__(self):
                return ~self.query

        class CustomItem(Item):
            def __contains__(self, item):
                return str(item).startswith(CUSTOM_TAG_PREFIX)

            def __len__(self):
                return 42

        # primary groups
        PURPOSE = Item("Purpose tags")
        VIRTUAL = Item("Virtualization types")
        DENSITY = Item("Density of the executing tasks")
        OS = Item("OS")
        CPU = Item("Processors")
        DISK = Item("Disk types")
        NET = Item("IP families")
        SERVICE = Item("Service tags")
        FEATURES = Item("Service features")
        VCS = Item("Tags related to version control system (Arcadia, mostly)")
        DC = Item("Data center tags")

        # secondary groups
        LINUX = Item("Linux versions")
        OSX = Item("OS X versions")
        INTEL = Item("Intel processor")
        AMD = Item("AMD processor")

        CUSTOM = CustomItem("Custom tags")

    @classmethod
    def filter(cls, tags, include_groups=True):
        tags = set(it.ifilter(lambda t: t in cls or str(t).startswith(CUSTOM_TAG_PREFIX), tags))
        # noinspection PyTypeChecker
        return set(
            it.chain(tags, it.imap(str, it.ifilter(lambda g: any(it.imap(lambda t: t in g, tags)), cls.Group)))
            if include_groups else
            tags
        )

    with Group.PURPOSE:
        # Please don't use these tags if you are not sure that you know what you are doing.
        GENERIC = Item("generic sandbox clients - default tag", default=True)
        SERVER = Item("sandbox-server* hosts")
        STORAGE = Item("sandbox-storage* hosts")
        POSTEXECUTE = Item("Hosts where transient statuses (STOPPING, RELEASING) are executing")
        RTY = Item("sandbox-rty* hosts")
        BROWSER = Item("hosts for Browser testing")
        BROWSER_MOBILE = Item("Hosts for mobile browser testing")
        YABS = Item("hosts for YABS testing")
        AQUA = Item("sandbox-agent-* hosts for generic AQUA tests (Direct, etc)")
        AQUA_ZEN = Item("K@portal_zen_aerotest for AQUA tests of Zen project")
        UKROP = Item("sandbox-ukrop* hosts for release brunches build")
        SKYNET = Item("hosts for skynet build")
        PORTO = Item("hosts with Porto")
        WS = Item("wsXX-* hosts for Report testing")
        OXYGEN = Item("sandbox-oxygen* hosts")
        COCAINE = Item("hosts for needs of Cocaine")
        ANTISPAM = Item("sandbox-antispam* hosts")
        MARKET = Item("Hosts for market")
        ACCEPTANCE = Item("hosts performing the acceptance of the Sandbox")
        VOID = Item("Special tag to define empty set of hosts")

    with Group.VIRTUAL:
        LXC = Item("hosts where tasks are executing in LXC-containter (multios clients, priveleged tasks, etc...)")
        PORTOD = Item("hosts where tasks are executing in Porto containters", default=False)
        OPENSTACK = Item("hosts in openstack")

    with Group.DENSITY:
        MULTISLOT = Item("Multi slot host (1 CPU, 4 GiB)", default=False)
        CORES1 = Item("Multi slot host (1 CPU, 4 GiB)", default=False)
        CORES4 = Item("Fat multi slot host (4 CPU, 16 GiB)", default=False)
        CORES8 = Item("CPU with 8 cores")
        CORES16 = Item("CPU with 16 cores")
        CORES24 = Item("CPU with 24 cores")
        CORES32 = Item("CPU with 32 cores")
        CORES56 = Item("CPU with 56 cores")
        CORES64 = Item("CPU with 64 cores")

    with Group.OS:
        LINUX_LUCID = Item("Linux Ubuntu 10.04")
        LINUX_PRECISE = Item("Linux Ubuntu 12.04")
        LINUX_TRUSTY = Item("Linux Ubuntu 14.04")
        LINUX_XENIAL = Item("Linux Ubuntu 16.04")
        LINUX_BIONIC = Item("Linux Ubuntu 18.04")
        OSX_MAVERICKS = Item("OS X 10.9")
        OSX_YOSEMITE = Item("OS X 10.10")
        OSX_EL_CAPITAN = Item("OS X 10.11")
        OSX_SIERRA = Item("OS X 10.12")
        OSX_HIGH_SIERRA = Item("OS X 10.13")
        CYGWIN = Item("Windows/Cygwin platform (experimental)")
        FREEBSD = Item("FreeBSD (deprecated)")

    with Group.CPU:
        INTEL_E5_2650 = Item()
        INTEL_E5_2660 = Item()
        INTEL_E5_2660V1 = Item()
        INTEL_E5_2660V4 = Item()
        INTEL_E5645 = Item()
        INTEL_E312XX = Item()
        INTEL_3720QM = Item()
        INTEL_X5675 = Item()
        AMD6176 = Item()
        ARM = Item()
        KVM = Item()

    with Group.DISK:
        HDD = Item("Client has HDD")
        SSD = Item("Client has SSD")

    with Group.NET:
        IPV4 = Item("Client has IPv4-address")
        IPV6 = Item("Client has IPv6-address")

    with Group.FEATURES:
        NEW_LAYOUT = Item("New resources location layout")
        SQUASHFS_TASKS_IMAGE = Item("SANDBOX-5068: Experimental feature - use SquashFS tasks image")

    with Group.VCS:
        ARCADIA_HG = Item("Has actual Arcadia Hg cache")

    with Group.SERVICE:
        # Don't even think about using these tags
        NEW = Item("New client")
        MAINTENANCE = Item("Client is on maintenance")

    with Group.DC:
        IVA = Item()
        SAS = Item()
        MYT = Item()
        VLA = Item()
        MAN = Item()
        UNK = Item()


Tag.Group.LINUX = (
    Tag.LINUX_LUCID,
    Tag.LINUX_PRECISE,
    Tag.LINUX_TRUSTY,
    Tag.LINUX_XENIAL,
    Tag.LINUX_BIONIC,
)
Tag.Group.OSX = (Tag.OSX_MAVERICKS, Tag.OSX_YOSEMITE, Tag.OSX_EL_CAPITAN, Tag.OSX_SIERRA, Tag.OSX_HIGH_SIERRA)
Tag.Group.INTEL = filter(lambda _: "INTEL" in str(_), Tag.Group.CPU)
Tag.Group.AMD = filter(lambda _: "AMD" in str(_), Tag.Group.CPU)

Query = Tag.Query


class DiskStatus(utils.Enum):
    """ Disk space status """
    utils.Enum.lower_case()

    OK = None
    WARNING = None
    CRITICAL = None


class JobAction(utils.Enum):
    EXECUTE = None
    SUSPEND = None
