import collections

from .. import utils


class Network(object):
    """ Various network-related commonly used constants """

    class Family(utils.Enum):
        """
        These constants represent the address (and protocol) families to be used commonly
        by different clients and server to report and check client's networking abilities.
        """

        V4 = 4
        V6 = 6

    class Type(utils.Enum):
        """
        Network schemas for containers (related: `bin.client.LXCNetwork`, `bin.client.PortoNetwork`)
        """

        L3 = None
        NAT = None
        MACVLAN = None


class OSFamily(utils.Enum):
    """ Supported operating system families. """
    utils.Enum.lower_case()

    ANY = None
    OSX = None
    LINUX = None
    FREEBSD = None
    LINUX_ARM = None
    CYGWIN = None
    WIN_NT = None


class StorageDeviceType(utils.Enum):
    """ Storage device types available on clients. """
    utils.Enum.lower_case()

    HDD = None
    SSD = None


class RamDriveType(utils.Enum):
    """ Supported RAM drive types. """
    utils.Enum.lower_case()

    TMPFS = None


class FileType(utils.Enum):
    """ File types """
    utils.Enum.lower_case()

    FILE = None
    DIRECTORY = None


class CpuManufacturer(utils.Enum):
    utils.Enum.lower_case()

    INTEL = None
    AMD = None


class CpuArchitecture(utils.Enum):
    utils.Enum.lower_case()

    I386 = None
    X86_64 = None
    IA64 = None


class Installation(utils.Enum):
    """ Installation type """

    class Group(utils.GroupEnum):
        LOCAL = None
        NONLOCAL = None

    with Group.LOCAL:
        LOCAL = None
        TEST = None

    with Group.NONLOCAL:
        PRE_PRODUCTION = None
        PRODUCTION = None


class OperationMode(utils.Enum):
    """ Operation mode """
    utils.Enum.lower_case()

    NORMAL = None
    READ_ONLY = None
    PRE_READ_ONLY = None


class DnsType(utils.Enum):
    """ Type of resolving config """
    utils.Enum.lower_case()

    DEFAULT = None  # system default nameservers
    DNS64 = None  # [d]ns64-cache as nameserver
    LOCAL = None  # use `/etc/resolv.conf.local` config if it exists


class DiskUsageType(utils.Enum):
    """ Type of task disk usage file """
    PEAK = None  # disk usage when task exceeds its disk requirements
    FINAL = None  # disk usage when task had finished


class MongoState(utils.Enum):
    """ Mongo DB process state """

    STARTUP = None
    PRIMARY = None
    SECONDARY = None
    RECOVERING = None
    STARTUP2 = None
    UNKNOWN = None
    ARBITER = None
    DOWN = None
    ROLLBACK = None
    REMOVED = None


class Upload(object):
    """ Upload state. """

    #: Filename template to cache OAuth token.
    TOKEN_CACHE_FILENAME_TMPL = "~/.sandbox/{}.oauth"
    #: Current upload protocol version
    VERSION = 2

    class Stream(utils.Enum):
        """ Type of upload stream. """

        PLAIN = 1   # Plain tarball will be used for upload
        ZIPPED = 1  # The upload tarball stream will be zipped additionally

    class State(object):
        """ Base type for state reporting type. """
        __slots__ = ()

        def __repr__(self):
            return "".join((
                "<", self.__class__.__name__, ">",
                "(", ", ".join(repr(getattr(self, _)) for _ in self.__slots__), ")"
            ))

    class Check(State):
        """ Files check stage. Calculates total files amount and total size. """
        __slots__ = ("amount", "size")

        def __init__(self):
            self.amount = 0
            self.size = 0

    class Prepare(State):
        """ Prepare stage. Provides task and resource IDs. """
        __slots__ = ("task_id", "resource_id")

        def __init__(self):
            self.task_id = None
            self.resource_id = None

    class DataTransfer(State):
        """ Data transfer stage. Provides amount of data in bytes, which has been transferred, and SHA1 of it. """
        __slots__ = ("done", "total", "checksum")

        def __init__(self):
            self.done = 0
            self.total = 0
            self.checksum = None

    class Share(State):
        """ Postprocessing stage. Provides information about data sharing. """
        __slots__ = ("task_state", "skynet_id", "md5sum", "meta")

        def __init__(self):
            self.task_state = None
            self.skynet_id = None
            self.md5sum = None
            self.meta = None


class BatchResultStatus(utils.Enum):
    SUCCESS = None
    WARNING = None
    ERROR = None


#: Special value to return in case of key not found in structure.
NotExists = type("NotExists", (object,), {"__nonzero__": lambda _: False})()

#: RAM drive requirement specification structure - type (always `TMPFS`), size in megabytes,
#: path (should be set to `None` - it will be provided at task's excution time).
RamDrive = collections.namedtuple("RamDrive", ("type", "size", "path"))


class Component(utils.Enum):
    """ Sandbox component names enumeration. """
    utils.Enum.lower_case()

    TESTS = None        #: Tests execution process
    AGENTR = None       #: AgentR process
    CLIENT = None       #: Client process
    EXECUTOR = None     #: Task executor process
    PROXY = None        #: Proxy service/daemon
    SERVICE = None      #: Service process
    SERVER = None       #: Legacy REST API/XMLRPC daemon
    SERVICEAPI = None   #: Modern REST API ServiceAPI "Mapple" daemon


class HTTPHeader(utils.Enum):
    """ Service-specific HTTP header names. """
    # Request (user agent-provided) headers
    RETRY_AFTER = "Retry-After"             #: Standard header to wait before retry the request
    ACCEPT_RANGES = "Accept-Ranges"         #: Standard header for partial content transfer. Used for resumed uploads
    CONTENT_RANGE = "Content-Range"         #: Data content range for partial content transfer. Used for resumed upload
    RESUME_AT = "X-Resume-At"               #: Offset to resume upload stream at. Used internally for resumed uploads
    USER_AGENT = "User-Agent"               #: Typically node ID send with this field
    REQUEST_ID = "X-Request-Id"             #: API request ID to track it over server logs
    READ_PREFERENCE = "X-Read-Preference"   #: Database read preference to be forced for the request
    NO_LINKS = "X-No-Links"                 #: Request to return no resource's data sources links
    TOUCH_RESOURCE = "X-Touch-Resource"     #: Request to "touch" resource together with GET request
    IGNORE_STATE = "X-Ignore-State"         #: Request to ignore broken resource state on getting its data via proxy
    COMPONENT = "X-Sandbox-Component"       #: Sandbox component name. Used for internal API requests only
    FORWARDED_SCHEME = "X-Scheme"           #: Forwarded request scheme (HTTP/HTTPS). Used by service balancer
    FORWARDED_PATH = "X-Forwarded-Path"     #: Forwarded resource path. Used for proxy-to-fileserver communication
    FORWARDED_USER = "X-Forwarded-User"     #: Forwarded user name. Used for superuser-authenticated requests
    #: Asks for response with object data on update request instead of HTTP 204 "No Content"
    WANT_UPDATED_DATA = "X-Request-Updated-Data"
    REAL_IP = "X-Real-Ip"                   #: Real IP of the request initiator (request passed through proxy)
    FORWARDED_FOR = "X-Forwared-For"        #: Chain of IPs the request was processed by, started with initiator's IP

    # Response (server-provided) headers
    LOCATION = "Location"                   #: Standard redirect location header
    TASKS_REVISION = "X-Tasks-Revision"     #: Server-side loaded tasks archive revision
    TASK_ID = "X-Sandbox-Task-Id"           #: Sandbox task ID
    RESOURCE_ID = "X-Sandbox-Resource-Id"   #: Sandbox resource ID
    DATA_SOURCE = "X-Data-Source"           #: Sandbox resource data provider host ID
    INT_REQUEST_ID = "X-Backend-Request-Id"  #: Internally generated request ID. Used on proxy to track requests
    BACKEND_NODE = "X-Backend-Server"       #: Sandbox backend node ID which processed the request
    SERVICE_MODE = "X-Service-Mode"         #: Current Sandbox service mode (see `OperationMode`)
    REQ_DURATION = "X-Request-Duration"     #: Request total duration
    REQ_METRICS = "X-Request-Metrics"       #: Request execution metrics
    DB_LOCKED_BY = "X-Database-Locked-By"   #: Backend node ID which performing database upgrade at the moment
    CURRENT_USER = "X-Current-User"         #: User name which is performing current API request (response)
    RESET_SESSION = "X-Need-Reset-Session"  #: Request to user agent (browser) to reset Yandex session
