from __future__ import print_function
from __future__ import absolute_import

import os
import re
import gc
import abc
import cgi
import time
import gzip
import math
import string
import random
import socket
import urllib
import inspect
import numbers
import hashlib
import httplib
import logging
import statvfs
import urlparse
import platform
import threading
import traceback
import contextlib
import subprocess
import collections
import Queue as queue
import datetime as dt
import functools as ft
import itertools as it

import requests

import xml.etree.ElementTree

logger = logging.getLogger(__name__)

DiskSpace = collections.namedtuple("DiskSpace", ("total", "free"))

_TOKEN_LEN = 32


# noinspection PyPep8Naming
class classproperty(object):
    """ Class-level property. I.e., it's like `@classmethod` and `@property` decorators together. """
    def __init__(self, getter):
        self.getter = getter

    def __get__(self, obj, objtype):
        return self.getter(objtype)


# noinspection PyPep8Naming
class singleton_classproperty(classproperty):
    __none = type("None", (object,), {})()
    __lock = threading.RLock()
    __value = __none

    def __get__(self, obj, objtype):
        if self.__value is not self.__none:
            return self.__value
        with self.__lock:
            value = self.__value
            if value is self.__none:
                value = self.__value = self.getter(objtype)
            return value


def is_classproperty(cls, attrname):
    """
    Test if class attribute named `attrname` is a property (i.e. a descriptor with `__get__` method)
    """

    for class_ in cls.__mro__:
        if attrname in class_.__dict__:
            descriptor = class_.__dict__[attrname]
            if hasattr(descriptor, "__get__"):
                return descriptor
            return None
    return None


class Enum(object):
    """
    Enumerator base class. All class members, which are declared in UPPERCASE will be
    treated as enumeration element.
    """
    class Item(object):
        # noinspection PyPep8Naming
        class __metaclass__(type):
            def __call__(cls, doc=None):
                return type(cls)(cls.__name__, (cls,), {"__doc__": doc})

            def __eq__(cls, other):
                return str(cls) == str(other)

            def __ne__(cls, other):
                return str(cls) != str(other)

            def __setattr__(cls, *_):
                raise AttributeError("Object is immutable")

            def __delattr__(cls, *_):
                raise AttributeError("Object is immutable")

            def __hash__(cls):
                return hash(cls.__name__)

            def __repr__(cls):
                return str(cls.__name__)

        def __init__(self, _=None):
            pass

    # noinspection PyPep8Naming
    class __metaclass__(type):
        def __new__(mcs, name, bases, namespace):
            cls = type.__new__(mcs, name, bases, namespace)
            if bases == (object,):
                return cls
            lower_case = namespace.get("__lower_case__")
            no_underscores = namespace.get("__no_underscores__")

            def translate(elem):
                if lower_case:
                    elem = elem.lower()
                if no_underscores:
                    elem = filter(lambda ch: ch not in "_", elem)
                return elem

            new_namespace = {}
            for k, value in namespace.iteritems():
                if value is None:
                    value = translate(k)
                elif isinstance(value, type) and issubclass(value, getattr(cls, "Item", ())):
                    value = type(k, (value,), {"__module__": None, "__doc__": value.__doc__})
                new_namespace[k] = value
            return type.__new__(mcs, name, bases, new_namespace)

        def __iter__(cls):
            """
            Returns all known constants.
            """
            for attr, val in cls.iteritems():
                yield val

        def __getitem__(cls, item):
            try:
                return getattr(cls, item)
            except AttributeError:
                raise KeyError("Enumerator '{}.{}' does not contain key {!r}".format(
                    cls.__module__, cls.__name__, item
                ))

        def iteritems(cls):
            """
            Returns all known attribute names and their constants.
            """
            for attr, val in (
                ((_, cls.__dict__.get(_)) for _ in cls.__names__)
                if hasattr(cls, "__names__") else
                cls.__dict__.iteritems()
            ):
                if attr.isupper():
                    yield attr, val

        def val2str(cls, item):
            for attr, val in cls.iteritems():
                if val == item:
                    return attr
            raise ValueError("Enumerator '{}.{}' does not contain value {!r}".format(
                cls.__module__, cls.__name__, item
            ))

    @staticmethod
    def preserve_order():
        """ Enum modifier, preserve order of enum items """
        inspect.currentframe().f_back.f_locals["__names__"] = filter(
            lambda _: _.isupper(),
            inspect.currentframe().f_back.f_code.co_names
        )

    @staticmethod
    def lower_case():
        """ Enum modifier, make default values as lowercased items names """
        inspect.currentframe().f_back.f_locals["__lower_case__"] = True

    @staticmethod
    def no_underscores():
        """
        Enum modifier: translate autogenerated names to these without underscores,
        for example, API_CALL -> "APICALL"
        """
        inspect.currentframe().f_back.f_locals["__no_underscores__"] = True


class GroupEnum(object):
    """
    Enum for groups

    Automagically fills out the primary groups if items defined in scope of group definition:

    .. code-block:: python

        class Item(Enum):
            class Group(GroupEnum):
                # primary groups
                GROUP1 = None
                GROUP2 = None

                # secondary groups
                GROUP3 = None

            with Group.GROUP1:
                ITEM1 = None
                ITEM2 = None

            with Group.GROUP2:
                ITEM3 = None
                ITEM4 = None

            ITEM5 = None

        Item.Group.GROUP3 = (Item.ITEM1, Item.ITEM3, Item.ITEM4)
    """
    class Item(frozenset):
        # noinspection PyPep8Naming
        class __metaclass__(type):
            def __call__(cls, doc=None, items=()):
                if not doc:
                    instance = super(cls, cls).__new__(cls, items)
                    instance.__init__(items)
                    return instance
                ret = type(cls)(cls.__name__, (cls,), {"__doc__": doc})
                return ret

        primary = False

        def __hash__(self):
            return hash(str(self))

        def __eq__(self, other):
            return str(self) == str(other)

        def __ne__(self, other):
            return str(self) != str(other)

        def __setattr__(self, *_):
            raise AttributeError("Object is immutable")

        def __delattr__(self, *_):
            raise AttributeError("Object is immutable")

        def __add__(self, other):
            return frozenset.__or__(self, other)

        def expand(self):
            return GroupEnum.expand(self)

    # noinspection PyPep8Naming
    class __metaclass__(Enum.__metaclass__):
        def __init__(cls, name, bases, namespace):
            if bases == (object,):
                return
            Enum.__metaclass__.__init__(cls, name, bases, namespace)
            for group in filter(lambda _: _.isupper(), namespace):
                # noinspection PyUnresolvedReferences
                item_cls = namespace[group] or cls.Item
                # noinspection PyCallByClass
                type.__setattr__(
                    cls,
                    group,
                    type(group, (item_cls,), {
                        "__doc__": item_cls.__doc__,
                        "__module__": None,
                        "__repr__": lambda _, g=group: g,
                        "__setattr__": frozenset.__setattr__,
                        "__enter__": lambda _: setattr(
                            _,
                            "items",
                            set(inspect.currentframe().f_back.f_locals)
                        ),
                        "__exit__": lambda _, *__: type.__setattr__(
                            cls,
                            type(_).__name__,
                            type(type(_).__name__, (item_cls,), {
                                "__doc__": type(_).__doc__,
                                "__module__": None,
                                "__repr__": lambda _, g=type(_).__name__: g,
                                "primary": True
                            })(items=filter(
                                lambda _: _.isupper(),
                                set(inspect.currentframe().f_back.f_locals) - _.items
                            ))
                        )
                    })()
                )

        def __setattr__(cls, group, items):
            if len(getattr(cls, group)):
                raise AttributeError("Group cannot be modified")
            # noinspection PyUnresolvedReferences
            type.__setattr__(
                cls,
                group,
                type(group, (cls.Item,), {
                    "__doc__": getattr(cls, group).__doc__,
                    "__module__": None,
                    "__repr__": lambda _, g=group: g
                })(items=items)
            )

        def __contains__(cls, group):
            if isinstance(group, basestring):
                group = getattr(cls, group, None)
            return group is not None and group in iter(cls)

    @classmethod
    def expand(cls, *items):
        """
        Expand list of groups and items to list of items

        :param items: list of groups and/or items or single group or item
        :return: list of items
        """
        # noinspection PyTypeChecker
        groups = {str(_): _ for _ in iter(cls)}
        return set(it.chain.from_iterable(
            list(groups.get(str(_), [_])) for _ in chain(*items)
        ))

    @classmethod
    def collapse(cls, *items):
        """
        Collapse items from list into appropriate groups

        :param items: list of groups and/or items
        :return: list of items and groups
        """
        items = set(chain(*items))
        # noinspection PyTypeChecker
        for group in iter(cls):
            if not group.primary:
                continue
            group_items = set(group)
            if group_items <= items:
                items -= group_items
                items.add(str(group))
        return items


class LazyLoader(object):
    """
    Module lazy loader. Replaces `self` on first `__getattr__()` method call with the imported module.
    It is a good tool to resolve cyclic dependencies - just load a module with this loader at some
    of loop's participant.
    """

    def __init__(self, module, symbol=None):
        """
        Default constructor.
        :param module:  A string with module's name to be lazy loaded.
        :param symbol:  Optional symbol name to be looked inside the module specified.
        """
        self.module, self.symbol = module, symbol

    def __getattr__(self, item):
        module = __import__('importlib').import_module(self.module)
        self = getattr(module, self.symbol) if self.symbol else module
        return getattr(self, item)

    def __call__(self, *args, **kwargs):
        module = __import__('importlib').import_module(self.module)
        self = getattr(module, self.symbol) if self.symbol else module
        return self(*args, **kwargs)


class SingletonMeta(abc.ABCMeta):
    """ Meta class for implementing of pattern 'Singleton' """

    def __call__(cls, *args, **kws):
        instance = cls.instance
        if instance is None:
            instance = cls.instance = super(cls, cls).__new__(cls, *args, **kws)
            instance.__init__(*args, **kws)
        return instance

    @property
    def tag(cls):
        return "_{}_{}__instance".format(cls.__module__, cls.__name__)

    @property
    def instance(cls):
        return getattr(cls, cls.tag, None)

    @instance.setter
    def instance(cls, instance):
        if cls.instance is not None:
            raise AttributeError("Cannot set instance because it already exists")
        setattr(cls, cls.tag, instance)


class ThreadSafeSingletonMeta(SingletonMeta):
    """ Meta class for implementing of pattern 'Singleton' """

    lock = threading.Lock()

    def __call__(cls, *args, **kws):
        instance = cls.instance
        if instance is None:
            with cls.lock:
                instance = cls.instance
                if instance is None:
                    instance = super(cls, cls).__new__(cls, *args, **kws)
                    instance.__init__(*args, **kws)
                    cls.instance = instance
        return instance


def not_each_time(period):
    """
    Decorator. Performs call to a target function less frequently than it called -
    the function will be called only once per period specified.
    :param period:  Period in seconds, the function should NOT be called after last call.
    :return:        Decorator function.
    """
    def _decorator(func):
        @ft.wraps(func)
        def _wrapper(*args, **kwargs):
            now = time.time()
            if not kwargs.pop('force', False) and _wrapper.last_called + period >= now:
                return
            _wrapper.last_called = now
            return func(*args, **kwargs)

        _wrapper.last_called = 0
        return _wrapper

    return _decorator


def ttl_cache(ttl, external_cache=None):
    """
    Decorate your function with ``ttl_cache`` to have its return value cached for ``ttl`` seconds
    for every set of parameters separately. In case you need cache for an infinite amount of time,
    use ``singleton`` decorator.
    All the cache records are stored in the function's ``cache`` attribute

    :param ttl: amount of time to cache call result for, seconds
    :param external_cache: dict used as cache
    """

    def wrapper(f):
        if external_cache is not None:
            cache = external_cache
        elif getattr(f, "cache", None) is None:
            cache = f.cache = {}
        else:
            cache = f.cache

        def func(*args, **kw):
            kwargs = tuple(sorted(kw.items())) if kw else ()
            key = (args, kwargs)
            v, t = cache.get(key, (None, None))
            if t and t > time.time():
                return v

            v = f(*args, **kw)
            t = time.time() + ttl
            cache[key] = (v, t)
            return v
        return func

    return wrapper


def singleton(f):
    """
    Same as ``ttl_cache``, but store the result forever.
    Useful if you need to call something exactly once for a given set of arguments.
    """

    return ttl_cache(float("inf"))(f)


def md5sum(path, md5all=True):
    """
    Calculate MD5 sum of a file, which is read chunk by chunk

    :return: md5, if `path` is a file, or an empty string
    """

    m = hashlib.md5()
    path = os.path.abspath(path)

    if os.path.isfile(path):
        with open(path, 'rb') as f:
            while True:
                data = f.read(8096)
                if not data:
                    break
                m.update(data)
        return m.hexdigest()
    else:
        return ""


def gzip_file(in_path, out_path, remove_source=False):
    with open(in_path, 'rb') as orig_file:
        with gzip.open(out_path, 'wb') as zipped_file:
            zipped_file.writelines(orig_file)
    if remove_source:
        os.remove(in_path)


def server_url(settings=None):
    if not settings:
        from . import config
        settings = config.Registry()
    addr = settings.server.web.address
    return (
        "http://{host}:{port}".format(host=addr.host, port=settings.server.api.port)
        if addr.show_port else
        "https://{}".format(addr.host)
    )


def get_task_link(task_id, settings=None):
    return urlparse.urljoin(server_url(settings), "task/{}".format(task_id))


def get_resource_link(resource_id, settings=None):
    return urlparse.urljoin(server_url(settings), "resource/{}".format(resource_id))


def get_scheduler_link(scheduler_id, settings=None):
    return urlparse.urljoin(server_url(settings), "scheduler/{}".format(scheduler_id))


def execution_dir_link(task_id, settings=None):
    """
    Obtain URL to a task's execution directory for a quick view (works with all Sandbox installation types)

    :param task_id: target task identifier
    :param settings: Sandbox settings object
    :type settings: sandbox.common.config.Registry
    :return: task execution directory URL
    """

    from .types import task as ctt
    if not settings:
        from . import config
        settings = config.Registry()

    fs_settings = settings.client.fileserver
    if fs_settings.proxy.host:
        return "https://{}/task/{}".format(fs_settings.proxy.host, task_id)

    # running on local Sandbox; there's no proxy but a fileserver
    return "http://{}:{}/{}".format(
        settings.server.web.address.host, fs_settings.port,
        "/".join(ctt.relpath(task_id))
    )


def _get_command_stdout_first_line(cmd):
    """
    Run a command, wait until it's finished and get the first line of its output

    :param cmd: a list of arguments
    :return: command's first line from STDOUT
    """

    p = subprocess.Popen(cmd, stdout=subprocess.PIPE)
    p.wait()
    return p.stdout.readlines()[0].strip()


def _get_cpu_model():
    """
    Obtain the host's CPU model

    :return: CPU model, or "unknown" if there's no way to figure it
    """

    cpu_model = 'unknown'
    arch = os.uname()[0].lower()
    if arch == 'freebsd':
        cpu_model = _get_command_stdout_first_line(['/sbin/sysctl', 'hw.model'])
        cpu_model = cpu_model.replace('hw.model:', '').strip()
    elif arch == 'linux' or arch.startswith('cygwin'):
        with open('/proc/cpuinfo') as f:
            model_name_line = next((line for line in f.readlines() if line.startswith('model name')), None)
            if model_name_line:
                splited = model_name_line.split(':', 1)
                if len(splited) > 1:
                    cpu_model = splited[1]
                cpu_model = cpu_model.strip()

    elif arch == 'darwin':
        cpu_model = _get_command_stdout_first_line(['/usr/sbin/sysctl', '-n', 'machdep.cpu.brand_string'])
        m = re.search(r'^.*Intel\(R\).+? (?:[^ -]+-)?([^ ]+) CPU.*$', cpu_model)
        if m:
            cpu_model = m.group(1)
    else:
        return cpu_model

    if cpu_model == '5148':
        cpu_model = 'E5148'

    if 'QEMU' in cpu_model:
        g = re.search('CPU version *([^ ]*)', cpu_model)
        if g:
            cpu_model = "QEMU-" + g.group(1)

    if 'AMD' in cpu_model:
        g = re.search('Processor *([^ ]*)', cpu_model)
        if g:
            cpu_model = "AMD" + g.group(1)

    if 'Intel' in cpu_model:
        g = re.search('(?:CPU|Xeon)\s+([^ ]*)', cpu_model)
        if g:
            cpu_model = g.group(1)
    return cpu_model


def _get_physmem():
    """ Host RAM amount in bytes """
    if platform.system().startswith("CYGWIN"):
        with open("/proc/meminfo", "r") as fh:
            for line in fh:
                if line.startswith("MemTotal:"):
                    return int(line.split()[1]) << 10
            return None
    else:
        import pkg_resources
        pkg_resources.require('psutil')
        import psutil
        return psutil.phymem_usage()[0]


def get_sysparams():
    """
        Retrieve information about a host

        :return: a dict with the following fields:

            :platform: host platform, a result of `platform.platform()` call
            :fqdn: the host's fully qualified domain name (for example, `sandbox-server01.search.yandex.net`)
            :arch: OS family (linux, freebsd, etc)
            :os: a dict with the following fields:

                :version: OS version
                :name: OS name

            :cpu: CPU model or `unknown`
            :ncpu: CPU cores number
            :physmem: a string containing psysical memory amount, bytes. It's usually
                a multiple of `1 << 20` but is always slightly less than the nominal amount.
                For example, if the host has a memory of size of 64Gb, `physmem` would contain
                `str(64399 * 1024 * 1024)` (expected would be `str(65536 * 1024 * 1024)`
            :fileserver: fileserver URL
            :tasks_dir: local path to Sandbox tasks code directory
    """

    from . import config
    settings = config.Registry()

    uname_info = os.uname()
    sys_params = {
        "platform": platform.platform(),
        "fqdn": settings.this.fqdn,
        "arch": settings.this.system.family,
        "os": {"version": uname_info[2], "name": settings.this.system.family},
        "cpu": _get_cpu_model(),
        "ncpu": settings.this.cpu.cores,
        "physmem": _get_physmem(),
        "fileserver": "http://{node}:{port}/".format(
            node=settings.this.fqdn,
            port=settings.client.fileserver.port,
        ),
        "tasks_dir": settings.client.tasks.data_dir,
    }
    return sys_params


def cleanup(path, ignore_path=set()):
    """
    Recursively clean task's executing directory, with exception of certain paths.

    :param path: directory to cleanup in
    :type path: str
    :param ignore_path: an iterable with absolute paths to ignore
    :type ignore_path: iterable
    """

    def _cleanup(path, ignore_path):
        """
        XXX: `shutil.rmtree()` doesn't check for symbolic links:
            http://svn.python.org/projects/python/trunk/Lib/shutil.py, and
            `os.walk()` is inconvenient to use due to file name generation order
        """

        if path in ignore_path:
            return 0
        if os.path.islink(path):
            os.unlink(path)
            return 1
        if not os.path.exists(path):
            return 0
        try:
            if not os.path.isdir(path):
                os.unlink(path)
                return 1
            else:
                # recursive call
                n = 0
                for name in os.listdir(path):
                    n += _cleanup(os.path.join(path, name), ignore_path)
                # remove empty directory after it's cleared
                if not os.listdir(path):
                    os.rmdir(path)
                return n
        except OSError as e:
            logger.exception(e)
        return 0
    n = _cleanup(path, ignore_path)
    return n


def get_disk_space(mount_point):
    st = os.statvfs(mount_point)
    return DiskSpace(
        st[statvfs.F_BLOCKS] * st[statvfs.F_FRSIZE],
        st[statvfs.F_BAVAIL] * st[statvfs.F_FRSIZE]
    )


def is_port_free(port, host='localhost', protocol='tcp'):
    """
    Check port availability

    :param port: port number (string or int)
    :param host: hostname
    :param protocol: 'tcp' or 'udp'
    """

    logger.info('Check port %s for host %s for %s protocol', port, host, protocol)
    try:
        if protocol == 'tcp':
            sock_type = socket.SOCK_STREAM
        elif protocol == 'udp':
            sock_type = socket.SOCK_DGRAM
        else:
            raise ValueError("Not supported protocol {}".format(protocol))
        sock = socket.socket(socket.AF_INET, sock_type)
        sock.connect((host, int(port)))
        sock.close()
        return False
    except socket.error:
        pass
    return True


def progressive_yielder(tick, max_tick, max_wait, sleep_first=True, sleep_func=None):
    """
    Yields passed amount of time progressively.
    :param tick:        Initial tick amount (in seconds, fractional).
    :param max_tick:    Maximum available tick amount in seconds.
    :param max_wait:    Maximum totally allowed wait amount in seconds.
    :param sleep_first: Sleep before first iteration.
    :param sleep_func:  sleep function, by default is time.sleep
    :return:            Total amount of wait in seconds.
    """
    sleep_func = sleep_func or time.sleep
    slept, started = tick if sleep_first else None, time.time()
    while slept < max_wait:
        if slept is not None:
            sleep_func(min(tick, max_wait - slept))
        yield slept, tick
        slept = time.time() - started
        tick = min(max_tick, (tick * 55 / 34) if tick else .001)


def progressive_waiter(tick, max_tick, max_wait, checker, sleep_first=True, inverse=False, sleep_func=None):
    """
    Waits for given event progressively.
    :param tick:        Initial tick amount (in seconds, fractional).
    :param max_tick:    Maximum available tick amount in seconds.
    :param max_wait:    Maximum totally allowed wait amount in seconds.
    :param checker:     Callable which should return `True` in case of event has been occurred.
    :param sleep_first: Sleep before first iteration.
    :param inverse:     Inverse condition.
    :param sleep_func:  sleep function, by default is time.sleep
    :return:            Tuple of last checker value and total amount of wait in seconds.
    """
    ret, slept = None, 0
    for slept, _ in progressive_yielder(tick, max_tick, max_wait, sleep_first=sleep_first, sleep_func=sleep_func):
        ret = checker()
        if bool(ret) ^ inverse:
            break
    return ret, slept


def checker_fetcher(url, md5=None, chunk_size=0x7FFF, **kwargs):
    """
    Fetch a data located at the given URL and check its content size and MD5 checksum if they're provided.
    Data is yielded by chunks.

    :param url: URL to fetch data from
    :param md5: MD5 checksum of the data
    :param chunk_size: size of a single chunk of data
    :param kwargs: addditional keyword arguments for `requests.get` method
    """

    r = requests.get(url, stream=True, **kwargs)
    r.raise_for_status()
    try:
        size = int(r.headers["content-length"])
    except (TypeError, ValueError, KeyError):
        size = None
    actual_size = 0
    actual_md5 = hashlib.md5()
    for chunk in r.iter_content(chunk_size):
        actual_size += len(chunk)
        actual_md5.update(chunk)
        yield chunk

    if size and actual_size != size:
        raise ValueError("Data size {} mismatch: expected {}".format(actual_size, size))
    if md5 and md5 != actual_md5.hexdigest():
        raise ValueError("MD5 checksum {!r} mismatch: expected {!r}".format(actual_md5.hexdigest(), md5))


def thname():
    """
    Shortcut for current thread name.
    """
    return threading.current_thread().name


def print_table(data, printer=print):
    """
    Prints the given data as pretty-printed table.
    :param data:    Array of table rows of strings. `None` row means horizontal line.
    :param printer: Print function to be used to output the actual content.
    """
    cols = len(data[0])
    _ESCAPE_SEQ_CUT_RE = re.compile(r"\x1b\[.*?m")
    for row in data:
        for i in xrange(cols) if row else []:
            if not isinstance(row[i], basestring) and row[i] is not None:
                row[i] = str(row[i])
    lens = [max(map(
        lambda x: len(_ESCAPE_SEQ_CUT_RE.sub("", x)) if x else 0,
        (row[i] for row in data if row)
    )) for i in xrange(cols)]
    hr = "+%s+" % "+".join("-" * (l + 2) for l in lens)

    for i, row in enumerate([None] + list(data) + [None]):
        if not row:
            printer(hr)
            continue
        printer("".join(chain(
            (
                " ".join([
                    " " if i and cell is None else "|",
                    cell or "",
                    " " * (width - len(_ESCAPE_SEQ_CUT_RE.sub("", cell)))
                ]) for i, (cell, width) in enumerate(zip(map(str, row), lens))
            ),
            "|"
        )))


def check_cqueue_results(results):
    errors = []
    for host, result, failure in results:
        if failure:
            if hasattr(failure, '_traceback'):
                errors.append('Error at %s: %s' % (host, failure._traceback))
            else:
                errors.append('Error at %s: %s' % (host, failure))

    if errors:
        raise Exception('\n'.join(errors))


def read_settings_value_from_file(value, ignore_file_existence=False, binary=False):
    """
    Return settings value via file if startswith file://
    otherwise return value itself

    :param value: value of path to file that contains value
    :param ignore_file_existence: if True then return None if file does not exist
    :param binary: open file for reading in binary mode

    :return: settings value or None
    :rtype: str or None
    """

    if isinstance(value, basestring) and value.startswith("file://"):
        path = value.split("file://")[1]
        path = os.path.expanduser(path)
        if ignore_file_existence and not os.path.exists(path):
            return
        with open(path, "rb" if binary else "r") as f:
            value = f.read().strip()
    return value


class FLock(object):
    def __init__(self, fname):
        self.__fname = fname
        self.__fd = None
        self.__fcntl = __import__("fcntl")

    def __enter__(self):
        self.__fd = os.open(self.__fname, os.O_RDONLY | os.O_CREAT, 0666)
        self.__fcntl.flock(self.__fd, self.__fcntl.LOCK_EX)

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.__fcntl.flock(self.__fd, self.__fcntl.LOCK_UN)
        try:
            os.close(self.__fd)
        except Exception:
            pass


def lock_on_port(port):
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.bind(('127.0.0.1', port))
    return sock


def percentile(iterable, percent):
    """
    Get `percent`-percentile of given iterable of ints.

    :param iterable: an iterable to calculate percentile. MUST BE SORTED
    :param percent: percentile value, a float in range [0.0, 1.0]
    :return: percentile (int) of given iterable if it's not empty, None otherwise
    :rtype: int or None
    """
    if not iterable:
        return
    k = (len(iterable) - 1) * percent
    floor = math.floor(k)
    ceil = math.ceil(k)
    if floor == ceil:
        return iterable[int(k)]
    item_0 = iterable[int(floor)] * (ceil - k)
    item_1 = iterable[int(ceil)] * (k - floor)
    return int(item_0 + item_1)


def chain(*args):
    """ "Safe" method to convert object(s) of almost any type to a generator. """
    for obj in args:
        if hasattr(obj, "__iter__"):
            for _ in obj:
                yield _
        else:
            yield obj


def as_list(*args):
    """
    Safe method to convert object(s) of almost any type to a list.
    Removes None elements.
    """
    return list(filter(lambda x: x is not None, chain(*args)))


class HostInfo(object):
    """Class provides interface to yandex-team services to get info about host"""

    BOT_API_URL = "http://bot.yandex-team.ru/api"
    CONDUCTOR_API_URL = "http://c.yandex-team.ru/api"
    RETRY_ATTEMPTS = 3

    class RetryableError(Exception):
        """Raise this exception when conductor response code is not OK to retry request"""

    @classmethod
    def call_api(cls, url):
        """
        Make request to API

        :param url: url to call API method
        :return: content of API response
        """
        resp = urllib.urlopen(url)
        if resp.code != httplib.OK:
            raise cls.RetryableError("API responses with code: {}".format(resp.code))
        return resp.read()

    @classmethod
    def _exceptions_handler(cls, func, default_value, *args, **kwargs):

        @ft.wraps(func)
        def wrapper():
            attempt = 1
            for x in xrange(cls.RETRY_ATTEMPTS):
                try:
                    return func(*args, **kwargs)
                except cls.RetryableError:
                    attempt += 1
                    if attempt == cls.RETRY_ATTEMPTS + 1:
                        return default_value
                    time.sleep(0.1)
                except IOError:
                    return default_value

        return wrapper

    @classmethod
    def _dc(cls, hostname):
        url = "{base}/hosts/{host}".format(base=cls.CONDUCTOR_API_URL, host=hostname)
        content = cls.call_api(url)
        tree = xml.etree.ElementTree.fromstring(content)
        return tree.find("item").find("datacenter").text

    @classmethod
    def dc(cls, hostname):
        """
        Get host data center by specified hostname

        :param hostname: string with hostname
        :return: host's data center
        :rtype: str
        """
        return cls._exceptions_handler(cls._dc, "unk", hostname)() or "unk"

    @classmethod
    def _has_ssd(cls, hostname):
        url = "{base}/consistof.php?name={host}".format(base=cls.BOT_API_URL, host=hostname)
        content = cls.call_api(url)
        # assume, that if golem response contains `/ssd/`, then ssd is on board
        ssd_in_response = '/ssd/' in content.lower()
        return ssd_in_response

    @classmethod
    def has_ssd(cls, hostname):
        """
        Check that solid state drive (ssd) is available on specified hostname

        :param hostname: string with hostname
        :return: flag that ssd is available
        :rtype: bool
        """
        return cls._exceptions_handler(cls._has_ssd, False, hostname)()

    @classmethod
    def neighbours(cls, this, hosts):
        import library.sky.hosts
        hosts = library.sky.hosts.braceExpansion(hosts)
        fqdns = {_: _.split(":")[0] for _ in hosts}
        dc = {_: cls.dc(_)[:3] for _ in fqdns.itervalues()}
        hosts.sort(key=lambda _: 0 if fqdns[_] == this.fqdn else (1 if dc[fqdns[_]] == this.dc else 2))
        return hosts


class Timer(collections.MutableMapping):
    """
    A simple context manager which measures time amount spent in its context.
    It also can measure sub-stages, which are available via operator `[]`.
    Also, it can automatically reduce the counter on block leave if it will be passed into the constructor.
    """
    __slots__ = ("_started", "_stopped", "_subruns")

    def __init__(self, countdown=0):
        self._started = None
        self._stopped = None
        self._countdown = countdown
        self._subruns = collections.OrderedDict()

    def start(self):
        if self._started is None:
            self._started = time.time()
        self._stopped = None
        return self

    def stop(self):
        self._stopped = time.time()
        return self

    def __enter__(self):
        return self.start()

    def __exit__(self, *_):
        self.stop()

    def __str__(self):
        ret = "{:.4g}s".format(self.secs) if self._started else super(Timer, self).__str__()
        if self._subruns:
            ret += " ({})".format("/".join("{}:{}".format(k, str(v)) for k, v in self._subruns.iteritems()))
        return ret

    @property
    def secs(self):
        return (self._stopped or time.time()) - self._started if self._started else None

    @property
    def left(self):
        if not self._countdown:
            return None
        self._countdown -= self.secs or 0
        return self._countdown

    def reset(self):
        self._started = None
        self._stopped = None

    def __float__(self):
        return self.secs

    def __getitem__(self, item):
        if not self._started:
            raise KeyError("Cannot start subtimer on not started timer object.")
        if item in self._subruns:
            return self._subruns[item]
        ret = self._subruns[item] = self.__class__()
        return ret

    def __setitem__(self, key, value):
        raise NotImplementedError("Timer does not allow to set items explicitly.")

    def __delitem__(self, item):
        del self._subruns[item]

    def __len__(self):
        return self._subruns.__len__()

    def __iter__(self):
        return self._subruns.__iter__()

    def __contains__(self, item):
        return item in self._subruns


def namedlist(name, fields):
    fields = tuple(fields.split() if isinstance(fields, basestring) else fields)
    namespace = {
        f: property(lambda s, i=i: s[i], lambda s, v, i=i: s.__setitem__(i, v))
        for i, f in enumerate(fields)
    }
    namespace["__slots__"] = fields
    namespace["_fields"] = fields
    namespace["__repr__"] = lambda self: "{}({})".format(
        name, ", ".join(it.imap(lambda _: "=".join((_[0], repr(_[1]))), it.izip(fields, self)))
    )
    return type(
        name,
        (list,),
        namespace
    )


class Api(type):
    """
    Metaclass for registering of APIs, for example to use via Synchrophazotron
    """

    # noinspection PyPep8Naming
    class __metaclass__(type):
        __apis__ = {}

        def __iter__(cls):
            return cls.__apis__.iteritems()

        def __getitem__(cls, name):
            return cls.__apis__[name]

        @staticmethod
        def register(method):
            assert isinstance(method, (staticmethod, classmethod))
            method.__func__.__isapimethod__ = True
            return method

    class Proxy(object):
        def __init__(self, cls, factory):
            self.__cls = cls
            self.__factory = factory

        def __enter__(self):
            self.__orig_methods = dict(map(
                lambda (name, method): (
                    name,
                    (
                        setattr(
                            self.__cls,
                            name,
                            (classmethod if hasattr(method, "im_func") else staticmethod)(
                                ft.wraps(method)(self.__factory(self.__cls, name))
                            )
                        ) or
                        setattr(
                            getattr(getattr(self.__cls, name), "im_func", None) or getattr(self.__cls, name),
                            "__orig_method__",
                            method
                        ) or
                        method
                    )
                ),
                ((_, getattr(self.__cls, _)) for _ in self.__cls.api_methods)
            ))
            return self.__cls

        def __exit__(self, *_):
            map(lambda (k, v): setattr(self.__cls, k, v), self.__orig_methods.iteritems())

    def __new__(mcs, name, bases, namespace):
        cls = super(Api, mcs).__new__(mcs, name, bases, namespace)
        api_methods = set(
            name
            for name, value in namespace.iteritems()
            if (
                isinstance(value, (staticmethod, classmethod)) and
                getattr(value.__func__, "__isapimethod__", False)
            )
        )
        for base in bases:
            for method_name in getattr(base, "__apimethods__", set()):
                value = getattr(cls, method_name, None)
                if getattr(value, "__isapimethod__", False):
                    api_methods.add(method_name)
        cls.__apimethods__ = frozenset(api_methods)
        mcs.__metaclass__.__apis__[name] = cls
        return cls

    @property
    def api_methods(cls):
        return cls.__apimethods__


# noinspection PyPep8Naming
class singleton_property(property):
    __none = type("None", (object,), {})()
    __lock = threading.RLock()
    __value = __none

    def __get__(self, obj, objtype=None):
        if obj is None:
            return super(singleton_property, self).__get__(obj, objtype)
        return (self.__cls_get if isinstance(obj, type) else self.__obj_get)(obj, objtype)

    def __delete__(self, obj):
        if isinstance(obj, type):
            self.__value = self.__none
        else:
            try:
                del obj.__dict__[self.fget.__name__]
            except KeyError as ex:
                raise AttributeError(ex.message)

    def __obj_get(self, obj, objtype):
        name = self.fget.__name__
        ret = obj.__dict__.get(name, self.__none)
        if ret is not self.__none:
            return ret
        with self.__lock:
            ret = obj.__dict__.get(name, self.__none)
            if ret is not self.__none:
                return ret
            return obj.__dict__.setdefault(name, super(singleton_property, self).__get__(obj, objtype))

    def __cls_get(self, obj, objtype):
        if self.__value is not self.__none:
            return self.__value
        with self.__lock:
            value = self.__value
            if value is self.__none:
                value = self.__value = super(singleton_property, self).__get__(obj, objtype)
            return value


def str2size(value):
    """ Transforms given string into integer. Additionally it processes suffixes K, M, G, T as powers of 1024. """
    _MULTIPLIERS = "KMGT"
    if isinstance(value, numbers.Number):
        return int(value)
    mul = value[-1]
    if mul in _MULTIPLIERS:
        res = 1
        for _ in _MULTIPLIERS:
            res <<= 10
            if _ == mul:
                break
        return int(float(value[:-1]) * res)
    else:
        return int(value)


###############################################################################
# A number of functions copied from :module:`kernel.util.misc`
###############################################################################

def size2str(size, till=None):
    """ Size to string formatter. """
    mods = " KMGTPEZY"
    size = float(size)
    if till:
        till = till.upper()
        assert len(till) == 1 and till in mods
    for mod in mods:
        if ((abs(size) < 0x400 or mod == mods[-1]) and not till) or mod == till:
            return "%.2f%siB" % (size, mod) if mod != " " else "%.2fbyte(s)" % (size,)
        size /= 0x400


def dt2str(x):
    """ Datetime to string formatter """
    return "None" if x is None else x.strftime("%Y-%m-%d %H:%M:%S")


def utcdt2iso(v=None):
    return (v or dt.datetime.utcnow()).isoformat() + "Z"


def td2str(delta, full=False):
    """ Timedelta to string formatter. """
    if isinstance(delta, (int, float, long, )):
        delta = dt.timedelta(seconds=delta)
    hrs, secs = divmod(delta.seconds, 3600)
    mins, secs = divmod(secs, 60)
    ret = ''

    suffixes = list('dhms') if not full else map(lambda x: ' %s(s)' % x, 'day hour minute second'.split())
    for fmt, val in zip(suffixes, (delta.days, hrs, mins, secs)):
        if val or ((ret or fmt == 's') and not full):
            ret += '{0:s}{1:0>2d}{2:s}'.format(' ' if ret else '', val, fmt)
    return ret


class SimpleComputationGraph(object):
    """
    Class for computation of simple job graph.

    Each job executes in separate thread, so the number of jobs shouldn't be very big.
    """

    Job = collections.namedtuple('Job', ['id', 'name', 'method', 'args', 'kwargs', 'wait_jobs'])

    def __init__(self, log=None):
        self._back_queue = queue.Queue()
        self._workers = []
        self._logger = log or logger

    def add_job(self, method, args=(), kwargs=None, wait_jobs=()):
        """
        Add job to execute. Waited jobs have to be added already.

        :param method: callable to run in job
        :type method: function
        :param args: args for `method`
        :param kwargs: kwargs for `method`
        :param wait_jobs: list if job ids to wait
        :return: job id
        """
        kwargs = kwargs or {}
        job_id = len(self._workers)
        if not all(0 <= wait_job_id < job_id for wait_job_id in wait_jobs):
            raise ValueError("It's allowed to wait only jobs that are registered already")
        job_name = '<{}|{}>'.format(job_id, method.__name__)
        job = self.Job(id=job_id, name=job_name, method=method, args=args, kwargs=kwargs, wait_jobs=wait_jobs)
        self._workers.append(threading.Thread(target=self._thread_error_reporter, args=(job, self._back_queue)))
        return job_id

    def run(self):
        for worker in self._workers:
            worker.start()
        map(threading.Thread.join, self._workers)
        rs = [self._back_queue.get() for _ in self._workers]
        if any(rs):
            raise next(iter(filter(None, rs)))

    def _thread_error_reporter(self, job, back_queue):
        if job.wait_jobs:
            self._logger.debug('Job %s: wait jobs %r', job.name, job.wait_jobs)
            for job_id in job.wait_jobs:
                self._workers[job_id].join()
        self._logger.debug('Job %s: start', job.name)
        try:
            job.method(*job.args, **job.kwargs)
            back_queue.put(None)
            self._logger.debug("Job %s: finish", job.name)
        except Exception as ex:
            self._logger.exception("Job %s: error in thread", job.name)
            back_queue.put(ex)


def grouper(iterable, n):
    """ Collect data into fixed-length chunks or blocks
        grouper('ABCDEFG', 3) --> ABC DEF G
    """

    group = []
    for item in iterable:
        group.append(item)
        if len(group) >= n:
            yield group
            group = []
    if group:
        yield group


def grouper_longest(iterable, n, fillvalue=None):
    """ Collect data into fixed-length chunks or blocks
        grouper('ABCDEFG', 3, 'x') --> ABC DEF Gxx
    """

    args = [iter(iterable)] * n
    return it.izip_longest(*args, fillvalue=fillvalue)


def chunker(data, size):
    """ The simplest chunker - yields `size`-length chunks of the given input sequence. """
    i = 0
    while True:
        chunk = data[i:i + size]
        if not chunk:
            break
        yield chunk
        i += size


def format_exception():
    """ Format exception traceback as an unicode string """
    trace = traceback.format_exc()
    return trace if isinstance(trace, unicode) else str(trace).decode("utf8")


# noinspection PyPep8Naming
class ident(str):
    # noinspection PyPep8Naming
    class __metaclass__(type):
        @staticmethod
        def __char_seq(f, l):
            return "".join(it.imap(chr, xrange(ord(f), ord(l) + 1)))
        # noinspection PyUnresolvedReferences
        __char_seq = __char_seq.__func__

        DIGITS = __char_seq("0", "9")
        UPPERS = __char_seq("A", "Z")
        LOWERS = __char_seq("a", "z")
        IDENT_TABLE = {
            0: dict(
                (c, (c.upper(), 1)) for c in it.chain("_", UPPERS, LOWERS)
            ),
            1: dict(it.chain(
                (("_", ("_", 1)),),
                ((c, (c, 2)) for c in DIGITS),
                ((c, (c, 3)) for c in UPPERS),
                ((c, (c.upper(), 4)) for c in LOWERS)
            )),
            2: dict(it.chain(
                (("_", ("_", 1)),),
                ((c, (c, 2)) for c in DIGITS),
                ((c, ("_" + c, 3)) for c in UPPERS),
                ((c, (c.upper(), 4)) for c in LOWERS)
            )),
            3: dict(it.chain(
                (("_", ("_", 1)),),
                ((c, ("_" + c, 2)) for c in DIGITS),
                ((c, (c, 3)) for c in UPPERS),
                ((c, (c.upper(), 4)) for c in LOWERS)
            )),
            4: dict(it.chain(
                (("_", ("_", 1)),),
                ((c, (c.upper(), 4)) for c in LOWERS),
                ((c, ("_" + c.upper(), 2)) for c in DIGITS),
                ((c, ("_" + c.upper(), 3)) for c in UPPERS)
            ))
        }

        def __call__(cls, name):
            if name.isupper():
                return name
            state = 0
            output = ""
            try:
                for c in name:
                    c, state = cls.IDENT_TABLE[state][c]
                    output += c
                return output
            except KeyError:
                raise ValueError("Invalid identifier: {!r}".format(name))


def random_string(length=8, characters=string.ascii_uppercase + string.digits):
    return "".join(random.choice(characters) for _ in range(length))


def force_unicode(s, encoding="utf-8", errors="strict"):
    if isinstance(s, unicode):
        return s
    try:
        if hasattr(s, "__unicode__"):
            return unicode(s)
        if not isinstance(s, basestring):
            s = str(s)
        return unicode(s, encoding, errors)
    except UnicodeError:
        if not isinstance(s, Exception):
            raise
        # The caller has passed in an Exception subclass populated with non-ASCII data without special handling
        # to display as a string. We need to handle this without raising further exceptions.
        return " ".join([force_unicode(arg, encoding, errors) for arg in s])


def force_int(value, default=0):
    try:
        return int(value)
    except (TypeError, ValueError):
        return default


def escape(html):
    """ Returns the given HTML with ampersands, quotes and angle brackets encoded. """
    return cgi.escape(force_unicode(html))


def obfuscate_token(token):
    """ Leaves only first eight characters of the string if string is a token. """

    if not isinstance(token, basestring) or not len(token) == _TOKEN_LEN:
        return token
    return token[:8]


def call_with(context_manager, func, *args, **kws):
    with context_manager:
        return func(*args, **kws)


@contextlib.contextmanager
def disabled_gc():
    """ Context manager for temporary disabling garbage collection """
    saved_threshold = gc.get_threshold()
    gc.set_threshold(0)
    try:
        yield None
    finally:
        gc.set_threshold(*saved_threshold)
