import os
import sys
import yaml
import logging
import datetime as dt
import subprocess as sp

from .. import utils
from .. import config
from .. import patterns
from ..types import client as ctt

from .. import errors


class VCSCacheUnavailable(errors.TaskFailure):
    pass


class CacheRecord(patterns.Abstract):

    __slots__ = ("path", "task_id", "update_time", "is_permanent", "rep_type", "url")
    __defs__ = ("", None, None, False, None, None)

    def __eq__(self, other):
        for slot in self.__slots__:
            if getattr(self, slot) != getattr(other, slot):
                return False
        return True

    def dump(self):
        return yaml.dump(dict(self), width=sys.maxsize)  # big number to guarantee storing one record per one line

    @classmethod
    def load(cls, s):
        try:
            return CacheRecord(**yaml.load(s))
        except yaml.YAMLError:
            # Do not init new fields using old metadata format
            return CacheRecord(*s.split("\t")[:3])

    @classmethod
    def read_cache_records(cls, metadata_file_path):
        if not os.path.isfile(metadata_file_path):
            return []

        cache = {}
        records = []

        for line in open(metadata_file_path):
            rec = CacheRecord.load(line)

            exists = cache.get(rec.path, None)
            if exists is None:
                exists = os.path.exists(rec.path)
                cache[rec.path] = exists

            if exists:
                records.append(rec)

        return records

    @classmethod
    def write_cache_records(cls, metadata_file_path, records):
        # don't use tempfile, because temporary file can be created on a different filesystem
        tmp = metadata_file_path + "~"
        with open(tmp, "w") as f:
            for record in records:
                f.write(record.dump())
        os.rename(tmp, metadata_file_path)


class CacheableVCS(object):
    METADATA_FILENAME = ".metadata"
    base_cache_dir = None
    REP_TYPE = None
    DIRS_LIMIT = None

    @classmethod
    def raise_if_unavailable(cls):
        """ Check if cache usage is banned on a host (=MULTISLOT) """

        if ctt.Tag.MULTISLOT in config.Registry().client.tags:
            raise VCSCacheUnavailable(
                "VCS cache usage is banned on MULTISLOT hosts, as it may be altered by multiple tasks at once"
            )

    @utils.classproperty
    def cache_metadata_path(cls):
        return os.path.join(cls.base_cache_dir, cls.METADATA_FILENAME)

    @classmethod
    def add_cache_metadata(cls, cache_dir, task_id, is_permanent=False, url=""):
        """ Save act of local VCS copy usage

        :param cache_dir: path to local vcs copy
        :param task_id: id of task that uses cache
        :param is_permanent: True if local vcs copy shouldn't be removed, usually it is trunk or master branch
        :param url: url to repository
        """
        with utils.FLock(cls.cache_metadata_path):
            records = CacheRecord.read_cache_records(cls.cache_metadata_path)
            records.append(CacheRecord(
                path=cache_dir,
                task_id=str(task_id),
                update_time=str(dt.datetime.now()),
                is_permanent=is_permanent,
                rep_type=cls.REP_TYPE,
                url=url,
            ))
            CacheRecord.write_cache_records(cls.cache_metadata_path, records)

    @classmethod
    def get_cache_dirs(cls):
        # metadata can contain a lot of duplicates
        return set(
            rec.path for rec in CacheRecord.read_cache_records(cls.cache_metadata_path)
            if rec.rep_type == cls.REP_TYPE
        )


class CacheableArcadia(CacheableVCS):
    REP_TYPE = "svn"
    DIRS_LIMIT = 5

    @utils.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class CacheableArcadiaTestData(CacheableArcadia):

    @utils.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.tests_data


class CacheableGit(CacheableVCS):
    REP_TYPE = "git"
    DIRS_LIMIT = 20

    @utils.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class CacheableHg(CacheableVCS):
    REP_TYPE = "hg"
    DIRS_LIMIT = 1
    RESOURCE_TYPE = "ARCADIA_HG_REPOSITORY"

    @utils.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class CacheableArcadiaApi(CacheableVCS):
    REP_TYPE = "aapi"
    DIRS_LIMIT = 1

    @utils.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class VCSCache(object):
    _LOG_NAME = "vcs"
    _cacheables = (CacheableArcadia, CacheableArcadiaTestData, CacheableGit, CacheableHg, CacheableArcadiaApi)

    def __init__(self, logger=None):
        self.logger = logger or logging.getLogger(self._LOG_NAME)

    def clean(self):
        for cacheable in self._cacheables:
            try:
                self._clean(cacheable)
            except Exception:
                self.logger.exception("Unable to clean %r in %s", cacheable.REP_TYPE, cacheable.base_cache_dir)

    def _rm(self, path):
        if not path:
            return
        try:
            with open(os.devnull, "wb") as devnull:
                sp.check_call(["/bin/rm", "-rf", path], stderr=devnull)
        except (sp.CalledProcessError, OSError) as ex:
            self.logger.error("Can not remove %r: %s", path, ex)

    def _remove_unregistered_paths(self, dir_path, records):
        registered_paths = [r.path for r in records]
        for sub_path in os.listdir(dir_path):
            path = os.path.join(dir_path, sub_path)
            if sub_path != CacheableVCS.METADATA_FILENAME and path not in registered_paths:
                self.logger.info("Deleting unregistered path %r", path)
                self._rm(path)

    def _get_caches_to_delete(self, cacheable, all_records):
        known_types = [_.REP_TYPE for _ in self._cacheables]
        path_updates = {}
        paths_to_delete = []
        permanent = {}
        for rec in all_records:
            if rec.rep_type == cacheable.REP_TYPE:
                permanent[rec.path] = rec.is_permanent  # check only the last permanency flag
        for rec in all_records:
            if rec.rep_type not in known_types:
                paths_to_delete.append(rec.path)
            elif rec.rep_type == cacheable.REP_TYPE and not permanent[rec.path]:
                path_updates[rec.path] = rec.update_time or ""
        if len(path_updates) > cacheable.DIRS_LIMIT:
            caches = sorted(path_updates.keys(), key=lambda path: path_updates[path])
            paths_to_delete.extend(caches[:-cacheable.DIRS_LIMIT])
        return paths_to_delete

    def _clean(self, cacheable):
        """
        :type cacheable: CacheableVCS
        """
        cache_dir = cacheable.base_cache_dir
        if not os.path.exists(cache_dir):
            return

        with utils.FLock(cacheable.cache_metadata_path):
            records = CacheRecord.read_cache_records(cacheable.cache_metadata_path)
            self._remove_unregistered_paths(cache_dir, records)
            caches_to_delete = self._get_caches_to_delete(cacheable, records)

            for cache_to_delete in set(caches_to_delete):
                cache_records = filter(lambda rec: rec.path == cache_to_delete, records)
                if cache_records:
                    self.logger.info("Deleting cache %r", cache_records[0])
                self._rm(cache_to_delete)

            CacheRecord.write_cache_records(
                cacheable.cache_metadata_path,
                [rec for rec in records if rec.path not in caches_to_delete]
            )
