from __future__ import absolute_import

import abc
import time
import logging

import kazoo.client
import kazoo.exceptions

from . import utils
from . import config


class _SingletonMeta(utils.SingletonMeta):
    def __call__(cls, *args, **kwargs):
        enabled = config.Registry().common.zookeeper.enabled
        obj = utils.SingletonMeta.__call__(cls if enabled else DummyZookeeper, *args, **kwargs)
        obj.enabled = enabled
        return obj


class _AbstractBase(object):
    """ An abstract base class for Zookeeper interface access. """
    __metaclass__ = _SingletonMeta

    def __init__(self, logger=None):
        self.client = None
        self.enabled = None
        self.logger = (logger or logging.getLogger()).getChild(__name__)

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, *_):
        self.stop()

    @abc.abstractmethod
    def start(self):
        return self

    @abc.abstractmethod
    def stop(self):
        self.client = None
        return self

    @abc.abstractmethod
    def restart(self):
        pass

    @abc.abstractmethod
    def state(self):
        pass

    @abc.abstractmethod
    def lock(self, *args):
        pass

    @abc.abstractmethod
    def list(self, *args):
        pass


class Zookeeper(_AbstractBase):
    __metaclass__ = _SingletonMeta

    @staticmethod
    def hosts(hosts):
        return ",".join(utils.HostInfo.neighbours(config.Registry().this, hosts.split(";")))

    def __init__(self, logger=None):
        super(Zookeeper, self).__init__(logger)
        self.config = config.Registry().common.zookeeper
        self.client = None

    def start(self):
        if self.client:
            return self
        attempts = 10
        self.logger.info("Connecting to Zookeper at '%s', locks root '%s'", self.config.hosts, self.config.root)
        zk_hosts = self.hosts(self.config.hosts)
        for i in xrange(1, attempts):
            # noinspection PyBroadException
            try:
                self.client = kazoo.client.KazooClient(hosts=zk_hosts, randomize_hosts=False)
                self.client.add_listener(self._listener)
                self.client.start(timeout=30)
                return super(Zookeeper, self).start()
            except Exception:
                self.logger.exception("Zookeper start attempt {}/{} failed".format(i, attempts))
                time.sleep(2 * i)
        self.client.start()
        return super(Zookeeper, self).start()

    def stop(self):
        if not self.client:
            return self
        self.logger.info("Stopping Zookeper")
        # noinspection PyBroadException
        try:
            self.client.stop()
        except Exception:
            self.logger.exception("Error stopping Zookeeper client")
        return super(Zookeeper, self).stop()

    def restart(self):
        self.client.restart()

    def state(self):
        return self.client.state

    def lock(self, *args):
        lock_path = '/'.join((self.config.root, "locks") + args)
        self.client.ensure_path(lock_path)
        ret = self.client.Lock(lock_path, config.Registry().this.fqdn)
        return ret

    def list(self, subdir="jobs"):
        if not self.client:
            return
        lock_path = '/'.join((self.config.root, "locks", subdir))
        return {
            _: self.client.Lock('/'.join((lock_path, _))).contenders()
            for _ in self.client.get_children(lock_path)
        }

    def _listener(self, state):
        sid = " Session ID is 0x{:x}".format(self.client.client_id[0]) if state == "CONNECTED" else ""
        self.logger.info("Zookeeper state has changed to '%s'.%s", state, sid)


class DummyZookeeper(_AbstractBase):
    """ Fake Zookeeper class to test code without Zookeeper instance. """

    __metaclass__ = utils.SingletonMeta

    class Lock(object):
        def __init__(self, client):
            self.client = client

        def __enter__(self):
            return self

        def __exit__(self, exc_type, exc_val, exc_tb):
            pass

    def start(self):
        self.client = type(
            "FakeZookeeperClient",
            (object,),
            {
                "connected": lambda _: True,
                "client_id": property(lambda _: (1, 1))
            }
        )
        return super(DummyZookeeper, self).start()

    def stop(self):
        return super(DummyZookeeper, self).stop()

    def restart(self):
        pass

    def state(self):
        return "CONNECTED"

    def lock(self, *_):
        return self.Lock(self.client)

    def list(self, *_):
        return {}
